package ohd.hseb.hefs.mefp.sources.historical;

import java.awt.BasicStroke;
import java.awt.Color;
import java.awt.Font;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.List;

import nl.wldelft.util.timeseries.TimeSeriesArray;
import ohd.hseb.charter.ChartEngine;
import ohd.hseb.charter.ChartEngineException;
import ohd.hseb.charter.ChartTools;
import ohd.hseb.charter.ChartConstants;
import ohd.hseb.charter.datasource.DataSourceGenerator;
import ohd.hseb.charter.datasource.XYChartDataSource;
import ohd.hseb.charter.datasource.XYChartDataSourceException;
import ohd.hseb.charter.datasource.instances.DataSetXYChartDataSource;
import ohd.hseb.charter.datasource.instances.NumericalXYChartDataSource;
import ohd.hseb.charter.parameters.DataSourceDrawingParameters;
import ohd.hseb.charter.parameters.SeriesDrawingParameters;
import ohd.hseb.hefs.mefp.models.precipitation.MEFPPrecipitationModelTools;
import ohd.hseb.hefs.pe.tools.LocationAndDataTypeIdentifier;
import ohd.hseb.hefs.utils.dist.Distribution;
import ohd.hseb.hefs.utils.dist.DistributionType;
import ohd.hseb.hefs.utils.tools.NumberTools;
import ohd.hseb.util.data.DataSet;
import ohd.hseb.util.misc.HStopWatch;

import org.apache.logging.log4j.Logger;
import org.apache.logging.log4j.LogManager;
import org.jfree.chart.JFreeChart;
import org.jfree.chart.plot.CombinedDomainXYPlot;
import org.jfree.chart.plot.XYPlot;
import org.jfree.data.Range;

import com.google.common.collect.Lists;

/**
 * Supplies data for display in a {@link DistributionDiagnosticPanel}.
 * 
 * @author Hank.Herr
 */
public class DistributionDiagnosticDataSourceSupplier implements DataSourceGenerator
{
    private static final Logger LOG = LogManager.getLogger(DistributionDiagnosticDataSourceSupplier.class);

    private final static int FIT_SAMPLE_VAR = 1;
    private final static int FIT_SAMPLE_EMP = 2;

    private final Double _zeroThreshold;
    private DataSet _cdfDataSet;
    private final List<Distribution> _distributions = new ArrayList<>();
    private final List<DistributionType> _distributionTypes = new ArrayList<>();
    private final List<Color> _distributionColors = new ArrayList<>();
    private final List<Double> _maximumErrors = new ArrayList<>();
    private final List<Double> _meanSquareErrors = new ArrayList<>();
    private final List<Double> _correlations = new ArrayList<>();
    private final String _diagnosticName;

    /**
     * @param diagnosticName The name of the diagnostic, which is incorporated in the plot titles of {@link ChartEngine}
     *            created by this.
     * @param values The values to be displayed.
     * @param zeroThreshold A threshold defining what zero is. If null, no zero-conditioning is incorporated
     *            (appropriate for temperature).
     * @param distributionTypes Types of distributions to include in the analysis.
     */
    public DistributionDiagnosticDataSourceSupplier(final String diagnosticName,
                                                    final double[] values,
                                                    final Double zeroThreshold,
                                                    final DistributionType[] distributionTypes)
    {
        _zeroThreshold = zeroThreshold;
        _diagnosticName = diagnosticName;
        DataSet empiricalDataSet = new DataSet(values.length, 3, false);
        empiricalDataSet.setSampleSize(values.length);
        empiricalDataSet.setVariable(FIT_SAMPLE_VAR, values);
        if(_zeroThreshold != null)
        {
            empiricalDataSet = empiricalDataSet.extractSubset(FIT_SAMPLE_VAR,
                                                              DataSet.GREATER_THAN_OR_EQUAL_TO,
                                                              _zeroThreshold);
        }
        initializeDataSet(empiricalDataSet, distributionTypes);
    }

    /**
     * Pulls the data from the provided time series, extracts only those whose values exceeds the zero threshold
     * provided. It then computes the empirical CDF (Weibull plotting position), fits all appropriate distributions, and
     * computes the maximum and mean-squared error terms. All items are recorded.
     * 
     * @param timeSeries The time series from which to extract data.
     * @param zeroThreshold The treshold that defines what zero is.
     */
    public DistributionDiagnosticDataSourceSupplier(final String diagnosticName,
                                                    final TimeSeriesArray timeSeries,
                                                    final Double zeroThreshold,
                                                    final DistributionType[] distributionTypes)
    {
        LOG.debug("PROCESSING TIME SERIES " + timeSeries
            + " ------------------------------------------------------------------------- ");
        _diagnosticName = diagnosticName;
        _zeroThreshold = zeroThreshold;
        if(_zeroThreshold != null)
        {
            initializeDataSet(new DataSet(timeSeries, 3).extractSubset(FIT_SAMPLE_VAR,
                                                                       DataSet.GREATER_THAN_OR_EQUAL_TO,
                                                                       _zeroThreshold), distributionTypes);
        }
        else
        {
            initializeDataSet(new DataSet(timeSeries, 3), distributionTypes);
        }
    }

    /**
     * Initializes the {@link #_cdfDataSet} to include the empirical cdf and distribution computed variables.
     * 
     * @param distributionTypes
     */
    private void initializeDataSet(final DataSet empiricalDataSet, final DistributionType[] distributionTypes)
    {
        LOG.debug("Diagnostic data set has " + empiricalDataSet.getSampleSize() + " many points.");

        empiricalDataSet.createCDFUsingWeibullPlottingPosition(FIT_SAMPLE_VAR, FIT_SAMPLE_EMP);
        empiricalDataSet.setFitSampleVariable(FIT_SAMPLE_VAR);
        empiricalDataSet.setFitCDFVariable(FIT_SAMPLE_EMP);

        for(final DistributionType type: distributionTypes)
        {
            final HStopWatch timer = new HStopWatch();

            empiricalDataSet.sortBy(FIT_SAMPLE_EMP);

            try
            {
                final Distribution distribution = MEFPPrecipitationModelTools.fitDistribution(empiricalDataSet,
                                                                                              type,
                                                                                              _zeroThreshold);

                _distributions.add(distribution);
                _distributionTypes.add(type);
            }
            catch(final Throwable t)
            {
                LOG.warn("Unable to fit a distribution of type " + type + " (see stack trace dumped to stdout): "
                    + t.getMessage());
                t.printStackTrace();
            }
            LOG.debug("Fit distribution " + type + " in " + timer.getElapsedMillis() + " milliseconds.");
        }

        //Size the final _cdfDataSet
        _cdfDataSet = new DataSet(empiricalDataSet, 3 + _distributionTypes.size() + _distributionTypes.size());

        //Populate it
        int distIndex = 0;
        for(final Distribution distribution: _distributions)
        {
            //Populate the quantile column for the distribution.
            for(int sample = 0; sample < _cdfDataSet.getSampleSize(); sample++)
            {
                _cdfDataSet.setValue(sample, FIT_SAMPLE_EMP + 1 + distIndex, //CDF vars are specified after the empirical.  Why the +1?
                                     distribution.functionCDF(_cdfDataSet.getValue(sample, FIT_SAMPLE_VAR)));
                _cdfDataSet.setValue(sample, FIT_SAMPLE_EMP + 1 + distributionTypes.length + distIndex, //Inverse CDF vars are specified after the CDF vars
                                     distribution.functionInverseCDF(_cdfDataSet.getValue(sample, FIT_SAMPLE_EMP)));
            }

            //Compute and store the errors.
            _maximumErrors.add(_cdfDataSet.maximumError(FIT_SAMPLE_EMP, FIT_SAMPLE_EMP + 1 + distIndex));
            _meanSquareErrors.add(_cdfDataSet.meanSquaredError(FIT_SAMPLE_EMP, FIT_SAMPLE_EMP + 1 + distIndex));
            _correlations.add(_cdfDataSet.correlation(FIT_SAMPLE_VAR, FIT_SAMPLE_EMP + 1 + distributionTypes.length
                + distIndex));

            distIndex++;
        }

        _distributionColors.addAll(Arrays.asList(ChartTools.buildESPADPColorPalette(_distributions.size())));
    }

    public String getDiagnosticName()
    {
        return _diagnosticName;
    }

    public DataSet getCDFDataSet()
    {
        return _cdfDataSet;
    }

    public List<Distribution> getFittedDistributions()
    {
        return _distributions;
    }

    public List<DistributionType> getDistributionTypes()
    {
        return _distributionTypes;
    }

    public Distribution getFittedDistribution(final DistributionType type)
    {
        return DistributionType.retrieveDistribution(_distributions, type);
    }

    public double getMaximumError(final DistributionType type)
    {
        final int distIndex = _distributionTypes.indexOf(type);
        return _maximumErrors.get(distIndex);
    }

    public double getMeanSquaredError(final DistributionType type)
    {
        final int distIndex = _distributionTypes.indexOf(type);
        return _meanSquareErrors.get(distIndex);
    }

    public double getCorrelation(final DistributionType type)
    {
        final int distIndex = _distributionTypes.indexOf(type);
        return _correlations.get(distIndex);
    }

    public double getMaximumError(final int distIndex)
    {
        return _maximumErrors.get(distIndex);
    }

    public double getMeanSquaredError(final int distIndex)
    {
        return _meanSquareErrors.get(distIndex);
    }

    public double getCorrelation(final int distIndex)
    {
        return _correlations.get(distIndex);
    }

    private int[] retrieveIndicesOfDistributions(final DistributionType... distributionTypes)
    {
        final int[] distIndices = new int[distributionTypes.length];
        int index = 0;
        for(final DistributionType type: distributionTypes)
        {
            final int distIndex = _distributionTypes.indexOf(type);
            distIndices[index] = distIndex;
            index++;
        }
        return distIndices;
    }

    /**
     * @param indexOfDataSourceToCreate The data source index assigned to the returned data source.
     * @param dataSetIndex Index of the time series provided in the constructor's list that is the data for which a Q-Q
     *            plot data source is to be created.
     * @param distributionTypes The distributions for which P-P plots are to be displayed in the same chart subplot.
     * @return A {@link DataSetXYChartDataSource} to provide the data with a source index of 0. The source index may
     *         need to be changed depending on how it is plotted, but that must be done by the outside caller of this
     *         method.
     * @throws XYChartDataSourceException If the constructor for {@link DataSetXYChartDataSource} fails for whatever
     *             reason.
     */
    public DataSetXYChartDataSource generatePPDataSource(final int indexOfDataSourceToCreate,
                                                         final DistributionType... distributionTypes) throws XYChartDataSourceException
    {
        //Prepare the variable arrays.
        final int[] fittedXVars = new int[distributionTypes.length];
        final int[] empiricalYVars = retrieveIndicesOfDistributions(distributionTypes);
        for(int i = 0; i < distributionTypes.length; i++)
        {
            fittedXVars[i] = FIT_SAMPLE_EMP;
            empiricalYVars[i] = empiricalYVars[i] + FIT_SAMPLE_EMP + 1; //Shifting the list index to match the data set variable index.
        }

        return new DataSetXYChartDataSource(this, indexOfDataSourceToCreate, _cdfDataSet, empiricalYVars, fittedXVars);
    }

    /**
     * @param indexOfDataSourceToCreate The data source index assigned to the returned data source.
     * @param dataSetIndex Index of the time series provided in the constructor's list that is the data for which a Q-Q
     *            plot data source is to be created.
     * @param distributionTypes The distributions for which Q-Q plots are to be displayed in the same chart subplot.
     * @return A {@link DataSetXYChartDataSource} to provide the data with a source index of 0. The source index may
     *         need to be changed depending on how it is plotted, but that must be done by the outside caller of this
     *         method.
     * @throws XYChartDataSourceException If the constructor for {@link DataSetXYChartDataSource} fails for whatever
     *             reason.
     */
    public DataSetXYChartDataSource generateQQDataSource(final int indexOfDataSourceToCreate,
                                                         final DistributionType... distributionTypes) throws XYChartDataSourceException
    {
        //Prepare the variable arrays.
        final int[] fittedXVars = new int[distributionTypes.length];
        final int[] empiricalYVars = retrieveIndicesOfDistributions(distributionTypes);
        for(int i = 0; i < distributionTypes.length; i++)
        {
            fittedXVars[i] = FIT_SAMPLE_VAR;
            empiricalYVars[i] = empiricalYVars[i] + FIT_SAMPLE_EMP + 1 + _distributionTypes.size(); //Shifting the list index to match the data set variable index.
        }

        return new DataSetXYChartDataSource(this, indexOfDataSourceToCreate, _cdfDataSet, empiricalYVars, fittedXVars);
    }

    /**
     * To build the data source, this first must determine the range of values for which to plot the fitted
     * distribution. To do that, it first builds a chart displaying the empirical distribution, only, and pulls the
     * domain axis range-of-values from that. It uses that to dictate the values for which to computed fitted
     * distribution values, and then returns it within the axisLimitsToUse argument.
     * 
     * @param indexOfDataSourceToCreate The data source index assigned to the returned data source.
     * @param dataSetIndex Index of the time series provided in the constructor's list that is the data for which a Q-Q
     *            plot data source is to be created.
     * @param axisLimitsToUse Object used to store the range of values for which distribution points were computed. It
     *            should be used as the default axis limts for the domain axis of the CDF plot.
     * @param distributionTypes The distributions for which CDF plots are to be displayed in the same chart as the
     *            empirical.
     * @return A {@link DataSetXYChartDataSource} to provide the data with a source index of 0. The source index may
     *         need to be changed depending on how it is plotted, but that must be done by the outside caller of this
     *         method.
     * @throws XYChartDataSourceException If the constructor for {@link DataSetXYChartDataSource} fails for whatever
     *             reason.
     */
    public NumericalXYChartDataSource generateCDFDataSource(final int indexOfDataSourceToCreate,
                                                            final double[] axisLimitsToUse,
                                                            final DistributionType... distributionTypes) throws XYChartDataSourceException
    {
        //Prepare the variable arrays.
        final List<double[]> xValues = new ArrayList<>();
        final List<double[]> yProbs = new ArrayList<>();

        //Empirical Points
        xValues.add(_cdfDataSet.getVariable(FIT_SAMPLE_VAR));
        yProbs.add(_cdfDataSet.getVariable(FIT_SAMPLE_EMP));

        //Determine the auto-bounds that will be calculated.
        double largestValue = _cdfDataSet.getLargest(FIT_SAMPLE_VAR) * 1.1d;
        double smallestValue = 0.0d;
        try
        {
            final ChartEngine engine = new ChartEngine(Lists.newArrayList((XYChartDataSource)(new NumericalXYChartDataSource(this,
                                                                                                                             indexOfDataSourceToCreate,
                                                                                                                             xValues,
                                                                                                                             yProbs))));
            final Range range = engine.buildChart().getXYPlot().getDomainAxis().getRange();
            largestValue = Math.ceil(range.getUpperBound());
            smallestValue = Math.floor(range.getLowerBound());
            axisLimitsToUse[0] = smallestValue;
            axisLimitsToUse[1] = largestValue;
        }
        catch(final ChartEngineException e)
        {
            throw new XYChartDataSourceException("Unable to construct CDF chart data source.", e);
        }

        //Add values to display for each distribution.
        final int[] distIndices = retrieveIndicesOfDistributions(distributionTypes);
        for(int i = 0; i < distributionTypes.length; i++)
        {
            final List<Double> fittedXValues = new ArrayList<>();
            final List<Double> fittedYProbs = new ArrayList<>();
            for(double workingValue = smallestValue; workingValue <= largestValue; workingValue += 0.01d)
            {
                fittedXValues.add(workingValue);
                fittedYProbs.add(_distributions.get(distIndices[i]).functionCDF(workingValue));
            }
            xValues.add(NumberTools.convertNumbersToDoublesArray(fittedXValues));
            yProbs.add(NumberTools.convertNumbersToDoublesArray(fittedYProbs));
        }

        return new NumericalXYChartDataSource(this, indexOfDataSourceToCreate, xValues, yProbs);
    }

    /**
     * @param indexOfDataSourceToCreate The data source index assigned to the returned data source.
     * @return A simple {@link NumericalXYChartDataSource} that specifies a line from 0,0 to 1,1.
     * @throws XYChartDataSourceException
     */
    public static NumericalXYChartDataSource buildSlopeOneLineDataSource(final int indexOfDataSourceToCreate,
                                                                         final double minValue,
                                                                         final double maxValue) throws XYChartDataSourceException
    {
        final double[] xValues = new double[]{minValue, maxValue};
        final double[] yValues = new double[]{minValue, maxValue};
        final List<double[]> xValuesList = new ArrayList<>();
        xValuesList.add(xValues);
        final List<double[]> yValuesList = new ArrayList<>();
        yValuesList.add(yValues);

        return new NumericalXYChartDataSource(new DataSourceGenerator()
        {
            @Override
            public StringBuffer generateOutputString(final XYChartDataSource calculatedSource,
                                                     final DataSourceDrawingParameters drawingParms)
            {
                return null;
            }

            @Override
            public String getDescriptiveName()
            {
                return "0-1 Line";
            }
        }, indexOfDataSourceToCreate, xValuesList, yValuesList);
    }

    @Override
    public StringBuffer generateOutputString(final XYChartDataSource calculatedSource,
                                             final DataSourceDrawingParameters drawingParms)
    {
        return null;
    }

    @Override
    public String getDescriptiveName()
    {
        return "Quantile-Quantile Source";
    }

    /**
     * Calls {@link #generatePPChartEngine(DistributionType[])} converting the provided collection to an array.
     */
    public ChartEngine generatePPChartEngine(final Collection<DistributionType> displayedTypes) throws Exception
    {
        return generatePPChartEngine(displayedTypes.toArray(new DistributionType[displayedTypes.size()]));
    }

    /**
     * @return A {@link ChartEngine} for displaying the P-P plot for the specified distributions and empirical
     *         distribution.
     * @throws Exception If {@link #generatePPDataSource(int, int, DistributionType...)} fails or if the
     *             {@link ChartEngine} fails to be constructed.
     */
    public ChartEngine generatePPChartEngine(final DistributionType[] displayedTypes) throws Exception
    {
        final NumericalXYChartDataSource source = generatePPDataSource(0, displayedTypes);
        final ChartEngine engine = new ChartEngine(Lists.newArrayList((XYChartDataSource)source));

        //Q-Q points
        engine.getChartParameters().getDataSourceParameters(0).setPlotterName("LineAndScatter");
        for(int seriesIndex = 0; seriesIndex < displayedTypes.length; seriesIndex++)
        {
            final SeriesDrawingParameters seriesParms = engine.getChartParameters()
                                                              .getDataSourceParameters(0)
                                                              .getSeriesDrawingParametersForSeriesIndex(seriesIndex);
            seriesParms.setLineWidth(0.0f);
            seriesParms.setShapeName(ChartConstants.SHAPE_NAMES[seriesIndex]);
            seriesParms.setLineColor(_distributionColors.get(_distributionTypes.indexOf(displayedTypes[seriesIndex])));
            seriesParms.setShapeFilled(false);
            seriesParms.setShapeSize(0.5d);
            seriesParms.setShowInLegend(true);
            seriesParms.setNameInLegend(displayedTypes[seriesIndex].toString());
        }

        //Legend
        engine.getChartParameters().getLegend().setVisible(true);
        engine.getChartParameters().getLegend().setPositionString("BOTTOM");

        //Fixed Axis Limits
        engine.getChartParameters().getDomainAxis().getNumericAxisParameters().setLowerBound(0.0d);
        engine.getChartParameters().getDomainAxis().getNumericAxisParameters().setUpperBound(1.0d);
        engine.getChartParameters().getDomainAxis().getNumericAxisParameters().setAutoRange(false);
        engine.getChartParameters()
              .getSubPlot(0)
              .getAxis(ChartConstants.LEFT_YAXIS_INDEX)
              .getNumericAxisParameters()
              .setLowerBound(0.0d);
        engine.getChartParameters()
              .getSubPlot(0)
              .getAxis(ChartConstants.LEFT_YAXIS_INDEX)
              .getNumericAxisParameters()
              .setUpperBound(1.0d);
        engine.getChartParameters()
              .getSubPlot(0)
              .getAxis(ChartConstants.LEFT_YAXIS_INDEX)
              .getNumericAxisParameters()
              .setAutoRange(false);

        //Labels
        if(_zeroThreshold != null)
        {
            engine.getChartParameters()
                  .getPlotTitle()
                  .setText(_diagnosticName + " P-P Plot for Values that Exceed the Zero-Threshold, " + _zeroThreshold);
        }
        else
        {
            engine.getChartParameters().getPlotTitle().setText(_diagnosticName + " P-P Plot for All Values");
        }
        engine.getChartParameters().getPlotTitle().setFont(new Font("Serif", Font.BOLD, 14));
        engine.getChartParameters().getDomainAxis().getLabel().setText("Theoretical Probability"); //Precip is always MM units in MEFP
        engine.getChartParameters().getSubPlot(0).getLeftAxis().getLabel().setText("Empirical Probability");

        return engine;
    }

    /**
     * Build and reformat a P-P {@link JFreeChart} as needed given the chart engine and identifier.
     */
    public JFreeChart buildPPChartFromEngine(final LocationAndDataTypeIdentifier identifier,
                                             final ChartEngine chartEngine) throws Exception
    {
        final JFreeChart builtChart = chartEngine.buildChart();
        ChartTools.addDiagonalLine(builtChart, 0, 100, new BasicStroke(1), Color.BLUE);
        return builtChart;
    }

    /**
     * Calls {@link #generateQQChartEngine(DistributionType[])} converting the provided collection to an array.
     */
    public ChartEngine generateQQChartEngine(final Collection<DistributionType> displayedTypes) throws Exception
    {
        return generateQQChartEngine(displayedTypes.toArray(new DistributionType[displayedTypes.size()]));
    }

    /**
     * @return A {@link ChartEngine} for displaying the Q-Q plot for the specified distributions and empirical
     *         distribution.
     * @throws Exception If {@link #generateQQDataSource(int, int, DistributionType...)} fails or if the
     *             {@link ChartEngine} fails to be constructed.
     */
    public ChartEngine generateQQChartEngine(final DistributionType[] displayedTypes) throws Exception
    {
        final NumericalXYChartDataSource source = generateQQDataSource(0, displayedTypes);
        final ChartEngine engine = new ChartEngine(Lists.newArrayList((XYChartDataSource)source));

        //Q-Q points
        engine.getChartParameters().getDataSourceParameters(0).setPlotterName("LineAndScatter");
        for(int seriesIndex = 0; seriesIndex < displayedTypes.length; seriesIndex++)
        {
            final SeriesDrawingParameters seriesParms = engine.getChartParameters()
                                                              .getDataSourceParameters(0)
                                                              .getSeriesDrawingParametersForSeriesIndex(seriesIndex);
            seriesParms.setLineWidth(0.0f);
            seriesParms.setShapeName(ChartConstants.SHAPE_NAMES[seriesIndex]);
            seriesParms.setLineColor(_distributionColors.get(_distributionTypes.indexOf(displayedTypes[seriesIndex])));
            seriesParms.setShapeFilled(false);
            seriesParms.setShapeSize(0.5d);
            seriesParms.setShowInLegend(true);
            seriesParms.setNameInLegend(displayedTypes[seriesIndex].toString());
        }

        //Legend
        engine.getChartParameters().getLegend().setVisible(true);
        engine.getChartParameters().getLegend().setPositionString("BOTTOM");

        //Labels
        if(_zeroThreshold != null)
        {
            engine.getChartParameters()
                  .getPlotTitle()
                  .setText(_diagnosticName + " Q-Q Plot for Values that Exceed the Zero-Threshold, " + _zeroThreshold);
        }
        else
        {
            engine.getChartParameters().getPlotTitle().setText(_diagnosticName + " Q-Q Plot for All Values");
        }
        engine.getChartParameters().getPlotTitle().setFont(new Font("Serif", Font.BOLD, 14));

        String units = "MM";
        if(_zeroThreshold == null)
        {
            units = "DegC";
        }
        engine.getChartParameters().getDomainAxis().getLabel().setText("Theoretical Quantile (" + units + ")"); //Precip is always MM units in MEFP
        engine.getChartParameters().getSubPlot(0).getLeftAxis().getLabel().setText("Data Quantile (" + units + ")");

        return engine;
    }

    /**
     * Build and reformat a Q-Q {@link JFreeChart} as needed given the chart engine and identifier.
     */
    public JFreeChart buildQQChartFromEngine(final LocationAndDataTypeIdentifier identifier,
                                             final ChartEngine chartEngine) throws Exception
    {
        final JFreeChart builtChart = chartEngine.buildChart();

        //I used to square this chart, but since a line with slope-1 is included, and is guaranteed to cover all displayed points, artificial
        //squaring should not be needed.  Note that artificial squaring can mess up the zoom state of the chart upon first rendering.
        ChartTools.squareAxes(builtChart, "left");

        final CombinedDomainXYPlot plot = (CombinedDomainXYPlot)builtChart.getPlot();

        //Precip: force axes to start at 0.
        if(identifier.isPrecipitationDataType())
        {
            ((XYPlot)plot.getSubplots().get(0)).getRangeAxis(0).setLowerBound(0);
            plot.getDomainAxis().setLowerBound(0);
        }

        ChartTools.addDiagonalLine(builtChart, 0, 10000000, new BasicStroke(1), Color.BLUE);

        return builtChart;
    }

    /**
     * Calls {@link #generateCDFChartEngine(DistributionType[])} converting the provided collection to an array.
     */
    public ChartEngine generateCDFChartEngine(final Collection<DistributionType> displayedTypes) throws Exception
    {
        return generateCDFChartEngine(displayedTypes.toArray(new DistributionType[displayedTypes.size()]));
    }

    /**
     * @return A {@link ChartEngine} that displays the cumulative distribution functions for the empirical and requested
     *         fitted distributions.
     * @throws Exception If the call to {@link #generateCDFDataSource(int, int, double[], DistributionType...)} fails or
     *             the construction of the {@link ChartEngine} fails.
     */
    public ChartEngine generateCDFChartEngine(final DistributionType[] displayedTypes) throws Exception
    {
        final double[] axisLimitsToUse = new double[2];
        final NumericalXYChartDataSource source = generateCDFDataSource(0, axisLimitsToUse, displayedTypes);

        final ChartEngine engine = new ChartEngine(Lists.newArrayList((XYChartDataSource)source));

        //CDF points
        engine.getChartParameters().getDataSourceParameters(0).setPlotterName("LineAndScatter");
        for(int seriesIndex = 0; seriesIndex < displayedTypes.length + 1; seriesIndex++)
        {
            //0 is empirical.  1+ is fitted.

            final SeriesDrawingParameters seriesParms = engine.getChartParameters()
                                                              .getDataSourceParameters(0)
                                                              .getSeriesDrawingParametersForSeriesIndex(seriesIndex);
            seriesParms.setLineWidth(1.0f);
            seriesParms.setShapeSize(0.0d);
            seriesParms.setShapeName(ChartConstants.SHAPE_NAMES[seriesIndex]);
            seriesParms.setShapeFilled(false);
            seriesParms.setShowInLegend(true);

            if(seriesIndex == 0) //Empirical is shapes only, no line.
            {
                seriesParms.setLineWidth(0.0f);
                seriesParms.setShapeSize(0.5d);
                seriesParms.setNameInLegend("Empirical");
                seriesParms.setLineColor(Color.black);
            }
            else
            {
                seriesParms.setNameInLegend(displayedTypes[seriesIndex - 1].toString());
                seriesParms.setLineColor(_distributionColors.get(_distributionTypes.indexOf(displayedTypes[seriesIndex - 1])));
            }
        }

        //Legend
        engine.getChartParameters().getLegend().setVisible(true);
        engine.getChartParameters().getLegend().setPositionString("BOTTOM");

        //Fixed Axis Limits
        engine.getChartParameters().getDomainAxis().getNumericAxisParameters().setLowerBound(axisLimitsToUse[0]);
        engine.getChartParameters().getDomainAxis().getNumericAxisParameters().setUpperBound(axisLimitsToUse[1]);
        engine.getChartParameters().getDomainAxis().getNumericAxisParameters().setAutoRange(false);
        engine.getChartParameters()
              .getSubPlot(0)
              .getAxis(ChartConstants.LEFT_YAXIS_INDEX)
              .getNumericAxisParameters()
              .setLowerBound(0.0d);
        engine.getChartParameters()
              .getSubPlot(0)
              .getAxis(ChartConstants.LEFT_YAXIS_INDEX)
              .getNumericAxisParameters()
              .setUpperBound(1.0d);
        engine.getChartParameters()
              .getSubPlot(0)
              .getAxis(ChartConstants.LEFT_YAXIS_INDEX)
              .getNumericAxisParameters()
              .setAutoRange(false);

        //Labels
        if(_zeroThreshold != null)
        {
            engine.getChartParameters()
                  .getPlotTitle()
                  .setText(_diagnosticName + " CDF Plot for Values that Exceed the Zero-Threshold, " + _zeroThreshold);
        }
        else
        {
            engine.getChartParameters().getPlotTitle().setText(_diagnosticName + " CDF Plot for All Values");
        }
        engine.getChartParameters().getPlotTitle().setFont(new Font("Serif", Font.BOLD, 14));

        if(_zeroThreshold != null)
        {
            engine.getChartParameters().getDomainAxis().getLabel().setText("Value (MM)");
        }
        else
        {
            engine.getChartParameters().getDomainAxis().getLabel().setText("Value (DegC)");
        }
        engine.getChartParameters().getSubPlot(0).getLeftAxis().getLabel().setText("Probability");

        return engine;
    }

    /**
     * Build and reformat a CDF {@link JFreeChart} as needed given the chart engine and identifier.
     */
    public JFreeChart buildCDFChartFromEngine(final LocationAndDataTypeIdentifier identifier,
                                              final ChartEngine chartEngine) throws Exception
    {
        return chartEngine.buildChart();
    }
}
