package ohd.hseb.hefs.mefp.sources.historical;

import java.awt.BorderLayout;
import java.awt.Component;
import java.awt.Dimension;
import java.awt.FlowLayout;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.Insets;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.LinkedHashMap;
import java.util.List;

import javax.swing.JButton;
import javax.swing.JComboBox;
import javax.swing.JFrame;
import javax.swing.JLabel;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JRootPane;
import javax.swing.JScrollPane;
import javax.swing.JSplitPane;
import javax.swing.JTable;
import javax.swing.ScrollPaneConstants;
import javax.swing.SwingUtilities;
import javax.swing.event.TableModelEvent;
import javax.swing.event.TableModelListener;

import org.jfree.chart.JFreeChart;

import com.google.common.collect.Lists;

import nl.wldelft.util.timeseries.SimpleEquidistantTimeStep;
import nl.wldelft.util.timeseries.TimeSeriesArray;
import nl.wldelft.util.timeseries.TimeSeriesArrays;
import ohd.hseb.charter.ChartEngine;
import ohd.hseb.charter.panel.ChartEngineChartAndTablePanel;
import ohd.hseb.charter.panel.CombinedDomainChartNavigationPanel;
import ohd.hseb.hefs.mefp.models.precipitation.MEFPPrecipitationModelTools;
import ohd.hseb.hefs.pe.core.ParameterEstimatorDiagnosticPanel;
import ohd.hseb.hefs.pe.tools.LocationAndDataTypeIdentifier;
import ohd.hseb.hefs.utils.dist.DistributionType;
import ohd.hseb.hefs.utils.gui.components.ComponentPanelGlassPane;
import ohd.hseb.hefs.utils.gui.jtable.GenericTable;
import ohd.hseb.hefs.utils.gui.tools.HSwingFactory;
import ohd.hseb.hefs.utils.gui.tools.SelfListeningButton;
import ohd.hseb.hefs.utils.gui.tools.SwingTools;
import ohd.hseb.hefs.utils.plugins.TimeStepGenericParameterPanel;
import ohd.hseb.hefs.utils.tools.MapTools;
import ohd.hseb.hefs.utils.tsarrays.TimeSeriesArraysTools;
import ohd.hseb.hefs.utils.tsarrays.agg.AggregationTools;

/**
 * Provides options for displaying Q-Q plots or CDF (prob-value) plots for historical data read in via MEFPPE.
 * 
 * @author Hank.Herr
 */
@SuppressWarnings("serial")
public class DistributionDiagnosticPanel extends ParameterEstimatorDiagnosticPanel
{
    private final static String NO_AGGREGATION = "none";

    //TODO Move these constants to appropriate MEFP general location
    public static DistributionType[] AVAILABLE_DISTRIBUTION_TYPES_TEMPERATURE = new DistributionType[]{DistributionType.NORMAL};

    private final LocationAndDataTypeIdentifier _identifier;

    //TODO Can the time series to display also be stored in the values objects???
    private final List<TimeSeriesArray> _timeSeriesToDisplay;
    private final LinkedHashMap<String, double[]> _valuesToDisplay;
    private final Double _precipitationZeroThreshold;

    /**
     * Populated during construction based on the provided time series, identifier, and precip threshold (if
     * appropriate).
     */
    private final List<DistributionDiagnosticDataSourceSupplier> _dataSuppliers = new ArrayList<>();

    private DistributionDiagnosticTableModel _model;
    private GenericTable<DistributionType> _table;
    private JScrollPane _tableScrollPane;
    private JComboBox _diagChoiceBox;
    private JSplitPane _splitPane;
    private ChartEngineChartAndTablePanel _chartPanel = null;
    private CombinedDomainChartNavigationPanel _navPanel;

    /**
     * Records the current aggregation time step as text.
     */
    private String _currentAggregation;

    /**
     * Displays the current aggregation time step.
     */
    private JLabel _currentAggregationLabel;

    /**
     * Click this button to change the aggregation.
     */
    private JButton _specifyAggregationButton;

    /**
     * @param identifier
     * @param precipitationZeroThreshold Zero threshold or null for temperature data.
     * @param valuesNames Names to use for the different sets of numbers for which to display the diagnostics. For
     *            example, "Observed".
     * @param valuesToDisplay The values associated with the names.
     */
    public DistributionDiagnosticPanel(final LocationAndDataTypeIdentifier identifier,
                                       final Double precipitationZeroThreshold,
                                       final Collection<String> valuesNames,
                                       final Collection<double[]> valuesToDisplay)
    {
        _identifier = identifier;
        _timeSeriesToDisplay = null;
        _valuesToDisplay = MapTools.createLinkedHashMap(valuesNames, valuesToDisplay);
        _precipitationZeroThreshold = precipitationZeroThreshold;
        _currentAggregation = null;
        try
        {
            updateDataSuppliers();
        }
        catch(final Exception e)
        {
            e.printStackTrace();
            throw new IllegalStateException("This should never happen: the updateDataSuppliers() method failed during construction of the distribution diag panel!");
        }
        createDisplay();
    }

    /**
     * @param identifier {@link LocationAndDataTypeIdentifier} corresponding to the displayed location.
     * @param timeSeriesToDisplay The time series from which diagnostics should be computed.
     * @param precipitationZeroThreshold The threshold. This can be null if the data type is temperature.
     */
    public DistributionDiagnosticPanel(final LocationAndDataTypeIdentifier identifier,
                                       final Collection<TimeSeriesArray> timeSeriesToDisplay,
                                       final Double precipitationZeroThreshold)
    {
        _identifier = identifier;
        _timeSeriesToDisplay = new ArrayList<>(timeSeriesToDisplay);
        _valuesToDisplay = null;
        _precipitationZeroThreshold = precipitationZeroThreshold;
        _currentAggregation = NO_AGGREGATION;
        try
        {
            updateDataSuppliers();
        }
        catch(final Exception e)
        {
            e.printStackTrace();
            throw new IllegalStateException("This should never happen: the updateDataSuppliers() method failed during construction of the distribution diag panel!");
        }
        createDisplay();
    }

    /**
     * Sets which distributions are currently checked. Calls
     * {@link DistributionDiagnosticTableModel#setCheckedRows(Iterable)}.
     * 
     * @param types
     */
    public void setCheckedDistributions(final DistributionType... types)
    {
        _model.setCheckedRows(Arrays.asList(types));
    }

    private void createDisplay()
    {
        //Define the aggregation components.
        _currentAggregationLabel = new JLabel("Aggregation: " + _currentAggregation);
        _specifyAggregationButton = new SelfListeningButton("Select Aggregation")
        {
            @Override
            public void actionPerformed(final ActionEvent e)
            {
                String initialValue = _currentAggregation;
                if(initialValue.equals(NO_AGGREGATION))
                {
                    initialValue = ((SimpleEquidistantTimeStep)_timeSeriesToDisplay.get(0).getHeader().getTimeStep()).toString();
                }
                final TimeStepGenericParameterPanel timeStepPanel = new TimeStepGenericParameterPanel(initialValue);
                final JPanel panel = new JPanel(new BorderLayout());
                panel.add(new JLabel("Click Cancel to select no aggregation."), BorderLayout.NORTH);
                panel.add(timeStepPanel.getEditingDisplayPanel(), BorderLayout.SOUTH);
                final int value = JOptionPane.showOptionDialog(SwingTools.getGlobalDialogParent(DistributionDiagnosticPanel.this),
                                                               panel,
                                                               "Select Aggregation to Perform",
                                                               JOptionPane.YES_NO_CANCEL_OPTION,
                                                               JOptionPane.QUESTION_MESSAGE,
                                                               null,
                                                               new String[]{"Use Aggregation", "No Aggregation",
                                                                   "Cancel"},
                                                               "default");
                if(value == JOptionPane.CANCEL_OPTION)
                {
                    return;
                }

                final String previousAgg = _currentAggregation;
                _currentAggregation = timeStepPanel.getCurrentValueOfManagedParameter();
                if(value == JOptionPane.NO_OPTION)
                {
                    _currentAggregation = NO_AGGREGATION;
                }
                try
                {
                    updateDataSuppliers();
                    updateDisplayedDiagnostic();
                    _currentAggregationLabel.setText("Aggregation: " + _currentAggregation);
                }
                catch(final Exception exc)
                {
                    exc.printStackTrace();
                    JOptionPane.showMessageDialog(SwingTools.getGlobalDialogParent(DistributionDiagnosticPanel.this),
                                                  "Unable to aggregated time series:\n" + exc.getMessage(),
                                                  "Error Aggregating Data!",
                                                  JOptionPane.ERROR_MESSAGE);
                    _currentAggregation = previousAgg;
                }
            }
        };

        //Define the diagnostic choices.
        final String[] diagChoices = new String[3 * _dataSuppliers.size()];
        int index = 0;
        for(final DistributionDiagnosticDataSourceSupplier supplier: _dataSuppliers)
        {
            //Index is increased AFTER its value is used.
            diagChoices[index++] = supplier.getDiagnosticName() + " - P-P Plot";
            diagChoices[index++] = supplier.getDiagnosticName() + " - Q-Q Plot";
            diagChoices[index++] = supplier.getDiagnosticName() + " - CDF Plot";
        }
        _diagChoiceBox = new JComboBox<String>(diagChoices);
        _diagChoiceBox.setSelectedIndex(0);
        _diagChoiceBox.addActionListener(new ActionListener()
        {
            @Override
            public void actionPerformed(final ActionEvent e)
            {
                //Set the supplier for the model appropriately, which triggers the table model listener below.
                //XXX This math should perhaps be put in a method somewhere, since it matches math performed in the updateDisplayedDiagnostic method below.
                final int diagSelectedIndex = _diagChoiceBox.getSelectedIndex();
                final int supplierIndex = diagSelectedIndex / 3;
                _model.setSupplier(_dataSuppliers.get(supplierIndex));
            }
        });

        //Define the table.
        _model = new DistributionDiagnosticTableModel(_dataSuppliers.get(0));
        _table = new GenericTable<>(_model);
        _tableScrollPane = new JScrollPane(_table);
        _model.addTableModelListener(new TableModelListener()
        {
            @Override
            public void tableChanged(final TableModelEvent e)
            {
                updateDisplayedDiagnostic();
            }
        });
        _tableScrollPane.setHorizontalScrollBarPolicy(ScrollPaneConstants.HORIZONTAL_SCROLLBAR_ALWAYS);
        _tableScrollPane.setVerticalScrollBarPolicy(ScrollPaneConstants.VERTICAL_SCROLLBAR_ALWAYS);
        _table.setAutoResizeMode(JTable.AUTO_RESIZE_OFF);

        //Put it all together.
        final JPanel rightPanel = new JPanel(new GridBagLayout());
        final GridBagConstraints cons = new GridBagConstraints();
        cons.insets = new Insets(2, 4, 2, 4);
        cons.gridy = 0;
        cons.weighty = 0;
        cons.weightx = 1; //XXX Must be positive to ensure that it fills in the mainPanel width with the table.
        cons.anchor = GridBagConstraints.CENTER;
        cons.fill = GridBagConstraints.NONE;
        rightPanel.add(new JLabel(_identifier.buildStringToDisplayInTree()), cons);
        if(_timeSeriesToDisplay != null)
        {
            cons.anchor = GridBagConstraints.NORTHWEST;
            cons.fill = GridBagConstraints.BOTH;
            cons.gridy++;
            rightPanel.add(_currentAggregationLabel, cons);
            cons.anchor = GridBagConstraints.CENTER;
            cons.fill = GridBagConstraints.NONE;
            cons.gridy++;
            rightPanel.add(_specifyAggregationButton, cons);
        }
        cons.anchor = GridBagConstraints.NORTHWEST;
        cons.fill = GridBagConstraints.BOTH;
        cons.gridy++;
        rightPanel.add(new JLabel("Select Diagnostic:"), cons);
        cons.gridy++;
        rightPanel.add(_diagChoiceBox, cons);
        cons.gridy++;
        rightPanel.add(new JLabel("Available Distributions:"), cons);
        cons.gridy++;
        cons.weighty = 1;
        rightPanel.add(_tableScrollPane, cons);

        //Setup the split pane.
        _splitPane = HSwingFactory.createJSPlitPane(JSplitPane.HORIZONTAL_SPLIT, new JPanel(), rightPanel, false);
        setLayout(new BorderLayout());
        add(_splitPane, BorderLayout.CENTER);
        rightPanel.setMinimumSize(new Dimension(200, 200));
        _splitPane.setResizeWeight(1);

        //Initialize the diagnostics display.
        SwingUtilities.invokeLater(new Runnable()
        {
            @Override
            public void run()
            {
                updateDisplayedDiagnostic();
            }
        });
    }

    /**
     * Updates the {@link #_dataSuppliers} based on the current aggregation setting.
     * 
     * @throws Exception If the aggregation fails for any reason.
     */
    private void updateDataSuppliers() throws Exception
    {
        _dataSuppliers.clear();

        //Time series based ....
        if(_timeSeriesToDisplay != null)
        {
            for(TimeSeriesArray ts: _timeSeriesToDisplay)
            {
                if(_identifier.isPrecipitationDataType())
                {
                    if(!_currentAggregation.equals(NO_AGGREGATION))
                    {
                        ts = AggregationTools.aggregateToSum(ts,
                                                             ts.getStartTime(),
                                                             ts.getEndTime(),
                                                             _currentAggregation,
                                                             "asTimeStep",
                                                             "ending",
                                                             false,
                                                             false);
                    }
                    _dataSuppliers.add(new DistributionDiagnosticDataSourceSupplier(ts.getHeader().getParameterId(),
                                                                                    ts,
                                                                                    _precipitationZeroThreshold,
                                                                                    MEFPPrecipitationModelTools.getSelectableDistributionTypes()));
                }
                else
                {
                    if(!_currentAggregation.equals(NO_AGGREGATION))
                    {
                        ts = AggregationTools.aggregateToMean(ts,
                                                              ts.getStartTime(),
                                                              ts.getEndTime(),
                                                              _currentAggregation,
                                                              "asTimeStep",
                                                              "ending",
                                                              false,
                                                              false);
                    }
                    _dataSuppliers.add(new DistributionDiagnosticDataSourceSupplier(ts.getHeader().getParameterId(),
                                                                                    ts,
                                                                                    null,
                                                                                    AVAILABLE_DISTRIBUTION_TYPES_TEMPERATURE));
                }
            }
        }
        //Based on valuesToDisplay...
        else
        {
            for(final String name: _valuesToDisplay.keySet())
            {
                if(_identifier.isPrecipitationDataType())
                {
                    _dataSuppliers.add(new DistributionDiagnosticDataSourceSupplier(name,
                                                                                    _valuesToDisplay.get(name),
                                                                                    _precipitationZeroThreshold,
                                                                                    MEFPPrecipitationModelTools.getSelectableDistributionTypes()));
                }
                else
                {
                    _dataSuppliers.add(new DistributionDiagnosticDataSourceSupplier(name,
                                                                                    _valuesToDisplay.get(name),
                                                                                    null,
                                                                                    AVAILABLE_DISTRIBUTION_TYPES_TEMPERATURE));
                }
            }
        }
    }

    /**
     * Updates the displayed diagnostic to match that currently implied by user selections in the table.
     */
    private void updateDisplayedDiagnostic()
    {
        if(_dataSuppliers.isEmpty())
        {
            _splitPane.setTopComponent(HSwingFactory.createErrorMessagePane("Error Building Diagnostic",
                                                                            "Time series could not be computed.  Check aggregation!"));
            return;
        }

        final int diagSelectedIndex = _diagChoiceBox.getSelectedIndex();
        final int supplierIndex = diagSelectedIndex / 3;
        final int typeIndex = diagSelectedIndex - (3 * supplierIndex);

        try
        {
            //Get the chart engine.
            ChartEngine chartEngine = null;
            JFreeChart builtChart = null;
            if(typeIndex == 0) //P-P
            {
                chartEngine = _dataSuppliers.get(supplierIndex).generatePPChartEngine(_model.getCheckedRows());
                builtChart = _dataSuppliers.get(supplierIndex).buildPPChartFromEngine(_identifier, chartEngine);
            }
            else if(typeIndex == 1) //Q-Q
            {
                chartEngine = _dataSuppliers.get(supplierIndex).generateQQChartEngine(_model.getCheckedRows());
                builtChart = _dataSuppliers.get(supplierIndex).buildQQChartFromEngine(_identifier, chartEngine);
            }
            else
            //CDF
            {
                chartEngine = _dataSuppliers.get(supplierIndex).generateCDFChartEngine(_model.getCheckedRows());
                builtChart = _dataSuppliers.get(supplierIndex).buildCDFChartFromEngine(_identifier, chartEngine);
            }

            //If _chartPanel is null, then we need to create it and setup the navigation panel.
            if(_chartPanel == null)
            {
                _chartPanel = new ChartEngineChartAndTablePanel(chartEngine, builtChart);
                _splitPane.setTopComponent(_chartPanel);

                //With the updated chart panel, setup the navigation component, glasspane, rootpane, and set the split pane top component.
                final JPanel navPanelHolder = new JPanel(new BorderLayout());
                navPanelHolder.setPreferredSize(new Dimension(190, 150));
                navPanelHolder.setOpaque(false);
                _navPanel = CombinedDomainChartNavigationPanel.applyPopupNavigationPanelToChart(_chartPanel.getChartPanel(),
                                                                                                navPanelHolder.getPreferredSize(),
                                                                                                true);
                navPanelHolder.add(_navPanel, BorderLayout.CENTER);
                navPanelHolder.setVisible(true);

                final JRootPane rootPane = new JRootPane();
                rootPane.setContentPane(_chartPanel);
                final ComponentPanelGlassPane glassPane = new ComponentPanelGlassPane(Lists.newArrayList((Component)navPanelHolder),
                                                                                      FlowLayout.TRAILING);
                glassPane.addToPane(rootPane);
                glassPane.setVisible(true);
                _splitPane.setTopComponent(rootPane);
            }
            //If _chartPanel is not null, set its displayed engine and chart and reset the navPanel accordingly.  Repaint the panel.
            else
            {
                _chartPanel.setChartEngineAndChart(chartEngine, chartEngine.buildChart());
                _navPanel.resetNavigatedChart();
                repaint();
            }

            //Put the new rootPane into place
        }
        catch(final Throwable t)
        {
            t.printStackTrace();
            _splitPane.setTopComponent(HSwingFactory.createErrorMessagePane("Error Building Diagnostic", t.getMessage()));
        }
    }

    public static void main(final String[] args) throws IOException
    {
        try
        {
            final TimeSeriesArrays tss = TimeSeriesArraysTools.readFromFile(new File("testdata/testBed/historicalData/CREC1HOF.precip.xml"));
            final DistributionDiagnosticPanel panel = new DistributionDiagnosticPanel(LocationAndDataTypeIdentifier.get(tss.get(0)),
                                                                                      TimeSeriesArraysTools.convertTimeSeriesArraysToList(tss),
                                                                                      0.25d);

            final JFrame frame = new JFrame("Q-Q TEST");
            frame.setContentPane(panel);
            frame.setSize(500, 500);
            frame.setDefaultCloseOperation(JFrame.EXIT_ON_CLOSE);
            frame.setVisible(true);
        }
        catch(final Exception e)
        {
            e.printStackTrace();
        }
    }

}
