package ohd.hseb.hefs.mefp.sources.historical;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

import javax.swing.table.DefaultTableModel;

import ohd.hseb.hefs.utils.dist.DistributionType;
import ohd.hseb.hefs.utils.gui.jtable.TableTools;
import ohd.hseb.hefs.utils.gui.jtable.models.GetRowTableModel;
import ohd.hseb.hefs.utils.gui.jtable.models.MaxWidthsTableModel;
import ohd.hseb.hefs.utils.gui.jtable.models.MinWidthsTableModel;
import ohd.hseb.hefs.utils.gui.jtable.models.PreferredWidthsTableModel;
import ohd.hseb.hefs.utils.gui.jtable.models.ToolTipTableModel;

/**
 * Model used to show the distributions fit via a {@link DistributionDiagnosticDataSourceSupplier}.
 * 
 * @author Hank.Herr
 */
@SuppressWarnings("serial")
public class DistributionDiagnosticTableModel extends DefaultTableModel implements MaxWidthsTableModel,
GetRowTableModel<DistributionType>, PreferredWidthsTableModel, MinWidthsTableModel, ToolTipTableModel
{

    private DistributionDiagnosticDataSourceSupplier _supplier;
    private final boolean[] _rowSelected;

    /**
     * @param supplier The {@link DistributionDiagnosticDataSourceSupplier} that supplies the data needed for this
     *            table.
     */
    public DistributionDiagnosticTableModel(final DistributionDiagnosticDataSourceSupplier supplier)
    {
        super();
        _supplier = supplier;
        _rowSelected = new boolean[supplier.getFittedDistributions().size()];
        Arrays.fill(_rowSelected, false);
        _rowSelected[0] = true;
    }

    /**
     * Set the rows checked for display, clearing out the checked list prior to setting.
     */
    public void setCheckedRows(final Iterable<DistributionType> types)
    {
        Arrays.fill(_rowSelected, false);
        for(final DistributionType type: types)
        {
            final int index = _supplier.getDistributionTypes().indexOf(type);
            _rowSelected[index] = true;
        }
    }

    /**
     * @return {@link List} of selected {@link DistributionType} instances based on the first column checked state of
     *         the table.
     */
    public List<DistributionType> getCheckedRows()
    {
        final List<DistributionType> selectedTypes = new ArrayList<>();
        for(int i = 0; i < _rowSelected.length; i++)
        {
            if(_rowSelected[i])
            {
                selectedTypes.add(getRow(i));
            }
        }
        return selectedTypes;
    }

    public void setSupplier(final DistributionDiagnosticDataSourceSupplier supplier)
    {
        _supplier = supplier;
        this.fireTableDataChanged();
    }

    @Override
    public int getRowCount()
    {
        if(_supplier == null)
        {
            return 0;
        }
        return _supplier.getFittedDistributions().size();
    }

    @Override
    public int getColumnCount()
    {
        return 5;
    }

    @Override
    public String getColumnName(final int columnIndex)
    {
        if(columnIndex == 0)
        {
            return "";
        }
        else if(columnIndex == 1)
        {
            return "Distribution";
        }
        else if(columnIndex == 2)
        {
            return "Corr";
        }
        else if(columnIndex == 3)
        {
            return "MaxErr";
        }
        else if(columnIndex == 4)
        {
            return "MSE";
        }
        return null;
    }

    @Override
    public Class<?> getColumnClass(final int columnIndex)
    {
        if(columnIndex == 0)
        {
            return Boolean.class;
        }
        else if(columnIndex == 1)
        {
            return String.class;
        }
        return Double.class;
    }

    @Override
    public boolean isCellEditable(final int rowIndex, final int columnIndex)
    {
        if(columnIndex == 0)
        {
            return true;
        }
        return false;
    }

    @Override
    public Object getValueAt(final int rowIndex, final int columnIndex)
    {
        if(columnIndex == 0)
        {
            return _rowSelected[rowIndex];
        }
        else if(columnIndex == 1)
        {
            return _supplier.getDistributionTypes().get(rowIndex);
        }
        else if(columnIndex == 2)
        {
            return _supplier.getCorrelation(rowIndex);
        }
        else if(columnIndex == 3)
        {
            return _supplier.getMaximumError(rowIndex);
        }
        else if(columnIndex == 4)
        {
            return _supplier.getMeanSquaredError(rowIndex);
        }
        return null;

    }

    @Override
    public void setValueAt(final Object aValue, final int rowIndex, final int columnIndex)
    {
        if(columnIndex == 0)
        {
            _rowSelected[rowIndex] = ((Boolean)aValue).booleanValue();
            this.fireTableDataChanged();
        }
    }

    @Override
    public DistributionType getRow(final int index)
    {
        return _supplier.getDistributionTypes().get(index);
    }

    @Override
    public int getRowIndex(final DistributionType row)
    {
        return _supplier.getDistributionTypes().indexOf(row);
    }

    @Override
    public Integer getPreferredWidth(final int column)
    {
        //Fixed checkbox column width.
        if(column == 0)
        {
            return getMaxWidth(0);
        }
        //Want distribution name column wider than the others.
        if(column == 1)
        {
            return 100;
        }
        else
        {
            return 50;
        }
    }

    @Override
    public Integer getMaxWidth(final int column)
    {
        //Fixed checkbox column width.
        if(column == 0)
        {
            return TableTools.getStatusColumnFixedWidth() + 4;
        }
        if(column == 1)
        {
            return 150;
        }
        return null;
    }

    @Override
    public Integer getMinWidth(final int column)
    {
        //Fixed checkbox column width.
        if(column == 0)
        {
            return getMaxWidth(0);
        }
        //Want distribution name column wider than the others.
        if(column == 1)
        {
            return 100;
        }
        else
        {
            return 50;
        }
    }

    @Override
    public String getColumnHeaderToolTip(final int modelColIndex)
    {
        return null;
    }

    @Override
    public String getCellToolTip(final int modelRowIndex, final int modelColIndex)
    {
        //Displays distribution info as a tool tip for the distribution ceell.
        if(modelColIndex == 1)
        {
            return _supplier.getFittedDistributions().get(modelRowIndex).toString();
        }
        return null;
    }

}
