package ohd.hseb.hefs.mefp.sources.historical;

import java.io.File;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;

import nl.wldelft.util.timeseries.TimeSeriesArray;
import nl.wldelft.util.timeseries.TimeSeriesArrays;
import ohd.hseb.hefs.mefp.sources.MEFPSourceDataHandler;
import ohd.hseb.hefs.mefp.tools.MEFPTools;
import ohd.hseb.hefs.mefp.tools.QuestionableMessageMap;
import ohd.hseb.hefs.mefp.tools.QuestionableTools;
import ohd.hseb.hefs.pe.model.ParameterEstimationException;
import ohd.hseb.hefs.pe.notice.StepUnitsUpdatedNotice;
import ohd.hseb.hefs.pe.sources.pixml.GenericPIXMLDataHandler;
import ohd.hseb.hefs.pe.tools.LocationAndDataTypeIdentifier;
import ohd.hseb.hefs.utils.notify.NoticePoster;
import ohd.hseb.hefs.utils.tools.ParameterId;
import ohd.hseb.hefs.utils.tsarrays.TimeSeriesHeaderInfo;

import org.apache.commons.io.FileUtils;
import org.apache.logging.log4j.Logger;
import org.apache.logging.log4j.LogManager;

/**
 * Load time series from historical time series output in XML and generate processed historical binary files. It is
 * assumed the XML files reside under $HEFSMODELS/mefppe/historical. All XML in this directory will be read and
 * inventoried when this handler is constructed.
 * 
 * @author hank.herr
 */
public class HistoricalDataHandler extends GenericPIXMLDataHandler implements MEFPSourceDataHandler
{
    private static final Logger LOG = LogManager.getLogger(HistoricalDataHandler.class);

    /**
     * Stores the prepared data file directory, based on the base directory. This must NOT be set to null here (in the
     * declaration) because of the order of events. In particular, if the below includes '= null;', then when the
     * {@link File} and {@link NoticePoster} constructor is called, this directory will be reset to null AFTER the time
     * series are already loaded from the prepared data file directory via the super constructor. I'm not sure why it
     * follows that order.
     */
    private File _preparedDataFileDirectory;

    public HistoricalDataHandler() throws Exception
    {
        this(new File("."), null);
    }

    /**
     * Sets the base directory and then calls addAllFilesInHistoricalDataDirectoryToUnreadList() followed by
     * initialize(). The result is an instance that has been initialized based on files found in the historical file
     * directory.
     * 
     * @param baseDirectory
     * @throws Exception
     */
    public HistoricalDataHandler(final File baseDirectory, final NoticePoster poster) throws Exception
    {
        super(baseDirectory, "historicalData", poster);

        //TODO Note that the super call above calls these two items.  However, I have to call it again now.  If I don't, then
        //when generatePreparedFileForIdentifier(...) is called below the _preparedDataFileDirectory will be null.  Why?
        //This seems completely redundant.
//        setDataHandlerBaseDirectory(baseDirectory);
//        initialize();
    }

    /**
     * @param identifier
     * @return The prepared data file for the given identifier, which must be a translated identifier (see
     *         {@link #translateIdentifier(LocationAndDataTypeIdentifier)} below).
     */
    protected File generatePreparedFileForIdentifier(final LocationAndDataTypeIdentifier identifier)
    {
        final String fileName = _preparedDataFileDirectory.getAbsolutePath() + "/"
            + MEFPTools.determineProcessedHistoricalDataFileName(identifier);
        return new File(fileName);
    }

    @Override
    public LocationAndDataTypeIdentifier translateIdentifier(final LocationAndDataTypeIdentifier identifier)
    {
        //min/max temp are converted to MAT.
        if(identifier.isTemperatureDataType())
        {
            if(identifier.isMinimumDataType() || identifier.isMaximumDataType())
            {
                //This version is called in order to preserve lat/lon, 
                //since this is called when building the initial list of identifiers.
                return LocationAndDataTypeIdentifier.get(identifier, ParameterId.MAT.name());
            }
        }
        return super.translateIdentifier(identifier);
    }

    @Override
    public boolean areAllRequiredTimeSeriesPresent(final LocationAndDataTypeIdentifier translatedIdentifier,
                                                   final List<TimeSeriesHeaderInfo> availableTimeSeriesHeaders)
    {
        //Observed temp needs a TMIN and TMAX variable.
        if(translatedIdentifier.isTemperatureDataType() && translatedIdentifier.isObservedDataType())
        {
            if(availableTimeSeriesHeaders.size() < 2)
            {
                return false;
            }
            boolean tminFound = false;
            boolean tmaxFound = false;
            for(final TimeSeriesHeaderInfo info: availableTimeSeriesHeaders)
            {
                tminFound = tminFound
                    || ParameterId.valueOf(info.getTimeSeriesHeader().getParameterId()).equals(ParameterId.TMIN);
                tmaxFound = tminFound
                    || ParameterId.valueOf(info.getTimeSeriesHeader().getParameterId()).equals(ParameterId.TMAX);
            }
            return (tminFound && tmaxFound);
        }
        return super.areAllRequiredTimeSeriesPresent(translatedIdentifier, availableTimeSeriesHeaders);
    }

    @Override
    public void setDataHandlerBaseDirectory(final File directory)
    {
        super.setDataHandlerBaseDirectory(directory);
        _preparedDataFileDirectory = new File(getDataHandlerBaseDirectory().getAbsolutePath()
            + "/processedHistoricalData");
    }

    @Override
    // Loads prepared time series into _identifierToTimeSeriesMap and _listOfTimeSeries.
    public void loadPreparedTimeSeries(final List<LocationAndDataTypeIdentifier> identifiers) throws Exception
    {
        //Error out if any identifier is a forecast identifier, which MEFPPE cannot work with.
        for(final LocationAndDataTypeIdentifier id: identifiers)
        {
            if(id.isForecastDataType())
            {
                throw new Exception("Identifier " + id.buildStringToDisplayInTree()
                    + " is a forecast data type identifier, which is not allowed for MEFPPE!");
            }
        }

        LOG.info("Loading all prepared time series for a list of " + identifiers.size()
            + " location and data type identifiers.");

        clearLoadedTimeSeries();

        // Loop through all identifiers.
        for(final LocationAndDataTypeIdentifier identifier: identifiers)
        {
            final File source = generatePreparedFileForIdentifier(identifier);
            final TimeSeriesArrays seriesList = ProcessedHistoricalBinaryFileTools.readSeries(source, identifier);
            for(int i = 0; i < seriesList.size(); i++)
            {
                final TimeSeriesArray ts = seriesList.get(i);
                addLoadedTimeSeries(ts);
            }
        }

        //Put the time series into their lists and check for empty.
        putTimeSeriesIntoLists();
        if(getListOfForecastTimeSeries().isEmpty() && getListOfObservedTimeSeries().isEmpty())
        {
            throw new Exception("No matching time series could be found.");
        }

        LOG.info("Done loading all prepared time series for the given location and data type identifiers. Found "
            + getListOfObservedTimeSeries().size() + " matching time series.");
    }

    @Override
    public void prepareDataFiles() throws Exception
    {
        final String directoryName = _preparedDataFileDirectory.getAbsolutePath();

        for(final LocationAndDataTypeIdentifier identifier: getObservedIdentifiersForWhichReqdDataWasFound())
        {
            final List<TimeSeriesArray> allTS = getLoadedTimeSeriesForIdentifer(identifier);
            if((allTS != null) && (!allTS.isEmpty()))
            {
                final File sbinFile = this.generatePreparedFileForIdentifier(identifier);
                FileUtils.forceMkdir(sbinFile.getParentFile());

                if(identifier.isPrecipitationDataType())
                {
                    try
                    {
                        ProcessedHistoricalBinaryFileTools.writePrecipitationSeries(sbinFile, allTS.get(0));
                    }
                    catch(final Exception e)
                    {
                        throw new Exception("Unable to write precipitation processed historical binary file "
                            + sbinFile.getAbsolutePath() + ": " + e.getMessage());
                    }
                }
                else if(identifier.isTemperatureDataType())
                {
                    try
                    {
                        ProcessedHistoricalBinaryFileTools.writeTemperatureSeries(sbinFile, allTS);
                    }
                    catch(final Exception e)
                    {
                        throw new Exception("Unable to write temperature processed historical binary file "
                            + sbinFile.getAbsolutePath() + ": " + e.getMessage());
                    }
                }

                try
                {
                    QuestionableTools.deleteQuestionableFile(directoryName, identifier);
                    QuestionableTools.createQuestionableFile(directoryName, identifier, allTS, LOG);
                    // uncomment for debugging
                    // final HashMap<String, HashMap<Long, HashMap<Long, String>>> testHash = loadQuestionableHash(identifier);
                    // System.err.println(testHash);
                }
                catch(final Throwable t)
                {
                    // do nothing
                }
            }
        }
        post(new StepUnitsUpdatedNotice(this,
                                        HistoricalPEStepProcessor.class,
                                        getObservedIdentifiersForWhichReqdDataWasFound()));

        //XXX DEBUG
//        final HashMap<Object, List<TimeSeriesArray>> sourceToTSMap2 = Maps.newHashMap();
//        for(final LocationAndDataTypeIdentifier identifier: _usedSources.keySet())
//        {
//            this.loadPreparedTimeSeries(Lists.newArrayList(identifier));
//            System.err.println("####>> for identifier " + identifier.buildStringToDisplayInTree() + " -- found "
//                + getIdentifierToTimeSeriesMap().values().size() + " ts");
//            if(sourceToTSMap2.get(_usedSources.get(identifier)) == null)
//            {
//                sourceToTSMap2.put(_usedSources.get(identifier),
//                                   Lists.newArrayList(getIdentifierToTimeSeriesMap().values()));
//            }
//            else
//            {
//                sourceToTSMap2.get(_usedSources.get(identifier)).addAll(getIdentifierToTimeSeriesMap().values());
//            }
//            System.err.println("####>>     map size -- " + sourceToTSMap2.get(_usedSources.get(identifier)).size());
//        }
//        for(final Object source: sourceToTSMap2.keySet())
//        {
//            final String newName = (String)source + ".new.xml";
//            System.err.println("####>> to source " + newName + " writing " + sourceToTSMap2.get(source).size());
//            TimeSeriesArraysTools.writeToFile(new File(newName), sourceToTSMap2.get(source));
//        }
//        System.err.println("####>> DONE!!!");
    }

    @Override
    public boolean havePreparedDataFilesBeenCreatedAlready(final LocationAndDataTypeIdentifier identifier)
    {
        final File preparedDataFile = this.generatePreparedFileForIdentifier(identifier);
        return preparedDataFile.exists();
    }

    @Override
    public boolean arePreparedDataFilesUpToDate(final LocationAndDataTypeIdentifier identifier)
    {
        final List<Object> sources = getSourcesProvidingDataForIdentifier(identifier);
        boolean result = true;
        for(final Object source: sources)
        {
            if(source instanceof String)
            {
                final File historicalXMLFile = new File((String)source);
                final File preparedDataFile = this.generatePreparedFileForIdentifier(identifier);
                result = result && (historicalXMLFile.lastModified() < preparedDataFile.lastModified());
            }
        }
        return result;
    }

    @Override
    public List<LocationAndDataTypeIdentifier> getIdentifiersWithData()
    {
        //Return only observed identifiers, because that list is what will drive the parameter estimation process.
        //There should be no forecasts, anyway.
        return this.getObservedIdentifiersForWhichReqdDataWasFound();
    }

    @Override
    public Collection<TimeSeriesArray> getLoadedForecastTimeSeries(final LocationAndDataTypeIdentifier identifier)
    {
        return new ArrayList<TimeSeriesArray>();
    }

    @Override
    public File getPreparedDataFilesDirectory()
    {
        return this._preparedDataFileDirectory;
    }

    @Override
    public List<File> generateListOfPreparedDataFiles(final LocationAndDataTypeIdentifier identifier)
    {
        final List<File> files = new ArrayList<File>();
        final File file = this.generatePreparedFileForIdentifier(identifier);
        if(file != null)
        {
            files.add(file);
        }
        return files;
    }

    @Override
    public String translateExternalParameter(final String parameterId)
    {
        //XXX May want to make this more generic.  Such as observed, temperature, min converts to TMIN and obs, temp, max converts to TMAX, regardless
        //of the external parameter id used.
        if(parameterId.equals(ParameterId.TAMN.toString()))
        {
            return ParameterId.TMIN.toString();
        }
        if(parameterId.equals(ParameterId.TAMX.toString()))
        {
            return ParameterId.TMAX.toString();
        }
        return parameterId;
    }

    @Override
    public boolean includeTimeSeries(final LocationAndDataTypeIdentifier identifier)
    {
//        super.includeTimeSeries(null);
        //Restrict checking to TMIN and TMAX for temperature data. I know this can be combined into one if,
        //but I think its easier to read when precip and temp are handled separately.
        if(identifier.isPrecipitationDataType())
        {
            return true;
        }

        //Temperature check.
        return ((identifier.getParameterId().equals(ParameterId.MAT.toString()))
            || (identifier.getParameterId().equals(ParameterId.TMIN.toString())) || (identifier.getParameterId().equals(ParameterId.TMAX.toString())));
//        return ((identifier.isPrecipitationDataType() || identifier.isTemperatureDataType()) && (!identifier.isForecastDataType()));
    }

    @Override
    public String getPIServiceQueryId()
    {
        return "All Historical Data";
    }

    @Override
    public String getPIServiceClientId()
    {
        return "MEFPPE";
    }

    @Override
    public TimeSeriesArray getSingleObservedTimeSeries(final LocationAndDataTypeIdentifier identifier) throws ParameterEstimationException
    {
        final Collection<TimeSeriesArray> TSs = getLoadedObservedTimeSeries(identifier);
        if(TSs.size() != 1)
        {
            throw new ParameterEstimationException("Expected one observed time series corresponding to identifier "
                + identifier.buildStringToDisplayInTree() + ", but found " + TSs.size());
        }
        return TSs.iterator().next();
    }

    @Override
    public Collection<TimeSeriesArray> loadHindcastTimeSeries(final LocationAndDataTypeIdentifier identifier,
                                                              final ParameterId dataTypeToLoad,
                                                              final long forecastTime) throws Exception
    {
        //No hindcast time series is needed here!!!
        return null;
    }

    /**
     * @param identifier - identifier used to construct the questionable file name
     * @return The map created by {@link QuestionableTools#toHash(java.io.File)}.
     * @throws Exception
     */
    @Override
    public QuestionableMessageMap loadQuestionableHash(final LocationAndDataTypeIdentifier identifier) throws Exception
    {
        final String obsDirectory = getPreparedDataFilesDirectory().getAbsolutePath();
        File questionableFile;

        // Get the obsHash

        QuestionableMessageMap obsHash = null;

        questionableFile = QuestionableTools.getFile(obsDirectory, identifier);

        if(questionableFile.exists())
        {
            obsHash = new QuestionableMessageMap(questionableFile);
        }

        return (obsHash); // could be null
    }

//THE BELOW ILLUSTRATES HOW TO CREATE MY OWN HASHMAP IN WHICH FILES CAN BE KEYS.  FILES CAUSE PROBLEMS BECAUSE TWO FILE OBJECTS
//CAN POINT TO THE SAME FILE BUT HAVE DIFFERENT HASHCODE VALUES.
//    /**
//     * Special inner class is needed to convert from File to String (absolute path) because the File hashCode method
//     * does not allow for two File objects that point to the same file to have the same key. This will map the File to
//     * its absolute path when putting it in the map and vice versa when pulling it out.
//     * 
//     * @author hank.herr
//     * @param <K> Usually Object
//     * @param <V> Whatever
//     */
//    private class SourceKeyLinkedHashMap<K, V> extends HashMap<K, V>
//    {
//        private static final long serialVersionUID = 1L;
//
//        @Override
//        public V get(Object key)
//        {
//            if(key instanceof File)
//            {
//                return super.get(((File)key).getAbsolutePath());
//            }
//            return super.get(key);
//        }
//
//        @Override
//        public V put(K key, V value)
//        {
//            this.keySet();
//            if(key instanceof File)
//            {
//                return super.put((K)((File)key).getAbsolutePath(), value);
//            }
//            return super.put(key, value);
//        }
//
//        @Override
//        public Set<K> keySet()
//        {
//            LinkedHashMap<K, V> keyMap = new LinkedHashMap<K, V>();
//
//            Set<K> ks = super.keySet();
//            for(K source: ks)
//            {
//                if(source instanceof String)
//                {
//                    keyMap.put((K)(new File((String)source)), null);
//                }
//                else
//                {
//                    keyMap.put(source, null);
//                }
//            }
//
//            return keyMap.keySet();
//        }
//    }

}
