package ohd.hseb.hefs.mefp.sources.historical;

import java.awt.Color;
import java.util.Collection;

import nl.wldelft.util.timeseries.TimeSeriesArray;
import ohd.hseb.charter.ChartEngine;
import ohd.hseb.charter.datasource.DefaultXYChartDataSource;
import ohd.hseb.charter.panel.ChartEngineTableModel;
import ohd.hseb.charter.parameters.DataSourceDrawingParameters;
import ohd.hseb.charter.parameters.SeriesDrawingParameters;
import ohd.hseb.hefs.mefp.tools.QuestionableMessageMap;
import ohd.hseb.hefs.pe.tools.DiagnosticChartBuilder;
import ohd.hseb.hefs.pe.tools.LocationAndDataTypeIdentifier;
import ohd.hseb.hefs.pe.tools.TimeSeriesChartDiagnosticPanel;
import ohd.hseb.hefs.pe.tools.TimeSeriesChartDiagnosticTableModel;
import ohd.hseb.hefs.pe.tools.TimeSeriesSorter;
import ohd.hseb.hefs.utils.gui.help.HelpFile;
import ohd.hseb.hefs.utils.tools.ParameterId;

import com.google.common.base.Supplier;

@SuppressWarnings("serial")
@HelpFile("PDFHELP:helpManual.pdf#HistoricalDiagnosticsPanelRef")
public class HistoricalDiagnosticPanel extends TimeSeriesChartDiagnosticPanel
{
    private final static Color QUESTIONABLE_MARK = new Color(255, 0, 0, 64); // red
    private final static Color QUESTIONABLE_ZONE = new Color(255, 0, 0, 64); // red

    private final boolean _precipitation;
    /**
     * The map of questionable data; must be available by the time {@link #buildTableModelSupplier()} is called.
     */
    private final QuestionableMessageMap _questionableMap;

    @SuppressWarnings("unchecked")
    public HistoricalDiagnosticPanel(final LocationAndDataTypeIdentifier identifier,
                                     final Collection<TimeSeriesArray> timeSeriesToDisplay,
                                     final QuestionableMessageMap questionableMap)
    {
        super(identifier, timeSeriesToDisplay);
        _questionableMap = questionableMap;
        _precipitation = identifier.isPrecipitationDataType();
        updateChart();
    }

    @Override
    protected boolean hasYearSpinner()
    {
        return false;
    }

    @Override
    protected boolean hasForecastList()
    {
        return false;
    }

    @Override
    protected boolean allowsSouthNavigationPanel()
    {
        return true;
    }

    /**
     * This override is needed in order to ensure that {@link #_questionableMap} gets to the table's
     * {@link TimeSeriesChartDiagnosticTableModel}.
     */
    @Override
    protected Supplier<ChartEngineTableModel> buildTableModelSupplier()
    {
        return new Supplier<ChartEngineTableModel>()
        {
            @Override
            public ChartEngineTableModel get()
            {
                final TimeSeriesChartDiagnosticTableModel model = new TimeSeriesChartDiagnosticTableModel();
                model.setQuestionableMap(_questionableMap);
                return model;
            }
        };
    }

    @Override
    protected ChartEngine buildChart() throws Exception
    {
        String parameterFile;
        if(_precipitation)
        {
            parameterFile = "mefppe/diagnostics/historicalPrecipitationDisplayDiagnostics.xml";
        }
        else
        {
            parameterFile = "mefppe/diagnostics/historicalTemperatureDisplayDiagnostics.xml";
        }

        //Specific action for temperature: Remove the MAT time series if empty.  If the MAT was not found or it was removed
        //here, update a flag to indicate that the series drawing parameters must be removed.  Also, ensure that the minimum time
        //series precedes the maximum time series so that it matches the product parameter ordering.
        boolean removeMATSeriesDrawingParms = false;
        if(getIdentifier().isTemperatureDataType())
        {
            final TimeSeriesSorter baseSorter = new TimeSeriesSorter(getAllSeries());

            //Get min and max.
            final TimeSeriesArray tminSeries = baseSorter.restrictViewToParameters(ParameterId.TMIN).getOnly();
            final TimeSeriesArray tmaxSeries = baseSorter.restrictViewToParameters(ParameterId.TMAX).getOnly();

            //Get mat if any.  If none is found or it is all empty, leave matSeries as null for later.
            TimeSeriesArray matSeries = null;
            final TimeSeriesSorter sorter = baseSorter.restrictViewToParameters(ParameterId.MAT);
            if(!sorter.isEmpty() && (sorter.getOnly().isEmpty())) //All missing MAT data!
            {
                removeMATSeriesDrawingParms = true;
            }
            else if(sorter.isEmpty()) //No MAT data.
            {
                removeMATSeriesDrawingParms = true;
            }
            else
            {
                //This will throw an exception only if more than one MAT is provided, which is bad.
                //The case of there being no MAT series is handled by the other parts of this if-else-clause.
                matSeries = sorter.getOnly();
            }

            //Reconstruct all series forcing the correct order: min must always be before max in all cases.
            getAllSeries().clear();
            if(matSeries != null)
            {
                getAllSeries().add(matSeries);
            }
            getAllSeries().add(tminSeries);
            getAllSeries().add(tmaxSeries);
        }

        final DiagnosticChartBuilder builder = new DiagnosticChartBuilder(getIdentifier(),
                                                                          parameterFile,
                                                                          HistoricalForecastSource.SOURCE_ID,
                                                                          getAllSeries());

        //Specific action for temperature: If the MAT needs to be removed, then we need to remove the series drawing parameters which were
        //read in during the construction of builder.
        if(removeMATSeriesDrawingParms)
        {
            final DataSourceDrawingParameters parms = builder.getParameters()
                                                             .getTemplateParameters()
                                                             .getChartDrawingParameters()
                                                             .getDataSourceParameters()
                                                             .get(0);

            final SeriesDrawingParameters tminParms = parms.getSeriesDrawingParametersForSeriesIndex(1);
            tminParms.setCreatedIndexOfSeriesToModify(0);
            final SeriesDrawingParameters tmaxParms = parms.getSeriesDrawingParametersForSeriesIndex(2);
            tmaxParms.setCreatedIndexOfSeriesToModify(1);
            parms.removeAllSeriesDrawingParameters();
            parms.addSeriesDrawingParameters(tminParms);
            parms.addSeriesDrawingParameters(tmaxParms);
        }
        builder.setQuestionableMap(_questionableMap);
        builder.addDomainThresholdsForQuestionableValues(QUESTIONABLE_MARK, QUESTIONABLE_ZONE);

        final ChartEngine engine = builder.buildChartEngine();

        //Setup the source names
        ((DefaultXYChartDataSource)engine.getDataSources().get(0)).setSourceNameInTable("Historical Observed");
        for(int i = 1; i < engine.getDataSources().size(); i++) //Other sources will only exist for certain circumstances and can be ignored
        {
            ((DefaultXYChartDataSource)engine.getDataSources().get(i)).setSourceNameInTable("Ignore: Extra Source Created for Charting ("
                + i + ")");
        }

        return engine;
    }
}
