package ohd.hseb.hefs.mefp.sources.historical;

import static com.google.common.collect.Lists.newArrayList;

import java.awt.BorderLayout;
import java.awt.Component;
import java.awt.Dimension;
import java.awt.event.ActionEvent;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;

import javax.swing.JButton;
import javax.swing.JOptionPane;
import javax.swing.event.ListSelectionEvent;
import javax.swing.event.ListSelectionListener;

import nl.wldelft.util.timeseries.DefaultTimeSeriesHeader;
import nl.wldelft.util.timeseries.TimeSeriesArray;
import ohd.hseb.hefs.mefp.pe.core.MEFPParameterEstimatorRunInfo;
import ohd.hseb.hefs.mefp.tools.QuestionableTools;
import ohd.hseb.hefs.pe.core.ParameterEstimatorStepOptionsPanel;
import ohd.hseb.hefs.pe.core.StepTableModel;
import ohd.hseb.hefs.pe.core.StepUnit;
import ohd.hseb.hefs.pe.notice.SelectedIdentifiersChangedNotice;
import ohd.hseb.hefs.pe.notice.StepUpdatedNotice;
import ohd.hseb.hefs.pe.tools.DefaultStepTableStatusProvider;
import ohd.hseb.hefs.pe.tools.GenericSummaryTablePanel;
import ohd.hseb.hefs.pe.tools.LocationAndDataTypeIdentifier;
import ohd.hseb.hefs.pe.tools.TimeSeriesSorter;
import ohd.hseb.hefs.utils.gui.help.HelpFile;
import ohd.hseb.hefs.utils.gui.tools.SelfListeningButton;
import ohd.hseb.hefs.utils.gui.tools.SwingTools;
import ohd.hseb.hefs.utils.jobs.GenericJob;
import ohd.hseb.hefs.utils.jobs.HJobMonitorDialog;
import ohd.hseb.hefs.utils.jobs.JobListener;
import ohd.hseb.hefs.utils.tools.ListTools;
import ohd.hseb.hefs.utils.tools.ParameterId;
import ohd.hseb.hefs.utils.tools.StringTools;

import org.apache.logging.log4j.Logger;
import org.apache.logging.log4j.LogManager;

import com.google.common.collect.Lists;

/**
 * Design: Display a table of location ids based on historical data read in (i.e., Locations Table). For each location,
 * allow for specifying either archive db input (forecast lid and SHEF pedstep and observed SHEF pedstep and maybe lid)
 * or synthetic station computation (MOS-driven synthetic station location id).
 * 
 * @author hank.herr
 */
@SuppressWarnings("serial")
@HelpFile("PDFHELP:helpManual.pdf#HistoricalDataSubpanelRef")
public class HistoricalPEStepOptionsPanel extends ParameterEstimatorStepOptionsPanel
{
    private static final Logger LOG = LogManager.getLogger(HistoricalPEStepOptionsPanel.class);
    private final HistoricalDataHandler _handler;

    private GenericSummaryTablePanel _locationPanel;
    private JButton _viewButton;
    private JButton _viewDistDiagButton;
    private JButton _removeQuestionableButton;

    public HistoricalPEStepOptionsPanel(final MEFPParameterEstimatorRunInfo runInfo,
                                        final HistoricalPEStepProcessor stepProcessor)
    {
        super(runInfo, stepProcessor);
        _handler = stepProcessor.getHistoricalDataHandler();
        initializeDisplay();
        updateEnablednessOfViewButton();
        updateEnablednessOfRemoveQuestionableButton();
    }

    @Override
    protected void initializeDisplay()
    {
        _viewButton = new SelfListeningButton("chart20x20",
                                              "View the historical time series for a single, selected location and data type.")
        {
            @Override
            public void actionPerformed(final ActionEvent e)
            {
                viewSelectedHistoricalTimeSeries();
            }
        };
        _viewDistDiagButton = new SelfListeningButton("cdfChart20x20",
                                                      "View the distribution diagnostic display for a single, selected location and data type.")
        {
            @Override
            public void actionPerformed(final ActionEvent e)
            {
                viewSelectedDistributionDiagnosticDisplay();
            }
        };
        _removeQuestionableButton = new SelfListeningButton("removeQuestionableFile20x20",
                                                            "<html>Remove files identifying questionable data for selected locations,"
                                                                + "<br>so that they are no longer marked as questionable.<br>Does not change any data.</html>")
        {
            @Override
            public void actionPerformed(final ActionEvent e)
            {
                removeQuestionableFilesForSelectedIdentifiers();
            }
        };

        _locationPanel = new GenericSummaryTablePanel(getRunInfo(),
                                                      "Summary of Available Historical Data",
                                                      new StepTableModel(getRunInfo(),
                                                                         HistoricalPEStepProcessor.class,
                                                                         new DefaultStepTableStatusProvider(getStepProcessor())),
                                                      true,
                                                      newArrayList((Component)_viewButton,
                                                                   (Component)_viewDistDiagButton,
                                                                   (Component)_removeQuestionableButton));

        _locationPanel.addListSelectionListener(new ListSelectionListener()
        {
            @Override
            public void valueChanged(final ListSelectionEvent e)
            {
                updateEnablednessOfViewButton();
                updateEnablednessOfRemoveQuestionableButton();
            }
        });

        setLayout(new BorderLayout());
        add(_locationPanel, BorderLayout.CENTER);
    }

    @Override
    public MEFPParameterEstimatorRunInfo getRunInfo()
    {
        return (MEFPParameterEstimatorRunInfo)super.getRunInfo();
    }

    @Override
    public HistoricalPEStepProcessor getStepProcessor()
    {
        return (HistoricalPEStepProcessor)super.getStepProcessor();
    }

    /**
     * Gets the selected identifiers and, if any of them include questionable files, enables the remove button.
     * Otherwise, the button is disabled.
     */
    private void updateEnablednessOfRemoveQuestionableButton()
    {
        final List<LocationAndDataTypeIdentifier> identifiers = _locationPanel.getSelectedIdentifiers();

        if(identifiers != null && !identifiers.isEmpty()) // identifiers have been selected
        {
            _removeQuestionableButton.setEnabled(QuestionableTools.isQuestionable(_handler.getPreparedDataFilesDirectory()
                                                                                          .getAbsolutePath(),
                                                                                  identifiers));
            if(_removeQuestionableButton.isEnabled())
            {
                _removeQuestionableButton.setToolTipText("<html>Remove files identifying questionable data for selected locations,"
                    + "<br>so locations are no longer marked as questionable." + "<br>Does not change any data.</html>");
            }
        }
        else
        // no identifiers have been selected
        {
            _removeQuestionableButton.setEnabled(false);
            _removeQuestionableButton.setToolTipText("Select rows for the locations for which to remove the questionable data mark.");
        }
    }

    private void updateEnablednessOfViewButton()
    {
        final LocationAndDataTypeIdentifier identifier = _locationPanel.getSelectedIdentifier();
        if(identifier != null)
        {
            _viewButton.setEnabled(_handler.havePreparedDataFilesBeenCreatedAlready(identifier));
            _viewDistDiagButton.setEnabled(_handler.havePreparedDataFilesBeenCreatedAlready(identifier));
            if(_viewButton.isEnabled())
            {
                _viewButton.setToolTipText("View the binary historical data for "
                    + identifier.buildStringToDisplayInTree() + ".");
                _viewDistDiagButton.setToolTipText("View distribution diagnostics for the historical data for "
                    + identifier.buildStringToDisplayInTree() + ".");
            }
            else
            {
                _viewButton.setToolTipText("<html>No binary historical data is available for "
                    + identifier.buildStringToDisplayInTree()
                    + ".<br>Click the Prepare Binary Files Button to construct them.</html>");
                _viewDistDiagButton.setToolTipText(_viewButton.getToolTipText());
            }
        }
        else
        {
            _viewButton.setEnabled(false);
            _viewButton.setToolTipText("Select one row to view historical data for that location and data type.");
            _viewDistDiagButton.setEnabled(false);
            _viewDistDiagButton.setToolTipText("Select one row to view historical data for that location and data type.");
        }
    }

    /**
     * Executes a {@link GenericJob} to remove the questionable files for all selected identifiers.
     */
    private void removeQuestionableFilesForSelectedIdentifiers()
    {
        final List<LocationAndDataTypeIdentifier> identifiers = _locationPanel.getSelectedIdentifiers();

        try
        {
            QuestionableTools.deleteQuestionableFiles(_handler.getPreparedDataFilesDirectory().getAbsolutePath(),
                                                      identifiers);
        }
        catch(final Throwable t)
        {
            t.printStackTrace();
            return;
        }

        _locationPanel.refreshTable();

        // Also refresh the Estimation Location Summary Panel
        post(new StepUpdatedNotice(this, HistoricalPEStepProcessor.class));

    }

    /**
     * Execute a {@link GenericJob} to load the time series for the selected location and display them within a
     * {@link HistoricalDiagnosticPanel}. Note that if the identifier is for temperature, a special portion of the code
     * will load the MAT time series, as well as the normal TMIN/TMAX time series.
     */
    private void viewSelectedHistoricalTimeSeries()
    {
        final List<LocationAndDataTypeIdentifier> identifiers = new ArrayList<LocationAndDataTypeIdentifier>();
        final LocationAndDataTypeIdentifier selectedIdent = _locationPanel.getSelectedIdentifier();
        if(selectedIdent == null)
        {
            return;
        }
        identifiers.add(selectedIdent);

        final GenericJob buildPanelJob = new GenericJob()
        {
            @Override
            public void processJob()
            {
                setIndeterminate(true);
                updateNote("Loading time series from source...");

                Collection<TimeSeriesArray> tsc = null;
                try
                {
                    tsc = loadHistoricalData(selectedIdent);
                }
                catch(final Exception e)
                {
                    e.printStackTrace();
                    fireProcessJobFailure(new Exception("Error loading historical time series for location "
                        + selectedIdent.buildStringToDisplayInTree() + ": " + e.getMessage()), true);
                    return;
                }

                //ensure every loaded time series has the same forecast time so that the forecast time sorting does not change
                //the relative ordering implied by the above code: MAT first, and then FMAP or TMIN/TMAX.  TimeSeriesSorter toList()
                //is called within the DiagnosticChartBuilder, and it will sort time series by forecast time.  For historical
                //time series, the forecast time does not matter.
                for(final TimeSeriesArray ts: tsc)
                {
                    ((DefaultTimeSeriesHeader)ts.getHeader()).setForecastTime(Long.MIN_VALUE);
                }

                updateNote("Constructing chart...");
                try
                {
                    final HistoricalDiagnosticPanel panel = new HistoricalDiagnosticPanel(_locationPanel.getSelectedIdentifier(),
                                                                                          tsc,
                                                                                          _handler.loadQuestionableHash(selectedIdent));
                    fireDiagnostic(panel);
                }
                catch(final Exception e)
                {
                    e.printStackTrace();
                }

                endTask();

                //Dump the time series to a file for reading
                //            TimeSeriesArray[] usedTS = new TimeSeriesArray[100];
                //            for(int i = 0; i < 100; i++)
                //            {
                //                usedTS[i] = ts.get(i);
                //            }
                //            ts = new TimeSeriesArrays(usedTS);
                //            TimeSeriesArraysTools.writeToFile(new File("testdata/diagnosticProducts/gfsTemperatureForecastDisplayDiagnosticsTS.xml"),
                //                                              ts);
            }
        };
        final HJobMonitorDialog jobDialog = new HJobMonitorDialog(this,
                                                                  "Building Chart to Display Historical Time Series",
                                                                  buildPanelJob,
                                                                  false);
        buildPanelJob.addListener(new JobListener()
        {
            @Override
            public void processJobFailure(final Exception exc, final GenericJob theJob, final boolean displayMessage)
            {
                //exc.printStackTrace();
                fireDiagnostic("Unable to build chart displaying historical time series:", exc.getMessage());
                JOptionPane.showMessageDialog(SwingTools.getGlobalDialogParent(HistoricalPEStepOptionsPanel.this),
                                              StringTools.wordWrap(exc.getMessage(), 80),
                                              "Unable to Build Diagnostic Panel",
                                              JOptionPane.ERROR_MESSAGE);
            }

            @Override
            public void processSuccessfulJobCompletion(final GenericJob theJob)
            {
            }
        });
        buildPanelJob.startJob();
        jobDialog.setMinimumSize(new Dimension(350, 10));
        jobDialog.setModal(true);
        jobDialog.setVisible(true);
    }

    private Collection<TimeSeriesArray> loadHistoricalData(final LocationAndDataTypeIdentifier selectedIdent) throws Exception
    {
        //Temperature specific code: loads the MAT time series for displaying with the standard prepared time series.
        //This is not required to construct the diagnostic: code inside of the HistoricalDiagnosticPanel handles the case
        //if the MAT time series is not found.
        final Collection<TimeSeriesArray> tsc = new ArrayList<TimeSeriesArray>();
        if(selectedIdent.isTemperatureDataType())
        {
            try
            {
                final LocationAndDataTypeIdentifier matIdent = LocationAndDataTypeIdentifier.get(selectedIdent.getLocationId(),
                                                                                                 ParameterId.MAT.toString());
                _handler.loadOriginalTimeSeries(Lists.newArrayList(matIdent));
                final TimeSeriesSorter sorter = new TimeSeriesSorter(_handler.getAllLoadedObservedTimeSeries());
                tsc.addAll(sorter.restrictViewToParameters(ParameterId.MAT));
            }
            catch(final Exception e)
            {
                LOG.warn("Unable to load MAT data: " + e.getMessage());
            }
        }

        //General code: Loads the prepared precip or temp time series.
        _handler.loadPreparedTimeSeries(Lists.newArrayList(selectedIdent));
        tsc.addAll(_handler.getAllLoadedObservedTimeSeries());
        return tsc;
    }

    /**
     * Execute a {@link GenericJob} to load the time series for the selected location and display them within a
     * {@link HistoricalDiagnosticPanel}. Note that if the identifier is for temperature, a special portion of the code
     * will load the MAT time series, as well as the normal TMIN/TMAX time series.
     */
    private void viewSelectedDistributionDiagnosticDisplay()
    {
        final List<LocationAndDataTypeIdentifier> identifiers = new ArrayList<LocationAndDataTypeIdentifier>();
        final LocationAndDataTypeIdentifier selectedIdent = _locationPanel.getSelectedIdentifier();
        if(selectedIdent == null)
        {
            return;
        }
        identifiers.add(selectedIdent);

        final GenericJob buildPanelJob = new GenericJob()
        {
            @Override
            public void processJob()
            {
                setIndeterminate(true);
                updateNote("Loading time series from source...");

                Collection<TimeSeriesArray> tsc = null;
                try
                {
                    tsc = loadHistoricalData(selectedIdent);
                }
                catch(final Exception e)
                {
                    e.printStackTrace();
                    fireProcessJobFailure(new Exception("Error loading historical time series for location "
                        + selectedIdent.buildStringToDisplayInTree() + ": " + e.getMessage()), true);
                    return;
                }

                updateNote("Constructing diagnostic display...");
                try
                {
                    final DistributionDiagnosticPanel panel = new DistributionDiagnosticPanel(selectedIdent, tsc, 0.25); //TODO 0.25... hardcoded?
                    fireDiagnostic(panel);
                }
                catch(final Exception e)
                {
                    e.printStackTrace();
                }

                endTask();
            }
        };
        final HJobMonitorDialog jobDialog = new HJobMonitorDialog(this,
                                                                  "Building Chart to Display Historical Time Series",
                                                                  buildPanelJob,
                                                                  false);
        buildPanelJob.addListener(new JobListener()
        {
            @Override
            public void processJobFailure(final Exception exc, final GenericJob theJob, final boolean displayMessage)
            {
                //exc.printStackTrace();
                fireDiagnostic("Unable to build chart displaying historical time series:", exc.getMessage());
                JOptionPane.showMessageDialog(SwingTools.getGlobalDialogParent(HistoricalPEStepOptionsPanel.this),
                                              StringTools.wordWrap(exc.getMessage(), 80),
                                              "Unable to Build Diagnostic Panel",
                                              JOptionPane.ERROR_MESSAGE);
            }

            @Override
            public void processSuccessfulJobCompletion(final GenericJob theJob)
            {
            }
        });
        buildPanelJob.startJob();
        jobDialog.setMinimumSize(new Dimension(350, 10));
        jobDialog.setModal(true);
        jobDialog.setVisible(true);
    }

    @Override
    public List<LocationAndDataTypeIdentifier> getStepUnitsToPerform()
    {
        return _locationPanel.getSelectedIdentifiers();
    }

    @Override
    public void reactToSelectedIdentifiersChanged(final SelectedIdentifiersChangedNotice evt)
    {
        _locationPanel.getModel().setIdentifiers(evt.getIdentifiers());
        _locationPanel.refreshTable();

        // Also refresh the Estimation Location Summary Panel
        post(new StepUpdatedNotice(this, HistoricalPEStepProcessor.class));
    }

    @Override
    public void gotoUnit(final Collection<StepUnit> units)
    {
        _locationPanel.selectIdentifiers(ListTools.convertCollection(units, (LocationAndDataTypeIdentifier)null));
    }
}
