package ohd.hseb.hefs.mefp.sources.historical;

import java.util.ArrayList;
import java.util.List;

import ohd.hseb.hefs.mefp.pe.core.MEFPParameterEstimatorRunInfo;
import ohd.hseb.hefs.mefp.tools.QuestionableStatus;
import ohd.hseb.hefs.mefp.tools.QuestionableTools;
import ohd.hseb.hefs.pe.core.GenericParameterEstimatorStepProcessor;
import ohd.hseb.hefs.pe.core.ParameterEstimatorStepOptionsPanel;
import ohd.hseb.hefs.pe.core.StepUnit;
import ohd.hseb.hefs.pe.tools.LocationAndDataTypeIdentifier;
import ohd.hseb.hefs.utils.jobs.JobMessenger;
import ohd.hseb.hefs.utils.status.StatusIndicator;
import ohd.hseb.hefs.utils.status.StatusLabel;

public class HistoricalPEStepProcessor extends GenericParameterEstimatorStepProcessor
{
    public static final String STATUS_NULL = "<html>The exported PI-xml time series is newer than the processed historical binary file.<br>"
        + "Please recreate the processed historical binary file via the historical data tabbed panel.</html>";
    public static final String STATUS_TRUE = "The processed historical binary file has been prepared and is up-to-date.";
    public static final String STATUS_FALSE = "<html>The processed historical binary file has not been prepared.<br>"
        + "Please create the processed historical binary file via the historical data tabbed panel.</html>";

    public static String PRECIPITATION_SBIN_DIR = "sbmap";
    public static String TEMPERATURE_SBIN_DIR = "sbtxtn";

    private final MEFPParameterEstimatorRunInfo _runInfo;
    private final HistoricalDataHandler _historicalDataHandler;

    /**
     * @param historicalDataHandler An initialized HistoricalDataHandler.
     */
    public HistoricalPEStepProcessor(final MEFPParameterEstimatorRunInfo runInfo)
    {
        _runInfo = runInfo;
        _historicalDataHandler = runInfo.getHistoricalDataHandler();
    }

    public HistoricalDataHandler getHistoricalDataHandler()
    {
        return this._historicalDataHandler;
    }

    @Override
    public StatusIndicator getStatus(final StepUnit unit)
    {
        if(!(unit instanceof LocationAndDataTypeIdentifier))
        {
            throw new IllegalArgumentException("Must be passed an Identifier.");
        }
        final LocationAndDataTypeIdentifier identifier = (LocationAndDataTypeIdentifier)unit;

        if(!doFilesExist(identifier))
        {
            return StatusLabel.make(false, STATUS_FALSE);
        }

        if(!areFilesUpdated(identifier))
        {
            return StatusLabel.make((Boolean)null, STATUS_NULL);
        }

        final String directoryName = _historicalDataHandler.generatePreparedFileForIdentifier(identifier).getParent();

        if(QuestionableTools.isQuestionable(directoryName, identifier))
        {
            // other StatusLabels use a Boolean
            return StatusLabel.make(QuestionableStatus.QUESTIONABLE);
        }

        return StatusLabel.make(true, STATUS_TRUE);
    }

    public boolean doFilesExist(final LocationAndDataTypeIdentifier identifier)
    {
        return _historicalDataHandler.havePreparedDataFilesBeenCreatedAlready(identifier);
    }

    public boolean areFilesUpdated(final LocationAndDataTypeIdentifier identifier)
    {
        return _historicalDataHandler.arePreparedDataFilesUpToDate(identifier);
    }

    @Override
    public void performStep(final StepUnit unit) throws Exception
    {
        final List<LocationAndDataTypeIdentifier> identifiers = new ArrayList<LocationAndDataTypeIdentifier>();
        identifiers.add((LocationAndDataTypeIdentifier)unit);

        try
        {
            JobMessenger.newMonitorSubJob();
            JobMessenger.setMaximumNumberOfSteps(2);
            JobMessenger.updateNote("Loading time series from exported XML file...");
            _historicalDataHandler.loadOriginalTimeSeries(identifiers);

            JobMessenger.madeProgress("Writing binary file...");
            _historicalDataHandler.prepareDataFiles();

            JobMessenger.madeProgress();
        }
        finally
        {
            JobMessenger.clearMonitorSubJob();
        }
    }

    @Override
    public ParameterEstimatorStepOptionsPanel constructOptionsPanel()
    {
        return new HistoricalPEStepOptionsPanel(_runInfo, this);
    }

    @Override
    public String getToolTipTextDescribingStep()
    {
        return "Read in historical data, check for questionable data, and build processed hsitorical binary files to be used in parameter files.";
    }

    @Override
    public String getShortNameForIdentifierTableColumn()
    {
        return "Hist";
    }

    @Override
    public String getToolTipTextForIdentifierTableColumnHeader()
    {
        return "<html>Displays if the processed historical binary file was prepared and is up-to-date.</html>";
    }

    @Override
    public String getTabNameForStep()
    {
        return "Historical Data";
    }

    @Override
    public String getStepNameForRunButton()
    {
        return "Prepare Historical Binary Files";
    }

    @Override
    public String getPerformStepPrefix()
    {
        return "Preparing binary files";
    }
}
