package ohd.hseb.hefs.mefp.sources.historical;

import java.io.File;
import java.io.IOException;
import java.util.List;

import com.google.common.collect.Lists;

import nl.wldelft.util.coverage.PointsGeometry;
import nl.wldelft.util.geodatum.GeoPoint;
import nl.wldelft.util.geodatum.Wgs1984Point;
import nl.wldelft.util.timeseries.DefaultTimeSeriesHeader;
import nl.wldelft.util.timeseries.ParameterType;
import nl.wldelft.util.timeseries.SimpleEquidistantTimeStep;
import nl.wldelft.util.timeseries.TimeSeriesArray;
import nl.wldelft.util.timeseries.TimeSeriesArrays;
import ohd.hseb.hefs.pe.tools.HEFSTools;
import ohd.hseb.hefs.pe.tools.LocationAndDataTypeIdentifier;
import ohd.hseb.hefs.pe.tools.TimeSeriesSorter;
import ohd.hseb.hefs.utils.tools.ParameterId;
import ohd.hseb.hefs.utils.tsarrays.TimeSeriesArraysTools;
import ohd.hseb.util.misc.HCalendar;

public abstract class ProcessedHistoricalBinaryFileTools
{

    /**
     * @param outputFile File to create, which will be a bin file via
     *            {@link TimeSeriesArraysTools#writeDataToBinFile(File, java.util.Collection)}.
     * @param timeSeries Time series to output, at least one of {@link ParameterId#TMIN} and {@link ParameterId#TMAX}
     *            must be indcluded, and only the first one of each will be output.
     * @throws IOException Problem writing file.
     * @throws IllegalArgumentException An expected time series could not be found (either min or max).
     */
    public static void writeTemperatureSeries(final File outputFile,
                                              final List<TimeSeriesArray> timeSeries) throws IOException,
                                                                                      IllegalArgumentException
    {
        final TimeSeriesSorter sorter = new TimeSeriesSorter(timeSeries);
        if(sorter.restrictViewToParameters(ParameterId.TMIN).isEmpty()
            || sorter.restrictViewToParameters(ParameterId.TMAX).isEmpty())
        {
            throw new IllegalArgumentException("Provided time series do not include both tmin and tmax data.");
        }

        final TimeSeriesArray minTS = TimeSeriesSorter.subListByParameterId(timeSeries, ParameterId.TMIN).get(0);
        final TimeSeriesArray maxTS = TimeSeriesSorter.subListByParameterId(timeSeries, ParameterId.TMAX).get(0);

        TimeSeriesArraysTools.writeDataToBinFile(outputFile, Lists.newArrayList(maxTS, minTS));
    }

//XXX FYI -- the diurnal patter temperature to max min code from the scientific Fortran prototype is here:
//                //Fortran code:
//                //TMIN(j) = (9.652*T22(j) - 6.432*T33(j) + 0.480*T44(j))/3.70
//                //TMAX(j) = (10.72*T33(j) - 0.670*T22(j) - 0.800*T44(j))/9.25

    /**
     * This requires that the precipitation time series was converted into a GMT time series from whatever time zone
     * (local?) the original time series is in. It directly writes out the given time series into a binary format, and
     * that requires Z-time data.
     * 
     * @param outputFile
     * @param timeSeries
     * @throws IOException
     */
    public static void writePrecipitationSeries(final File outputFile,
                                                final TimeSeriesArray timeSeries) throws IOException
    {
        //This requires
        TimeSeriesArraysTools.writeDataToBinFile(outputFile, Lists.newArrayList(timeSeries));
    }

    /**
     * Performs a straight read, returning the found time series.
     */
    public static TimeSeriesArrays readSeries(final File inputFile,
                                              final LocationAndDataTypeIdentifier identifier) throws IOException
    {
        if(identifier.isPrecipitationDataType())
        {
            return readPrecipitationSeries(inputFile, identifier);
        }
        else if(identifier.isTemperatureDataType())
        {
            return readTemperatureSeries(inputFile, identifier);
        }
        else
        {
            throw new IOException("Unsupported parameter type: " + identifier.getParameterId());
        }
    }

    /**
     * @param identifier Identifier from which to copy header information.
     * @return A header to use for reading in binary temperature data with a TMIN parameterId. It will contain all
     *         needed header parameters copied from the given identifier. However, it will NOT contain the coordinates!
     */
    public static DefaultTimeSeriesHeader prepareTMINHeader(final LocationAndDataTypeIdentifier identifier)
    {
        final DefaultTimeSeriesHeader header = new DefaultTimeSeriesHeader();
        header.setLocationId(identifier.getLocationId());
        header.setLocationName(identifier.getLocationId());
        header.setLocationDescription(identifier.getLocationId());
        header.setParameterType(ParameterType.ACCUMULATIVE);
        header.setForecastTime(Long.MIN_VALUE);
        header.setUnit("DEGC");
        header.setTimeStep(SimpleEquidistantTimeStep.getInstance(HCalendar.MILLIS_IN_HR * 24));
        header.setParameterId(ParameterId.TMIN.name());
        header.setParameterName(ParameterId.TMIN.name());
        header.setGeometry(new PointsGeometry(new GeoPoint[]{
            new Wgs1984Point(identifier.getLocationLatitude(), identifier.getLocationLongitude())}));
        return header;
    }

    /**
     * The data in the binary files are recorded at 12Z of each day (24h time steps).
     * 
     * @param inputFile File to read.
     * @param identifier Identifier for which to extract data. This is used as the basis for the headers of the returned
     *            time series.
     * @return Time series found in the file for the given identifier. It is assumed that the provided identifier is a
     *         temperature identifier and that will will need to acquire both {@link ParameterId#TMIN} and
     *         {@link ParameterId#TMAX} data.
     * @throws IOException
     */
    public static TimeSeriesArrays readTemperatureSeries(final File inputFile,
                                                         final LocationAndDataTypeIdentifier identifier) throws IOException
    {
        // Make Time Series.
        final DefaultTimeSeriesHeader header = prepareTMINHeader(identifier);
        final List<TimeSeriesArray> readIn = Lists.newArrayList();
        readIn.addAll(TimeSeriesArraysTools.readDataFromBinFile(header, inputFile));

        //Check
        if(readIn.size() != 2)
        {
            throw new IllegalStateException("Did not find two time series in the temperature historical data binary file.");
        }

        //Read in time series now contains two time series that are exactly identical in terms of header info.  The first
        //contains TMAX data, while the second contains TMIN data.  The reason TMAX is is first is because the old
        //version of this file put TMAX first and that made its way into the testing historical bin files.  So, we'll
        //just keep it as TMAX first for now.  Change the first time series header to TMAX.
        ((DefaultTimeSeriesHeader)readIn.get(0).getHeader()).setParameterId(ParameterId.TMAX.name());
        ((DefaultTimeSeriesHeader)readIn.get(0).getHeader()).setParameterName(ParameterId.TMAX.name());

        return TimeSeriesArraysTools.convertListOfTimeSeriesToTimeSeriesArrays(readIn);
    }

    public static DefaultTimeSeriesHeader prepareMAPHeader(final LocationAndDataTypeIdentifier identifier)
    {
        final DefaultTimeSeriesHeader header = new DefaultTimeSeriesHeader();
        header.setLocationId(identifier.getLocationId());
        header.setLocationName(identifier.getLocationId());
        header.setLocationDescription(identifier.getLocationId());
        header.setParameterType(ParameterType.ACCUMULATIVE);
        header.setForecastTime(Long.MIN_VALUE);
        header.setUnit("MM");
        header.setTimeStep(SimpleEquidistantTimeStep.getInstance(HCalendar.MILLIS_IN_HR * 6)); //This is set below!
        header.setParameterId(HEFSTools.DEFAULT_PRECIPITATION_IDENTIFIER_PARAMETER_ID);
        header.setParameterName(HEFSTools.DEFAULT_PRECIPITATION_IDENTIFIER_PARAMETER_ID);
        header.setGeometry(new PointsGeometry(new GeoPoint[]{
            new Wgs1984Point(identifier.getLocationLatitude(), identifier.getLocationLongitude())}));
        return header;
    }

    /**
     * Reads in the processed historical binary file. The data is stored for each julian day, where a julian day is 12Z
     * to 12Z, meaning there are four values per day: 18Z, 0Z, 6Z, and 12Z. The first julian day in the file is 1/1 for
     * the initial year specified in the header. The last day is 12/31 of the last year specified in the header.
     * 
     * @param inputFile
     * @param identifier
     * @return
     * @throws IOException
     */
    public static TimeSeriesArrays readPrecipitationSeries(final File inputFile,
                                                           final LocationAndDataTypeIdentifier identifier) throws IOException
    {
        TimeSeriesArray map = new TimeSeriesArray(prepareMAPHeader(identifier));

        final List<TimeSeriesArray> results = TimeSeriesArraysTools.readDataFromBinFile(map, inputFile);
        if(results.size() != 1)
        {
            throw new IllegalStateException("Did not find exactly one precipitation time series in the file "
                + inputFile.getAbsolutePath() + ".");
        }
        map = results.get(0);

        final TimeSeriesArrays arrays = new TimeSeriesArrays(DefaultTimeSeriesHeader.class, 1);
        arrays.add(map);
        return arrays;
    }
}
