package ohd.hseb.hefs.mefp.sources.historical;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.util.Calendar;
import java.util.GregorianCalendar;
import java.util.TimeZone;

import nl.wldelft.util.timeseries.DefaultTimeSeriesHeader;
import nl.wldelft.util.timeseries.ParameterType;
import nl.wldelft.util.timeseries.SimpleEquidistantTimeStep;
import nl.wldelft.util.timeseries.TimeSeriesArray;
import nl.wldelft.util.timeseries.TimeSeriesArrays;
import ohd.hseb.hefs.pe.tools.HEFSTools;
import ohd.hseb.hefs.pe.tools.LocationAndDataTypeIdentifier;
import ohd.hseb.hefs.utils.datetime.AstronomicalJulianDay;
import ohd.hseb.hefs.utils.datetime.DateTools;
import ohd.hseb.hefs.utils.tools.StreamTools;
import ohd.hseb.util.io.EndianConvertingInputStream;
import ohd.hseb.util.io.EndianConvertingOutputStream;
import ohd.hseb.util.misc.HCalendar;

/**
 * Static methods to read/write Sequential Binary Time Series.
 * 
 * @author alexander.garbarino
 */
public abstract class SequentialBinaryFileTools
{

    public static float MEFP_MISSING_VALUE = -99.0f;

    public static void writeTemperatureSeries(final File outputFile, final TimeSeriesArray timeSeries) throws IOException
    {
        //For first day, take the time series start time and set the hour of the day to 12Z.  If 
        //the new time is before the time series start time, back up a day.  
        final Calendar firstDay = HCalendar.computeCalendarFromMilliseconds(timeSeries.getStartTime());
//REMOVED!  I need to use the first value in the time series as the first value for a day (value 1 of 4).
//        firstDay.set(Calendar.HOUR_OF_DAY, 12);
//        if(firstDay.getTimeInMillis() > timeSeries.getStartTime())
//        {
//            firstDay.add(Calendar.DAY_OF_YEAR, -1);
//        }

        //The last day is the day of the time series end time.
        final Calendar lastDay = HCalendar.computeCalendarFromMilliseconds(timeSeries.getEndTime());

        //Open up the file for output
        final FileOutputStream outputStream = new FileOutputStream(outputFile);
        final EndianConvertingOutputStream stream = new EndianConvertingOutputStream(outputStream);

        try
        {

            //First and last days to astronomical julian day.  Note that because the first data value is 
            //not necessarily at exactly a 12Z time, we need to use a Math.ceil later in order to output
            //to the bin file.
            final AstronomicalJulianDay firstJulianDay = new AstronomicalJulianDay(firstDay);
            final AstronomicalJulianDay lastJulianDay = new AstronomicalJulianDay(lastDay);

            //Output the date info, first.
            stream.writeIntSwap(16);
            stream.writeIntSwap((int)Math.ceil(firstJulianDay.getJulianDay()));
            stream.writeIntSwap((int)Math.floor(lastJulianDay.getJulianDay()));
            stream.writeIntSwap(DateTools.countNumberOfDaysBetweenTwoTimes(firstDay.getTimeInMillis(),
                                                                           lastDay.getTimeInMillis()) + 1);
            stream.writeByte((byte)'D');
            stream.writeByte((byte)'E');
            stream.writeByte((byte)'G');
            stream.writeByte((byte)'C');
            stream.writeIntSwap(16);

            long workingTime;
            int measurementIndex = 0;
            float tmax = Float.MIN_VALUE;
            float tmin = Float.MAX_VALUE;
            float prevTmax = MEFP_MISSING_VALUE;

            //Compute the number of values output per day
            final int numberOfValuesPerDay = (int)(24 * HCalendar.MILLIS_IN_HR / timeSeries.getTimeStep()
                                                                                           .getStepMillis());

            //Prep the measurement index so that it is not before firstDay.  There is a possibility that one
            //(or more?) values in the time series lie before the first day for which values are output. 
//XXX Measurement index points to the first value of the time series.  We assume it is non-missing and use it to start
//day one.  Hence the below code is commented out.  THIS IS A WORK AROUND TO DEAL WITH DIURNAL PATTERN PROBLEMS
//UNTIL A LONG-TERM SOLUTION IS FOUND ... SEE TASK 63 in the mefppe assignments spreadsheet.
//            while(timeSeries.getTime(measurementIndex) < firstDay.getTimeInMillis())
//            {
//                measurementIndex++;
//            }

            //Loop through every day, starting with the firstDay.
            for(long startOfWorkingDayMillis = firstDay.getTimeInMillis(); startOfWorkingDayMillis < lastDay.getTimeInMillis(); startOfWorkingDayMillis += 24 * HCalendar.MILLIS_IN_HR)

            {
                tmax = Float.MIN_VALUE;
                tmin = Float.MAX_VALUE;

                final float values[] = new float[numberOfValuesPerDay];
                boolean oneNotFound = false;

                //Loop over the all the values in the current working day.
                for(int valueIndex = 0; valueIndex < numberOfValuesPerDay; valueIndex++)
                {
                    //If we have more measurements to analyze
                    if(measurementIndex < timeSeries.size())
                    {
                        //Compute the working time.  By using timeSeries startTime and interval to determine the firstDay
                        //hour of day (above), we are guaranteed that the working time will eventually fall on a measurement
                        //time. If it does, then output the value and increase the measurement index.  In theory, the next time
                        //valueIndex increases, it should yield the next measurement time.
                        workingTime = startOfWorkingDayMillis + valueIndex * timeSeries.getTimeStep().getStepMillis();
                        if(timeSeries.getTime(measurementIndex) == workingTime)
                        {
                            values[valueIndex] = timeSeries.getValue(measurementIndex);
                            measurementIndex++;
                        }
                        else
                        {
                            oneNotFound = true;
                        }
                    }

                }

                //Compute tmin and tmax using code found from original EPP3 fortran.
                if(oneNotFound)
                {
                    tmax = MEFP_MISSING_VALUE;
                    tmin = MEFP_MISSING_VALUE;
                }
                else
                {
                    //Fortran code:
                    //TMIN(j) = (9.652*T22(j) - 6.432*T33(j) + 0.480*T44(j))/3.70
                    //TMAX(j) = (10.72*T33(j) - 0.670*T22(j) - 0.800*T44(j))/9.25
                    tmin = (float)((9.652 * values[1] - 6.432 * values[2] + 0.480 * values[3]) / 3.70);
                    tmax = (float)((10.72 * values[2] - 0.670 * values[1] - 0.800 * values[3]) / 9.25);
                }

                //Write output.
                stream.writeIntSwap(8);
                stream.writeFloatSwap(prevTmax);
                stream.writeFloatSwap(tmin);
                stream.writeIntSwap(8);

                prevTmax = tmax;
            }
        }
        catch(final IOException e)
        {
            throw e;
        }
        finally
        {
            stream.close();
        }
    }

    /**
     * This algorithm ASSUMES that the 2nd value in the datacard file is the first value for a julian day. In other
     * words, that second value is the best value to use for 18Z (this is the case for most of the US, so why not?). In
     * reality, we need them to import the data in the appropriate time zone and select the time closes to 18Z... I
     * think (confirm with Limin). Regardless, given that second value is the first value for a julian day, the rest of
     * the time series is output in order.
     * 
     * @param outputFile
     * @param timeSeries
     * @throws IOException
     */
    public static void writePrecipitationSeries(final File outputFile, final TimeSeriesArray timeSeries) throws IOException
    {
        //Compute the start hour for the first day based on the time series start time and making
        //sure the hour is not before 12, because days are 12Z-12Z.  This yields the first hour after
        //12 that lines up with the time series times.
        //
        //XXX The below will only work if the data time series came from a datacard file that was imported
        //in straight GMT.  So the first value in the datacard file is for 6Z.  The first value in the 
        //sequential binary file will be the first value at 12Z or later (the second value in the file, if
        //the file was read in in GMT).
        //
        //This is definitely overkill... it just determines the first hour with data in the time series after
        //12Z
//        Calendar firstDay = HCalendar.computeCalendarFromMilliseconds(timeSeries.getStartTime());
//        int hour = firstDay.get(Calendar.HOUR_OF_DAY);
//        if(hour < 12)
//        {
//            hour += 24;
//        }
//        while(hour >= 12)
//        {
//            hour -= timeSeries.getTimeStep().getStepMillis() / HCalendar.MILLIS_IN_HR;
//        }
//        hour += timeSeries.getTimeStep().getStepMillis() / HCalendar.MILLIS_IN_HR;

        //This applies the same workaround used for temperature: we want to ensure that the second value on a
        //line gets mapped to the first value for a julian day in the file.  So it becomes the hour of the first
        //value.  The last hour, then allows for the last complete day of the last year to be output: its one step
        //prior to hour.
        final Calendar firstDay = HCalendar.computeCalendarFromMilliseconds(timeSeries.getStartTime());
        int firstHour = firstDay.get(Calendar.HOUR_OF_DAY) + 6; //I know precip data is in 6h time steps.
        if(firstHour >= 24)
        {
            firstHour -= 24;
        }
        int lastHour = firstHour - 6;
        if(lastHour < 0)
        {
            lastHour += 24;
        }

        //The first day is January 1 of the year with the hour as computed above.  The last day is 
        //Jan 1 of the year AFTER the end time with hour 0.  Note that this lastDay will produce more
        //data than is necessary, but since all those values are missing, it won't make any difference.
        firstDay.set(firstDay.get(Calendar.YEAR) - 1, GregorianCalendar.DECEMBER, 31, firstHour, 0, 0);
        final Calendar lastDay = HCalendar.computeCalendarFromMilliseconds(timeSeries.getEndTime());
        lastDay.set(lastDay.get(Calendar.YEAR), GregorianCalendar.DECEMBER, 31, lastHour, 0, 0);

        //XXX The last julian day of data in the file may be 12/31 12Z - 1/1/12Z.  In which case, the endTime
        //of the time series will be 1 year AFTER the last julian day's year.  Hence, by adding 1 above, we 
        //windup with a year that is 1 too large in lastDay.  The algorithm below fills in that year with 
        //missings so there is no negative impact.  However, this should be fixed.  Note that a fix will also
        //need to be applied to the reading mechanism.  This error shows up with BLKO2.MAP06 data. 
        //
        //Temperature likely needs a similar change.

        //Open up the file for output
        final FileOutputStream outputStream = new FileOutputStream(outputFile);
        final EndianConvertingOutputStream stream = new EndianConvertingOutputStream(outputStream);

        try
        {
            //Output the date info, first.
            stream.writeIntSwap(16);
            stream.writeIntSwap(firstDay.get(Calendar.YEAR) + 1);
            stream.writeIntSwap(lastDay.get(Calendar.YEAR));
            stream.writeIntSwap(DateTools.countNumberOfDaysBetweenTwoTimes(firstDay.getTimeInMillis(),
                                                                           lastDay.getTimeInMillis()) + 1);
            stream.writeByte((byte)'m');
            stream.writeByte((byte)'m');
            stream.writeByte((byte)'\0');
            stream.writeByte((byte)'\0');
            stream.writeIntSwap(16);

            long workingTime;
            int measurementIndex = 0;

            //Compute the number of values output per day
            final int numberOfValuesOutput = (int)(24 * HCalendar.MILLIS_IN_HR / timeSeries.getTimeStep()
                                                                                           .getStepMillis());

            //Prep the measurement index so that it is not before firstDay.  There is a possibility that one
            //(or more?) values in the time series lie before the first day for which values are output.
            while(timeSeries.getTime(measurementIndex) < firstDay.getTimeInMillis())
            {
                measurementIndex++;
            }

            //Loop through every day, starting with the firstDay.
            for(long startOfWorkingDayMillis = firstDay.getTimeInMillis(); startOfWorkingDayMillis < lastDay.getTimeInMillis(); startOfWorkingDayMillis += 24 * HCalendar.MILLIS_IN_HR)

            {
                stream.writeIntSwap(numberOfValuesOutput * 4);

                //Loop over the number of values to print per day.
                for(int valueIndex = 0; valueIndex < numberOfValuesOutput; valueIndex++)
                {
                    //If we have no more measurements, output missing.
                    if(measurementIndex >= timeSeries.size())
                    {
                        stream.writeFloatSwap(MEFP_MISSING_VALUE);
                    }
                    //Otherwise...
                    else
                    {
                        //XXX This appears to output the 12Z value as the first value for a line.  Is this right?
                        //It might depends on how the data is exported.

                        //Compute the working time.  By using timeSeries startTime and interval to determine the firstDay
                        //hour of day (above), we are guaranteed that the working time will eventually fall on a measurement
                        //time. If it does, then output the value and increase the measurement index.  In theory, the next time
                        //valueIndex increases, it should yield the next measurement time.
                        workingTime = startOfWorkingDayMillis + valueIndex * timeSeries.getTimeStep().getStepMillis();
                        if(timeSeries.getTime(measurementIndex) == workingTime)
                        {
                            if(timeSeries.getValue(measurementIndex) == Float.NaN)
                            {
                                stream.writeFloatSwap(MEFP_MISSING_VALUE);
                            }
                            else
                            {
                                stream.writeFloatSwap(timeSeries.getValue(measurementIndex));
                            }
                            measurementIndex++;
                        }
                        else
                        {
                            stream.writeFloatSwap(MEFP_MISSING_VALUE);
                        }
                    }
                }

                stream.writeIntSwap(numberOfValuesOutput * 4);
            }
        }
        catch(final IOException e)
        {
            throw e;
        }
        finally
        {
            stream.close();
            outputStream.close();
        }
    }

    public static TimeSeriesArrays readSeries(final File inputFile, final LocationAndDataTypeIdentifier identifier) throws IOException
    {
        if(identifier.isPrecipitationDataType())
        {
            return readPrecipitationSeries(inputFile, identifier);
        }
        else if(identifier.isTemperatureDataType())
        {
            return readTemperatureSeries(inputFile, identifier);
        }
        else
        {
            throw new IOException("Unsupported parameter type: " + identifier.getParameterId());
        }
    }

    /**
     * Throws an exception explaining that the given file is in the wrong format.
     * 
     * @param f puts the file name in the exception
     * @throws Exception
     */
    private static void wrongFormat(final File file, final int index) throws IOException
    {
        System.err.println(String.format("File not formatted correctly(%d): %s", index, file.toString()));
    }

    public static TimeSeriesArrays readTemperatureSeries(final File inputFile,
                                                         final LocationAndDataTypeIdentifier identifier) throws IOException
    {
        // Make Time Series.
        DefaultTimeSeriesHeader header = new DefaultTimeSeriesHeader();
        header.setLocationId(identifier.getLocationId());
        header.setLocationName(identifier.getLocationId());
        header.setLocationDescription(identifier.getLocationId());
        header.setParameterType(ParameterType.ACCUMULATIVE);
        header.setForecastTime(Long.MIN_VALUE);
        header.setUnit("DEGC");
        header.setTimeStep(SimpleEquidistantTimeStep.getInstance(HCalendar.MILLIS_IN_HR * 24));

        header.setParameterId(HEFSTools.DEFAULT_TMAX_PARAMETER_ID);
        header.setParameterName(HEFSTools.DEFAULT_TMAX_PARAMETER_ID);
        final TimeSeriesArray tmax = new TimeSeriesArray(header);

        header = new DefaultTimeSeriesHeader(header);
        header.setParameterId(HEFSTools.DEFAULT_TMIN_PARAMETER_ID);
        header.setParameterName(HEFSTools.DEFAULT_TMIN_PARAMETER_ID);
        final TimeSeriesArray tmin = new TimeSeriesArray(header);

        // Open up file for reading.
        final FileInputStream inputStream = new FileInputStream(inputFile);
        final EndianConvertingInputStream stream = new EndianConvertingInputStream(inputStream);

        try
        {
            //Check first value for validity
            if(stream.readIntSwap() != 16)
            {
                wrongFormat(inputFile, -1);
            }

            // Read first / last day.  The firstDay and lastDay point to the END of the julian day.
            final AstronomicalJulianDay firstJulianDay = new AstronomicalJulianDay(stream.readIntSwap());
            final AstronomicalJulianDay lastJulianDay = new AstronomicalJulianDay(stream.readIntSwap());
            final Calendar firstDay = firstJulianDay.toGregorian();
            final Calendar lastDay = lastJulianDay.toGregorian();

            // Number of days between the two dates, inclusive.
            final int numDays = DateTools.countNumberOfDaysBetweenTwoTimes(firstDay.getTimeInMillis(),
                                                                           lastDay.getTimeInMillis()) + 1;
            if(stream.readIntSwap() != numDays)
            {
                wrongFormat(inputFile, -2);
            }

            if(stream.readByte() != (byte)'D' || stream.readByte() != (byte)'E' || stream.readByte() != (byte)'G'
                || stream.readByte() != (byte)'C' || stream.readIntSwap() != 16)
            {
                wrongFormat(inputFile, -3);
            }

            /*
             * // Loop through days. Calendar day = firstJulianDay.toGregorian(); day.set(Calendar.HOUR_OF_DAY, 12); //
             * Always middle of day. // CHPS measures accumulated temperature by the end of the period, instead of the
             * beginning, // so shift the time forward by a day. day.add(Calendar.DAY_OF_YEAR, 1);
             */

            // Loop through each day - read rest of file.
            int readInt = stream.readIntSwap(); // Read first border.
            int dayCount = 0;
            // Compute start time
            firstDay.set(Calendar.HOUR_OF_DAY, 12); // Time periods start at 12h.
            firstDay.set(Calendar.MILLISECOND, 0);
            long time = firstDay.getTimeInMillis();
            final long timeStep = HCalendar.MILLIS_IN_HR * 24;
            while(readInt == 8) // Loop until we misread a border - probably EOF.
            {
                float value = stream.readFloatSwap();
                if(value != MEFP_MISSING_VALUE)
                {
                    tmax.putValue(time, value);
                }

                value = stream.readFloatSwap();
                if(value != MEFP_MISSING_VALUE)
                {
                    tmin.putValue(time, value);
                }

                if(stream.readIntSwap() != 8) // End border.
                {
                    wrongFormat(inputFile, dayCount);
                }

                dayCount++;
                time += timeStep;
                readInt = stream.readIntSwap(); // Start border of next section.
            }

            // 0 is EOF.
            if(readInt != 0 || dayCount != numDays)
            {
                wrongFormat(inputFile, -4);
            }
        }
        finally
        {
            stream.close();
        }

        final TimeSeriesArrays arrays = new TimeSeriesArrays(DefaultTimeSeriesHeader.class, 2);
        arrays.add(tmax);
        arrays.add(tmin);
        return arrays;
    }

    /**
     * Reads in the sequential binary file. The data is stored for each julian day, where a julian day is 12Z to 12Z,
     * meaning there are four values per day: 18Z, 0Z, 6Z, and 12Z. The first julian day in the file is 1/1 for the
     * initial year specified in the header. The last day is 12/31 of the last year specified in the header.
     * 
     * @param inputFile
     * @param identifier
     * @return
     * @throws IOException
     */
    public static TimeSeriesArrays readPrecipitationSeries(final File inputFile,
                                                           final LocationAndDataTypeIdentifier identifier) throws IOException
    {
        // Make Time Series.
        final DefaultTimeSeriesHeader header = new DefaultTimeSeriesHeader();
        header.setLocationId(identifier.getLocationId());
        header.setLocationName(identifier.getLocationId());
        header.setLocationDescription(identifier.getLocationId());
        header.setParameterType(ParameterType.ACCUMULATIVE);
        header.setForecastTime(Long.MIN_VALUE);
        header.setUnit("MM");
        header.setTimeStep(SimpleEquidistantTimeStep.getInstance(HCalendar.MILLIS_IN_HR * 24)); //This is set below!
        header.setParameterId(HEFSTools.DEFAULT_PRECIPITATION_IDENTIFIER_PARAMETER_ID);
        header.setParameterName(HEFSTools.DEFAULT_PRECIPITATION_IDENTIFIER_PARAMETER_ID);
        final TimeSeriesArray map = new TimeSeriesArray(header);

        // Open up file for reading.
        final FileInputStream inputStream = new FileInputStream(inputFile);
        final EndianConvertingInputStream stream = new EndianConvertingInputStream(inputStream);

        try
        {

            if(stream.readIntSwap() != 16)
            {
                wrongFormat(inputFile, -1);
            }

            // Read first / last day.  firstDay will point to the start time of the first julian day in the file, which 
            // ends at 12Z on 1/1 of the first year.  The lastDay will point to 12/30 12Z (start time of the last day) at 12Z.  Note
            //that lastDay is only used to compute the number of days in the file, which is why it points to the start of
            //the day (12/30 at 12Z) and then we add 1 to the numDays computation, below.
            final TimeZone tz = TimeZone.getTimeZone("GMT");
            final Calendar firstDay = Calendar.getInstance(tz);
            firstDay.set(stream.readIntSwap() - 1, GregorianCalendar.DECEMBER, 31, 12, 0, 0); // Read start year.
            firstDay.set(Calendar.MILLISECOND, 0);
            final Calendar lastDay = Calendar.getInstance(tz);
            lastDay.set(stream.readIntSwap(), GregorianCalendar.DECEMBER, 30, 12, 0, 0); // Read end year (inclusive).
            lastDay.set(Calendar.MILLISECOND, 0);

            // Number of days between the two dates.
            final int numDays = DateTools.countNumberOfDaysBetweenTwoTimes(firstDay.getTimeInMillis(),
                                                                           lastDay.getTimeInMillis()) + 1;
            final int sbinNumDays = stream.readIntSwap();
            if(sbinNumDays != numDays)
            {
                wrongFormat(inputFile, -2);
            }

            final String units = "" + (char)stream.readByte() + (char)stream.readByte() + (char)stream.readByte()
                + (char)stream.readByte();
            final int last = stream.readIntSwap();
            if(!units.trim().equalsIgnoreCase("mm") || last != 16)
            {
                wrongFormat(inputFile, -3);
            }

            // Loop through each day - read rest of file.
            int readInt = stream.readIntSwap(); // Read first border.
            final int border = readInt; // Integer that opens/closes each day.
            final int entriesPerDay = border / 4;
            int dayCount = 0;
            long time = firstDay.getTimeInMillis();
            final long timeStep = HCalendar.MILLIS_IN_HR * 24 / entriesPerDay;
            header.setTimeStep(SimpleEquidistantTimeStep.getInstance(timeStep));
            while(readInt == border) // Loop until we misread a border - probably EOF.
            {
                for(int i = 0; i < entriesPerDay; i++)
                {
                    //Add to time before putting the value, because time initially points to the start time of a julian day,
                    //but the data must be stored for the end time of the first step (i.e., 18Z).
                    time += timeStep;
                    final float value = stream.readFloatSwap();
                    if(value != MEFP_MISSING_VALUE)
                    {
                        map.putValue(time, value);
                    }

                }

                if(stream.readIntSwap() != border) // End border.
                {
                    wrongFormat(inputFile, dayCount);
                }

                dayCount++;
                readInt = stream.readIntSwap(); // Start border of next section.
            }

            // 0 is EOF.
            if(readInt != 0 || dayCount != numDays)
            {
                wrongFormat(inputFile, -4);
            }

            final TimeSeriesArrays arrays = new TimeSeriesArrays(DefaultTimeSeriesHeader.class, 1);
            arrays.add(map);
            return arrays;
        }
        finally
        {
            StreamTools.closeStream(stream);
        }
    }
}
