package ohd.hseb.hefs.mefp.sources.plugin;

import java.io.File;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;

import nl.wldelft.util.timeseries.TimeSeriesArray;
import nl.wldelft.util.timeseries.TimeSeriesArrays;
import ohd.hseb.graphgen.utils.GraphGenCalculatorTools;
import ohd.hseb.hefs.mefp.sources.AbstractMEFPSourceDataHandler;
import ohd.hseb.hefs.pe.tools.LocationAndDataTypeIdentifier;
import ohd.hseb.hefs.pe.tools.TimeSeriesSorter;
import ohd.hseb.hefs.utils.tools.FileTools;
import ohd.hseb.hefs.utils.tools.ListTools;
import ohd.hseb.hefs.utils.tools.ParameterId;
import ohd.hseb.hefs.utils.tsarrays.TimeSeriesArraysTools;
import ohd.hseb.util.misc.SegmentedLine;

import org.apache.logging.log4j.Logger;
import org.apache.logging.log4j.LogManager;

import com.google.common.collect.ArrayListMultimap;
import com.google.common.collect.Lists;
import com.google.common.collect.Multimap;

/**
 * Plug-in handler reads in files within the base directory under a subdirectory called "pluginData" and within another
 * subdirectory matching {@link #_pluginId}. File names are acquired calling
 * {@link PluginDataHandler#getPluginDataFile(File, String, boolean)}. For precip data, only time series with
 * parameterId {@link ParameterId#FMAP} will be used, whereas for temperature both {@link ParameterId#TFMN} and
 * {@link ParameterId#TFMX} will be read. The attribute {@link #_identifierToEnsembleMap} stores the entire ensemble
 * found, while {@link #_identifierToEnsembleMeanMap} stores the ensemble mean. The method
 * {@link #getAllLoadedForecastTimeSeries()} and {@link #getLoadedForecastTimeSeries(LocationAndDataTypeIdentifier)}
 * both only return the ensemble mean, whereas {@link #getAllLoadedForecastTimeSeriesForDiagnostics()} returns
 * everything.
 * 
 * @author hankherr
 */
public class PluginDataHandler extends AbstractMEFPSourceDataHandler
{
    private static final Logger LOG = LogManager.getLogger(PluginDataHandler.class);

    public static final String ASSUMED_ENSEMBLE_ID = "Plugin";

    /**
     * The id of the plugin, which dictates directories to use.
     */
    private final String _pluginId;

    /**
     * Stores the location of the directory specific to this plug-in.
     */
    private File _preparedDataFileDirectory = null;

    /**
     * Stores the reforecast ensembles.
     */
    private final Multimap<LocationAndDataTypeIdentifier, TimeSeriesArrays> _identifierToEnsembleMap = ArrayListMultimap.create();;

    /**
     * Stores the ensemble means.
     */
    private final Multimap<LocationAndDataTypeIdentifier, TimeSeriesArray> _identifierToEnsembleMeanMap = ArrayListMultimap.create();;

    public PluginDataHandler(final String pluginId)
    {
        _pluginId = pluginId;
    }

    public PluginDataHandler(final File baseDirectory, final String pluginId)
    {
        this(pluginId);
        setDataHandlerBaseDirectory(baseDirectory);
    }

    /**
     * @return {@link Collection} of time series to be displayed as diagnostics.
     */
    public Collection<TimeSeriesArray> getAllLoadedForecastTimeSeriesForDiagnostics()
    {
        return ListTools.concat(TimeSeriesArraysTools.convertTimeSeriesArraysToList(TimeSeriesArraysTools.convertTimeSeriesArraysCollections(_identifierToEnsembleMap.values())),
                                _identifierToEnsembleMeanMap.values());
    }

    /**
     * @return The number of files that pass the check {@link #isPluginDataFile(File)} within the directory returned by
     *         {@link #getPreparedDataFilesDirectory()}.
     */
    public int getNumberOfExistingPluginDataFilesForSource()
    {
        final File[] files = getPreparedDataFilesDirectory().listFiles();
        int count = 0;
        if(files != null)
        {
            for(final File file: files)
            {
                if(isPluginDataFile(file))
                {
                    count++;
                }
            }
        }
        return count;
    }

    @Override
    public void setDataHandlerBaseDirectory(final File directory)
    {
        super.setDataHandlerBaseDirectory(directory);
        _preparedDataFileDirectory = new File(directory.getAbsolutePath() + "/pluginData/" + _pluginId);
    }

    @Override
    public File getPreparedDataFilesDirectory()
    {
        return _preparedDataFileDirectory;
    }

    @Override
    public void loadOriginalTimeSeries(final List<LocationAndDataTypeIdentifier> identifiers) throws Exception
    {
        throw new Exception("GriddedPluginDataHandler instances do not load the original gridded time series.");
    }

    @Override
    public void loadPreparedTimeSeries(final List<LocationAndDataTypeIdentifier> identifiers) throws Exception
    {
        clearLoadedTimeSeries();

        LOG.info("Loading " + _pluginId + " reforecasts for "
            + LocationAndDataTypeIdentifier.createListOfIdentifierStrings(identifiers) + "...");
        for(final LocationAndDataTypeIdentifier identifier: identifiers)
        {
            //Read the time series.
            final List<File> forecastFiles = generateListOfPreparedDataFiles(identifier);
            final TimeSeriesArrays tss = TimeSeriesArraysTools.readFromFiles(forecastFiles);
            if(tss == null)
            {
                continue; //No ts file found.
            }

            //The time series within either be single valued or ensembles.  We will force single-valued to have ensemble ids to simplify
            //things and make all ensemble ids identical.
            TimeSeriesArraysTools.setAllEnsembleIds(tss, ASSUMED_ENSEMBLE_ID);

            //Get the reforecast (possibly) ensembles for all required data types.
            final TimeSeriesSorter sorter = new TimeSeriesSorter(TimeSeriesArraysTools.convertTimeSeriesArraysToList(tss));
            final List<TimeSeriesArrays> ensembles = new ArrayList<>();
            if(identifier.isPrecipitationDataType())
            {
                ensembles.addAll(TimeSeriesArraysTools.createMapOfForecastTimeToTimeSeries(sorter.restrictViewToParameters(ParameterId.FMAP))
                                                      .values());
            }
            else
            {
                ensembles.addAll(TimeSeriesArraysTools.createMapOfForecastTimeToTimeSeries(sorter.restrictViewToParameters(ParameterId.TFMN))
                                                      .values());
                ensembles.addAll(TimeSeriesArraysTools.createMapOfForecastTimeToTimeSeries(sorter.restrictViewToParameters(ParameterId.TFMX))
                                                      .values());
            }

            //Put the ensembles into the map and compute the means.
            _identifierToEnsembleMap.putAll(identifier, ensembles);
            _identifierToEnsembleMeanMap.putAll(identifier,
                                                GraphGenCalculatorTools.computeEnsembleMeanTimeSeries(ensembles));
        }
        LOG.info("Done loading " + _pluginId + " reforecasts for "
            + LocationAndDataTypeIdentifier.createListOfIdentifierStrings(identifiers) + "...");
    }

    @Override
    public void prepareDataFiles() throws Exception
    {
        throw new Exception("PluginDataHandler does not prepare data.  "
            + "The files are prepared in the appropriate panel within the Setup Panel.");
    }

    @Override
    public Collection<TimeSeriesArray> getLoadedForecastTimeSeries(final LocationAndDataTypeIdentifier identifier)
    {
        return _identifierToEnsembleMeanMap.get(identifier);
    }

    @Override
    public Collection<TimeSeriesArray> getAllLoadedForecastTimeSeries()
    {
        return _identifierToEnsembleMeanMap.values();
    }

    @Override
    public void initialize() throws Exception
    {
    }

    @Override
    public List<File> generateListOfPreparedDataFiles(final LocationAndDataTypeIdentifier translatedIdentifier)
    {
        return Lists.newArrayList(PluginDataHandler.getPluginDataFile(_preparedDataFileDirectory,
                                                                      translatedIdentifier.getLocationId(),
                                                                      translatedIdentifier.isPrecipitationDataType()));
    }

    @Override
    public void clearLoadedTimeSeries()
    {
        _identifierToEnsembleMap.clear();
        _identifierToEnsembleMeanMap.clear();
    }

    @Override
    public Collection<LocationAndDataTypeIdentifier> getIdentifiersWithData()
    {
        return _identifierToEnsembleMeanMap.keys();
    }

    @Override
    public boolean havePreparedDataFilesBeenCreatedAlready(final LocationAndDataTypeIdentifier translatedIdentifier)
    {
        for(final File file: generateListOfPreparedDataFiles(translatedIdentifier))
        {
            if(!file.exists())
            {
                return false;
            }
        }
        return true;
    }

    @Override
    public boolean arePreparedDataFilesUpToDate(final LocationAndDataTypeIdentifier translatedIdentifier)
    {
        // No up-to-date checking needed.
        return true;
    }

    /**
     * @return True if the name of the file provided matches the pattern generated by
     *         {@link #getPluginDataFile(File, String, boolean)}, meaning its a plausible reforecast data file.
     */
    public static boolean isPluginDataFile(final File file)
    {
        final SegmentedLine segLine = new SegmentedLine(file.getName(), ".", SegmentedLine.MODE_ALLOW_EMPTY_SEGS);
        if(segLine.getNumberOfSegments() != 4)
        {
            return false;
        }
        if(!segLine.getSegment(1).equals("precipitation") && !segLine.getSegment(1).equals("temperature"))
        {
            return false;
        }
        if(!segLine.getSegment(2).equals("reforecasts"))
        {
            return false;
        }
        if(!segLine.getSegment(3).equals("fi") && !segLine.getSegment(3).equals("xml"))
        {
            return false;
        }
        return true;
    }

    /**
     * File name will be [base directory]/[locationId].[precipitation or temperature].reforecasts.fi.
     * 
     * @param baseDirectory Directory that contains the file to look for.
     * @param locationId Location id to look for.
     * @param precipitation True for precipitation, false for temperature.
     * @return The name of the file that contains the reforecasts under that base directory.
     */
    public static File getPluginDataFile(final File baseDirectory, final String locationId, final boolean precipitation)
    {
        String dataType = "precipitation";
        if(!precipitation)
        {
            dataType = "temperature";
        }

        return FileTools.newFile(baseDirectory, locationId + "." + dataType + ".reforecasts.fi");
    }

}
