package ohd.hseb.hefs.mefp.sources.plugin;

import java.io.File;

import ohd.hseb.hefs.mefp.models.parameters.MEFPSourceModelParameters;
import ohd.hseb.hefs.mefp.pe.core.MEFPParameterEstimatorRunInfo;
import ohd.hseb.hefs.mefp.pe.estimation.GenericMEFPSourceControlOptions;
import ohd.hseb.hefs.mefp.pe.estimation.MEFPPrecipitationSourceControlOptions;
import ohd.hseb.hefs.mefp.sources.AbstractMEFPForecastSource;
import ohd.hseb.hefs.mefp.sources.MEFPForecastSource;
import ohd.hseb.hefs.mefp.sources.MEFPSourceControlOptions;
import ohd.hseb.hefs.pe.core.ParameterEstimatorRunInfo;
import ohd.hseb.hefs.pe.core.ParameterEstimatorStepProcessor;
import ohd.hseb.hefs.pe.model.AlgorithmModelParameters;
import ohd.hseb.hefs.pe.sources.ForecastSourceDefinitionXMLHandler;
import ohd.hseb.hefs.pe.sources.SourceDataHandler;
import ohd.hseb.hefs.pe.tools.LocationAndDataTypeIdentifier;
import ohd.hseb.hefs.utils.notify.NoticePoster;
import ohd.hseb.hefs.utils.tools.ParameterId.Type;
import ohd.hseb.hefs.utils.xml.vars.XMLInteger;

/**
 * {@link MEFPForecastSource} used for all MEFPPE plug-in forecast sources.
 * 
 * @author hankherr
 */
public class PluginForecastSource extends AbstractMEFPForecastSource
{

    /**
     * The steps associated with this source. This can only be specified by XML reading and, for that, call
     * {@link #getSourceDefinitionHandler(ParameterEstimatorRunInfo)}.
     */
    private ReforecastPreparationSteps _reforecastPrepSteps;

    /**
     * The default number of forecast days to use as an estimation option. -1 is the default value so the control
     * options use their defualt, which is 0.
     */
    private final XMLInteger _defaultNumberOfForecastDays = new XMLInteger("defaultNumberOfForecastDays");

    /**
     * The default initial year of reforecast data. By default, it is {@link Integer#MIN_VALUE} so that the control
     * option default, 1979, is used.
     */
    private final XMLInteger _defaultInitialYear = new XMLInteger("defaultInitialYear");

    /**
     * The default last year of reforecast data. By default, it is {@link Integer#MAX_VALUE} so that the control option
     * default, 2010, is used.
     */
    private final XMLInteger _defaultLastYear = new XMLInteger("defaultLastYear");

    public ReforecastPreparationSteps getReforecastPrepSteps()
    {
        return _reforecastPrepSteps;
    }

    @Override
    public MEFPSourceControlOptions createControlOptions(final Type type)
    {
        switch(type)
        {
            case TEMPERATURE:
                return new GenericMEFPSourceControlOptions(getSourceId().toLowerCase() + "TemperatureControlOptions",
                                                           this);
            case PRECIPITATION:
                return new MEFPPrecipitationSourceControlOptions(getSourceId().toLowerCase()
                    + "PrecipitationControlOptions", this);
            default:
                throw new IllegalArgumentException(type + "not supported.");
        }
    }

    @Override
    public String getName()
    {
        return getSourceId() + " Forecasts";
    }

    @Override
    protected void setSourceDataHandler(final SourceDataHandler handler)
    {
        if(!(handler instanceof PluginDataHandler))
        {
            throw new IllegalArgumentException("The SourceDataHandler is not a PluginDataHandler.");
        }
        super.setSourceDataHandler(handler);
    }

    /**
     * @return a {@link ReforecastPreparationSteps} object given the system files directory in the provided run-time
     *         information. If the provided run info was null, then the return value of this method is null.
     */
    @Override
    public ForecastSourceDefinitionXMLHandler getSourceDefinitionHandler(final ParameterEstimatorRunInfo runInfo)
    {
        if(runInfo == null)
        {
            return null;
        }
        if(_reforecastPrepSteps == null)
        {
            _reforecastPrepSteps = new ReforecastPreparationSteps(runInfo.getSystemFilesDirectory(),
                                                                  _defaultNumberOfForecastDays,
                                                                  _defaultInitialYear,
                                                                  _defaultLastYear);
        }
        return _reforecastPrepSteps;
    }

    @Override
    public MEFPSourceModelParameters getSourceModelParameters(final LocationAndDataTypeIdentifier identifier,
                                                              final AlgorithmModelParameters algorithmParameters)
    {
        final MEFPSourceModelParameters parameters = new MEFPSourceModelParameters(identifier,
                                                                                   this,
                                                                                   algorithmParameters,
                                                                                   false);
        return parameters;
    }

    @Override
    public void initializeSourceDataHandler(final File baseDirectory, final NoticePoster poster) throws Exception
    {
        this.setSourceDataHandler(new PluginDataHandler(baseDirectory, getSourceId()));
    }

    @Override
    public ParameterEstimatorStepProcessor getSourceStepProcessor(final ParameterEstimatorRunInfo runInfo)
    {
        return new PluginPEStepProcessor((MEFPParameterEstimatorRunInfo)runInfo, getSourceId());
    }

    @Override
    public int getDefaultNumberOfForecastDays()
    {
        if(_defaultNumberOfForecastDays.get() == null)
        {
            return -1;
        }
        return _defaultNumberOfForecastDays.get();
    }

    @Override
    public int getDefaultInitialYear()
    {
        if(_defaultInitialYear.get() == null)
        {
            return Integer.MAX_VALUE;
        }
        return _defaultInitialYear.get();
    }

    @Override
    public int getDefaultLastYear()
    {
        if(_defaultLastYear.get() == null)
        {
            return Integer.MIN_VALUE;
        }
        return _defaultLastYear.get();
    }

}
