package ohd.hseb.hefs.mefp.sources.plugin;

import java.awt.BorderLayout;
import java.awt.Cursor;
import java.awt.FlowLayout;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.Insets;
import java.awt.event.ActionEvent;
import java.util.HashMap;

import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.SwingUtilities;

import ohd.hseb.hefs.pe.core.ParameterEstimatorRunInfo;
import ohd.hseb.hefs.pe.notice.StepStatusRefreshAllNotice;
import ohd.hseb.hefs.utils.gui.tools.ScrollableJPanel;
import ohd.hseb.hefs.utils.gui.tools.SelfListeningButton;
import ohd.hseb.hefs.utils.gui.tools.SwingTools;
import ohd.hseb.hefs.utils.jobs.GenericJob;
import ohd.hseb.hefs.utils.jobs.JobListener;
import ohd.hseb.util.misc.HString;

import org.apache.logging.log4j.Logger;
import org.apache.logging.log4j.LogManager;

import com.google.common.eventbus.Subscribe;

/**
 * Displays {@link ReforecastPreparationStepJobMonitorPanel} instances for each
 * {@link ReforecastPreparationStepProcessor} provided via {@link PluginForecastSource#getReforecastPrepSteps()} for the
 * {@link PluginForecastSource} provided to the constructor. Also provided are two self-listening buttons:
 * {@link #_runButton} and {@link #_updateStatusButton}. It listens for events posted through {@link #_runInfo} as
 * specified by the implements list of subscribers.
 * 
 * @author hankherr
 */
@SuppressWarnings("serial")
public class PluginSetupPanel extends JPanel implements
UpdateAllPluginReforecastPreparationStepStatusNotice.Subscriber, ReforecastPreparationStepRunningNotice.Subscriber,
ReforecastPreparationStepCompletedNotice.Subscriber
{
    private static final Logger LOG = LogManager.getLogger(PluginSetupPanel.class);
    private final ParameterEstimatorRunInfo _runInfo;
    private final PluginForecastSource _source;
    private final ReforecastPreparationStepsProcessor _processor;
    private final HashMap<ReforecastPreparationStepProcessor, ReforecastPreparationStepJobMonitorPanel> _processorToPanelMap = new HashMap<>();
    private SelfListeningButton _runButton;
    private SelfListeningButton _updateStatusButton;

    /**
     * @param runInfo The run-time information associated with this PE.
     * @param source The source associated with this plug-in.
     */
    public PluginSetupPanel(final ParameterEstimatorRunInfo runInfo, final PluginForecastSource source)
    {
        _runInfo = runInfo;
        _runInfo.register(this);
        _source = source;
        _processor = new ReforecastPreparationStepsProcessor(getRunInfo(), source.getReforecastPrepSteps(), source);
        createDisplay();
        updateAllStatus(false);
        _runInfo.post(new ReforecastPreparationStepCompletedNotice(null));
    }

    /**
     * Uses {@link #_processor} to run all of the steps. It first queries if the run is a continuation or a re-run.
     */
    private void runSteps()
    {
        final int option = JOptionPane.showOptionDialog(SwingTools.getGlobalDialogParent(this),
                                                        "<html>Do you want to continue each step where the last run left off or rerun all steps from the beginning.<br><br>"
                                                            + "<b>By clicking Rerun, ALL previously acquired results are discarded before executing the steps!</b><br><br>"
                                                            + "NOTE: If performance of CHPS is slow when preparing the reforecasts, it is recommended you<br>"
                                                            + "shutdown CHPS, remove the log file log.txt, delete the localDataStore, and try again.</html>",
                                                        "Continue or Rerun?",
                                                        JOptionPane.YES_NO_CANCEL_OPTION,
                                                        JOptionPane.QUESTION_MESSAGE,
                                                        null,
                                                        new String[]{"Continue", "Rerun", "Cancel"}, //Continue is yes and Rerun is no.
                                                        "Continue");
        if(option == JOptionPane.CANCEL_OPTION)
        {
            return;
        }

        //Rerun selected...
        if(option == JOptionPane.NO_OPTION)
        {
            final int count = ((PluginDataHandler)_source.getSourceDataHandler()).getNumberOfExistingPluginDataFilesForSource();
            if(count > 0)
            {
                final int confirmOption = JOptionPane.showOptionDialog(SwingTools.getGlobalDialogParent(this),
                                                                       "The MEFPPE run area data directory for source "
                                                                           + _source.getSourceId()
                                                                           + " contains "
                                                                           + count
                                                                           + " files that will be removed.\n"
                                                                           + "Are you sure you want to rerun all steps from the beginning?",
                                                                       "Confirm Rerun?",
                                                                       JOptionPane.OK_CANCEL_OPTION,
                                                                       JOptionPane.QUESTION_MESSAGE,
                                                                       null,
                                                                       new String[]{"Rerun", "Cancel"}, //OK is rerun
                                                                       "Cancel");
                if(confirmOption == 1) //For some reason JOptionPane.CANCEL_OPTION does not work.  However, I know 1 is cancel.
                {
                    return;
                }
            }
            _processor.clearProgress();
        }

        _processor.addListener(new JobListener()
        {
            @Override
            public void processJobFailure(final Exception exc, final GenericJob theJob, final boolean displayMessage)
            {
                LOG.error("Error preparing reforecasts:" + exc.getMessage());
                JOptionPane.showMessageDialog(PluginSetupPanel.this,
                                              "Error preparing reforecasts:\n\n"
                                                  + HString.wordWrap(exc.getMessage(), 120),
                                              "Error Preparing Reforecasts for Source " + _source.getSourceId() + "!",
                                              JOptionPane.ERROR_MESSAGE);
                updateAllStatus(false);//Its unclear to me why this is needed when the StepsProcessor should fire the UpdateAll*Notice, but it is needed.
                _processor.removeListener(this);
            }

            @Override
            public void processSuccessfulJobCompletion(final GenericJob theJob)
            {
                LOG.error("Successfully prepared reforeasts!");
                JOptionPane.showMessageDialog(PluginSetupPanel.this,
                                              "Successfully prepared reforeasts!",
                                              "Successfully Prepared Reforecasts for Source " + _source.getSourceId(),
                                              JOptionPane.INFORMATION_MESSAGE);
                updateAllStatus(false);//Its unclear to me why this is needed when the StepsProcessor should fire the UpdateAll*Notice, but it is needed.
                _processor.removeListener(this);
            }
        });

        _processor.startJob();
    }

    /**
     * Call to update the status of all displayed {@link ReforecastPreparationStepJobMonitorPanel} instances. The update
     * occurs by calling the update within each processor.
     * 
     * @param fullUpdate True to do a full determination, which may involve SFTPing to get a new list of available
     *            files. False to do it quick and dirty based only on locally, quickly available information.
     */
    private void updateAllStatus(final boolean fullUpdate)
    {
        PluginSetupPanel.this.setCursor(Cursor.getPredefinedCursor(Cursor.WAIT_CURSOR));
        for(final ReforecastPreparationStepInstructions instructions: _source.getReforecastPrepSteps().getSteps())
        {
            _processor.getProcessor(instructions).updateJobMonitorStatus(fullUpdate);
        }
        PluginSetupPanel.this.setCursor(Cursor.getDefaultCursor());

    }

    /**
     * Creates the subpanels for each processor and adds the other needed crap.
     */
    private void createDisplay()
    {
        final JPanel fullPanel = new ScrollableJPanel(true, false);
        fullPanel.setLayout(new GridBagLayout());
        final GridBagConstraints constraints = SwingTools.returnGridBagConstraints(0,
                                                                                   0,
                                                                                   1,
                                                                                   1,
                                                                                   1,
                                                                                   0,
                                                                                   GridBagConstraints.NORTHWEST,
                                                                                   GridBagConstraints.BOTH,
                                                                                   new Insets(0, 0, 0, 0),
                                                                                   0,
                                                                                   0);

        //Add each panel to full panel.
        for(final ReforecastPreparationStepInstructions instructions: _source.getReforecastPrepSteps().getSteps())
        {
            final ReforecastPreparationStepProcessor stepProcessor = _processor.getProcessor(instructions);
            stepProcessor.setSource(_source);
            final ReforecastPreparationStepJobMonitorPanel subPanel = new ReforecastPreparationStepJobMonitorPanel(getRunInfo(),
                                                                                                                   stepProcessor);
            _processorToPanelMap.put(stepProcessor, subPanel);

            fullPanel.add(subPanel, constraints);
            constraints.gridy++;
        }
        constraints.weighty = 1;
        fullPanel.add(new JPanel(), constraints);

        //Put full panel in a scroll pane.
        final JScrollPane scrollPane = new JScrollPane(fullPanel);
        scrollPane.setHorizontalScrollBarPolicy(JScrollPane.HORIZONTAL_SCROLLBAR_NEVER);
        scrollPane.setVerticalScrollBarPolicy(JScrollPane.VERTICAL_SCROLLBAR_ALWAYS);

        //RUN BUTTON WITH ASSOCIATED ACTION!
        _runButton = new SelfListeningButton("Prepare Reforecasts")
        {
            @Override
            public void actionPerformed(final ActionEvent e)
            {
                runSteps();
            }
        };

        //UPDATE STATUS BUTTON WITH ASSOCIATED ACTION!
        _updateStatusButton = new SelfListeningButton("Update Progress Status")
        {
            @Override
            public void actionPerformed(final ActionEvent e)
            {
                updateAllStatus(true);

                //Make sure run buttons are redetermined.  If this button is clicked, then nothing can be running.
                //Hence, I should be able to post a run-complete notice without it causing issues.  This wil force all
                //monitor panels to redetermine the status of their run buttons.  This cannot be done within
                //updateAllStatus(...) because it would cause issues with the buttons redrawing while steps are still
                //processing.
                getRunInfo().post(new ReforecastPreparationStepCompletedNotice(null));
            }
        };

        this.setLayout(new BorderLayout());
        add(SwingTools.wrapInFlowLayout(_updateStatusButton, FlowLayout.CENTER), BorderLayout.NORTH);
        add(scrollPane, BorderLayout.CENTER);
        add(SwingTools.wrapInFlowLayout(_runButton, FlowLayout.CENTER), BorderLayout.SOUTH);
    }

    private ParameterEstimatorRunInfo getRunInfo()
    {
        return _runInfo;
    }

    /**
     * Calls {@link #updateAllStatus(boolean)} with false, so that it is a quick and dirty update.
     */
    @Override
    @Subscribe
    public void reactToUpdateAllPluginReforecastPreparationStepNotice(final UpdateAllPluginReforecastPreparationStepStatusNotice notice)
    {
        SwingUtilities.invokeLater(new Runnable()
        {
            @Override
            public void run()
            {
                getRunInfo().post(new StepStatusRefreshAllNotice(this));
                updateAllStatus(false);
            }
        });
    }

    @Override
    @Subscribe
    public void reactToReforecastPreparationStepCompleted(final ReforecastPreparationStepCompletedNotice evt)
    {
        _updateStatusButton.setEnabled(true);
        _runButton.setEnabled(true);
        setCursor(Cursor.getDefaultCursor());
    }

    @Override
    @Subscribe
    public void reactToReforecastPreparationStepRunning(final ReforecastPreparationStepRunningNotice evt)
    {
        _updateStatusButton.setEnabled(false);
        _runButton.setEnabled(false);
        setCursor(Cursor.getPredefinedCursor(Cursor.WAIT_CURSOR));
    }

}
