package ohd.hseb.hefs.mefp.sources.plugin;

import java.io.File;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;

import ohd.hseb.hefs.utils.log4j.LoggingTools;
import ohd.hseb.hefs.utils.tools.ListTools;
import ohd.hseb.hefs.utils.xml.CollectionXMLReader;
import ohd.hseb.hefs.utils.xml.CollectionXMLWriter;
import ohd.hseb.hefs.utils.xml.XMLReadable;
import ohd.hseb.hefs.utils.xml.XMLReader;
import ohd.hseb.hefs.utils.xml.XMLReaderException;
import ohd.hseb.hefs.utils.xml.XMLReaderFactory;
import ohd.hseb.hefs.utils.xml.XMLTools;
import ohd.hseb.hefs.utils.xml.XMLWritable;
import ohd.hseb.hefs.utils.xml.XMLWriter;
import ohd.hseb.hefs.utils.xml.vars.XMLFile;
import ohd.hseb.hefs.utils.xml.vars.XMLInteger;

import org.apache.logging.log4j.Logger;
import org.apache.logging.log4j.LogManager;

import com.google.common.collect.Lists;

/**
 * Stores a list of (unique; no repeats allowed) files that have already been processed.
 * 
 * @author Hank.Herr
 */
public class ProcessedFilesList implements XMLReadable, XMLWritable
{
    private static final Logger LOG = LogManager.getLogger(ProcessedFilesList.class);

    /**
     * The number of files that can be processed.
     */
    private final XMLInteger _expectedNumberOfFilesTotal = new XMLInteger("expectedNumberOfFiles", 0);

    /**
     * The list of files currently processed. This list must not be exposed to outside callers; rather only the methods
     * herein should be called to interact with it.
     */
    private final List<XMLFile> _processedFilesList = new ArrayList<>();

    /**
     * This is *NOT* output to XML but, rather, is tied to the method {@link #writeToSystemFile()}.<br>
     * Can this be placed elsewhere? XXX
     */
    private File _outputSystemFile = null;

    /**
     * Check the current {@link #_processedFilesList} and remove any files that are not part of the passed in,
     * presumably updated, list of files to process.
     * 
     * @param filesToProcess The complete list of files that must be acquired and processed through the CHPS
     *            workflow(s).
     */
    public void updateProcessedFilesListAgainstFilesToProcess(final Collection<File> filesToProcess)
    {
        for(int i = _processedFilesList.size() - 1; i >= 0; i--)
        {
            final XMLFile file = _processedFilesList.get(i);
            if(!filesToProcess.contains(file.get()))
            {
                _processedFilesList.remove(file);
            }
        }
        setExpectedNumberOfFiles(filesToProcess.size());
    }

    public int getExpectedNumberOfFiles()
    {
        return _expectedNumberOfFilesTotal.get();
    }

    public int getNumberOfProcessedFiles()
    {
        return _processedFilesList.size();
    }

    public void setExpectedNumberOfFiles(final int n)
    {
        _expectedNumberOfFilesTotal.set(n);
    }

    public boolean containsFile(final File file)
    {
        return _processedFilesList.contains(new XMLFile(getListElementTagName(), file, false));
    }

    public void addProcessedFile(final File file)
    {
        if(!containsFile(file))
        {
            _processedFilesList.add(new XMLFile(getListElementTagName(), file, false));
        }
    }

    /**
     * @return True if {@link #getExpectedNumberOfFiles()} equals {@link #getNumberOfProcessedFiles()} AND the number of
     *         expected files is positive! If its zero, then we assume that the status cannot be acquired or has not yet
     *         been determined.
     */
    public boolean haveAllFilesBeenProcessed()
    {
        return (getExpectedNumberOfFiles() == getNumberOfProcessedFiles()) && (getExpectedNumberOfFiles() > 0);
    }

    /**
     * Removes all repeat {@link XMLFile} instance from within {@link #_processedFilesList} making use of
     * {@link ListTools#addAllUniqueItemsToList(List, Collection)}.
     */
    public void removeRepeats()
    {
        final List<XMLFile> toAdd = Lists.newArrayList(_processedFilesList);
        _processedFilesList.clear();
        ListTools.addAllUniqueItemsToList(_processedFilesList, toAdd);
    }

    public void clearProcessedFiles()
    {
        _processedFilesList.clear();
    }

    private String getListElementTagName()
    {
        return "file";
    }

    /**
     * If set, then whenever {@link #writeToSystemFile()} is called, the file {@link #_outputSystemFile} will be updated
     * with the current list.
     * 
     * @param outputFile The .xml/.fi file to update with a list of the files already processed.
     */
    public void setOutputSystemFile(final File outputFile)
    {
        _outputSystemFile = outputFile;
    }

    /**
     * Writes this set of parameters to the supplied {@link #_outputSystemFile}.
     */
    public void writeToSystemFile()
    {
        if(_outputSystemFile != null)
        {
            try
            {
                XMLTools.writeXMLFileFromXMLWriter(_outputSystemFile, this, true);
            }
            catch(final Exception e)
            {
                LoggingTools.outputStackTraceAsDebug(LOG, e);
                LOG.error("Unable to update processed files list in file " + _outputSystemFile + ": " + e.getMessage());
            }
        }
    }

    public String getXMLTagName()
    {
        return "processedFiles";
    }

    @Override
    public XMLWriter getWriter()
    {
        final CollectionXMLWriter writer = new CollectionXMLWriter(getXMLTagName(), _processedFilesList);
        writer.addAttribute(_expectedNumberOfFilesTotal, true);
        return writer;
    }

    /**
     * The reader {@link XMLReader#finalizeReading()} method will remove any repeat files after reading.
     */
    @Override
    public XMLReader getReader()
    {
        final CollectionXMLReader<XMLFile> reader = new CollectionXMLReader(getXMLTagName(),
                                                                            _processedFilesList,
                                                                            new XMLReaderFactory<XMLFile>()
                                                                            {

                                                                                @Override
                                                                                public XMLFile get()
                                                                                {
                                                                                    return new XMLFile(getListElementTagName(),
                                                                                                       false);
                                                                                }
                                                                            })
        {
            @Override
            public void finalizeReading() throws XMLReaderException
            {
                super.finalizeReading();
                removeRepeats();
            }
        };
        reader.addAttribute(_expectedNumberOfFilesTotal, true);
        return reader;
    }

}
