package ohd.hseb.hefs.mefp.sources.plugin;

import java.awt.BorderLayout;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.Insets;
import java.awt.event.ActionEvent;

import javax.swing.BorderFactory;
import javax.swing.JButton;
import javax.swing.JLabel;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JProgressBar;
import javax.swing.JTextArea;
import javax.swing.JToolBar;
import javax.swing.SwingUtilities;

import ohd.hseb.hefs.pe.core.ParameterEstimatorRunInfo;
import ohd.hseb.hefs.utils.gui.tools.HSwingFactory;
import ohd.hseb.hefs.utils.gui.tools.SelfListeningButton;
import ohd.hseb.hefs.utils.gui.tools.SwingTools;
import ohd.hseb.hefs.utils.jobs.GenericJob;
import ohd.hseb.hefs.utils.jobs.JobListener;
import ohd.hseb.hefs.utils.jobs.JobMonitor;
import ohd.hseb.hefs.utils.jobs.JobMonitorAttr;
import ohd.hseb.hefs.utils.tools.IconTools;

import com.google.common.eventbus.Subscribe;

/**
 * {@link JPanel} that displays the status for a single {@link ReforecastPreparationStepProcessor} and provides a run
 * button. This listens for {@link ReforecastPreparationStepRunningNotice} and
 * {@link ReforecastPreparationStepCompletedNotice} notices in order to determine the state of the run button.
 * 
 * @author hankherr
 */
@SuppressWarnings("serial")
public class ReforecastPreparationStepJobMonitorPanel extends JPanel implements JobMonitor,
ReforecastPreparationStepRunningNotice.Subscriber, ReforecastPreparationStepCompletedNotice.Subscriber
{
    private final ParameterEstimatorRunInfo _runInfo;
    private final ReforecastPreparationStepProcessor _stepProcessor;

    /**
     * The progress bar for the currently running job.
     */
    private JProgressBar _progressBar;

    /**
     * The note label above the progress bar.
     */
    private JLabel _noteLabel;

    /**
     * The icon, name, and tooltip for this button will change depending on
     * {@link ReforecastPreparationStepRunningNotice} and {@link ReforecastPreparationStepCompletedNotice} notices
     * posted through {@link #_runInfo}.
     */
    private JButton _runButton;

    public ReforecastPreparationStepJobMonitorPanel(final ParameterEstimatorRunInfo runInfo,
                                                    final ReforecastPreparationStepProcessor stepProcessor)
    {
        _runInfo = runInfo;
        _runInfo.register(this);
        _stepProcessor = stepProcessor;
        createDisplay();

        stepProcessor.setJobMonitor(this);
    }

    /**
     * Asks the user if the run is to be from scratch or a continuation, and then runs {@link #_stepProcessor}.
     */
    private void startProcessorJob()
    {
        final int option = JOptionPane.showOptionDialog(SwingTools.getGlobalDialogParent(this),
                                                        "<html>Do you want to continue where the last run left off or rerun the step from the beginning.<br><br>"
                                                            + "<b>By clicking Rerun, ALL previously acquired results are discarded before executing the step!</b></html>",
                                                        "Continue or Rerun?",
                                                        JOptionPane.YES_NO_CANCEL_OPTION,
                                                        JOptionPane.QUESTION_MESSAGE,
                                                        null,
                                                        new String[]{"Continue", "Rerun", "Cancel"}, //Continue is yes and Rerun is no.
                                                        "Continue");
        if(option == JOptionPane.CANCEL_OPTION)
        {
            return;
        }
        if(option == JOptionPane.NO_OPTION) //Rerun
        {
            _stepProcessor.clearProgress();
            getRunInfo().post(new UpdateAllPluginReforecastPreparationStepStatusNotice(this));
        }

        _stepProcessor.addListener(new JobListener()
        {
            //The job monitor status will be updated based on information on hand only for eitehr success or failure.
            //The cursor will also be set to default.

            //A failure will result in a message being displayed.  
            @Override
            public void processJobFailure(final Exception exc, final GenericJob theJob, final boolean displayMessage)
            {
                if(displayMessage)
                {
                    JOptionPane.showMessageDialog(SwingTools.getGlobalDialogParent(ReforecastPreparationStepJobMonitorPanel.this),
                                                  exc.getMessage(),
                                                  "Error Processing Step " + _stepProcessor.getName(),
                                                  JOptionPane.ERROR_MESSAGE);
                }
                SwingUtilities.invokeLater(new Runnable()
                {
                    @Override
                    public void run()
                    {
                        _stepProcessor.updateJobMonitorStatus(false);
                    }
                });
                _stepProcessor.removeListener(this);
                _stepProcessor.setCanceledDoNotUpdateProgress(false);
            }

            @Override
            public void processSuccessfulJobCompletion(final GenericJob theJob)
            {
                SwingUtilities.invokeLater(new Runnable()
                {
                    @Override
                    public void run()
                    {
                        _stepProcessor.updateJobMonitorStatus(false);
                    }
                });
                _stepProcessor.removeListener(this);
                _stepProcessor.setCanceledDoNotUpdateProgress(false);
            }
        });

        //Start process.
        _stepProcessor.startJob();
    }

    /**
     * Cancels the {@link #_stepProcessor} job.
     */
    private void stopProcessorJob()
    {
        _stepProcessor.setCanceledDoNotUpdateProgress(true);
    }

    /**
     * The run button is created within this method and runs only this process.
     */
    private void createDisplay()
    {
        _noteLabel = new JLabel("Note Label");
        _progressBar = new JProgressBar(0, 1);

        _runButton = new SelfListeningButton("")
        {
            @Override
            public void actionPerformed(final ActionEvent e)
            {
                if(_runButton.getName().equals("Run"))
                {
                    startProcessorJob();
                }
                else
                {
                    stopProcessorJob();
                    _runButton.setEnabled(false);
                }
            }
        };
        final JToolBar runButtonWrapper = new JToolBar();
        runButtonWrapper.add(_runButton);
        runButtonWrapper.setFloatable(false);

        runButtonWrapper.setBorderPainted(false);
        runButtonWrapper.setOpaque(false);

        //Summary label wrapper to be added later.
//        String instructionSummaryText = "<html><body style='width:100%; word-wrap: break-word'>"
//            + _stepProcessor.getInstructions().getInstructionSummaryForInterface().replace("\n", "<br>") + "</html>";
        final JTextArea textArea = new JTextArea();
        textArea.setEditable(false);
        textArea.setText(_stepProcessor.getInstructions().getInstructionSummaryForInterface());
        textArea.setLineWrap(true);
        textArea.setWrapStyleWord(true);
        final JPanel wrappedAreaPanel = new JPanel(new BorderLayout());
        wrappedAreaPanel.add(textArea, BorderLayout.CENTER);
        wrappedAreaPanel.setBorder(HSwingFactory.createTitledBorder(BorderFactory.createEtchedBorder(),
                                                                    "Instruction Summary",
                                                                    null));

        //Setup the layout
        setLayout(new GridBagLayout());
        final GridBagConstraints constraints = SwingTools.returnGridBagConstraints(0,
                                                                                   0,
                                                                                   1,
                                                                                   1,
                                                                                   1,
                                                                                   1,
                                                                                   GridBagConstraints.NORTHWEST,
                                                                                   GridBagConstraints.BOTH,
                                                                                   new Insets(2, 2, 2, 2),
                                                                                   0,
                                                                                   0);
        add(_noteLabel, constraints);

        constraints.gridy = 1;
        add(_progressBar, constraints);

        constraints.gridy = 2;
        constraints.weighty = 0d;
        add(wrappedAreaPanel, constraints);
        constraints.weighty = 1d;

        constraints.gridy = 0;
        constraints.gridx = 1;
        constraints.gridheight = 3;
        constraints.weightx = 0;
        constraints.fill = GridBagConstraints.NONE;
        constraints.anchor = GridBagConstraints.NORTH;
        add(runButtonWrapper, constraints);

        setBorder(HSwingFactory.createTitledBorder(BorderFactory.createEtchedBorder(), _stepProcessor.getInstructions()
                                                                                                     .getStepId()
            + ": " + _stepProcessor.getInstructions().getStepDescription(), null));
    }

    private ParameterEstimatorRunInfo getRunInfo()
    {
        return _runInfo;
    }

    @Override
    public void updateProgress(final JobMonitorAttr entity)
    {
        if(entity.getProgress() < entity.getMinimum())
        {
            _progressBar.setMinimum(0);
            _progressBar.setMaximum(1);
            _progressBar.setValue(0);
            _progressBar.setIndeterminate(false);
            _progressBar.setEnabled(false);
        }
        else
        {
            _progressBar.setEnabled(true);
            _progressBar.setMinimum(entity.getMinimum());
            _progressBar.setMaximum(entity.getMaximum());
            _progressBar.setValue(entity.getProgress());
            _progressBar.setIndeterminate(entity.isIndeterminate());
        }
        _noteLabel.setText(entity.getNote());
    }

    @Override
    @Subscribe
    public void reactToReforecastPreparationStepCompleted(final ReforecastPreparationStepCompletedNotice evt)
    {
        if(_stepProcessor.isStepComplete())
        {
            _runButton.setName("Run");
            _runButton.setIcon(IconTools.getHSEBIcon("playComplete24x24"));
            _runButton.setToolTipText("Run has been completed. Click to rerun.");
        }
        else
        {
            if(_stepProcessor.canStepBePerformed())
            {
                _runButton.setName("Run");
                _runButton.setIcon(IconTools.getHSEBIcon("playBlue24x24"));
                _runButton.setToolTipText("Click to run the step.");
            }
            else
            {
                _runButton.setName("Not");
                _runButton.setIcon(IconTools.getHSEBIcon("playBlue24x24"));
                _runButton.setToolTipText("Step cannot be run, because it has nothing to do.");
            }
        }
        _runButton.setEnabled(_stepProcessor.canStepBePerformed());
    }

    @Override
    @Subscribe
    public void reactToReforecastPreparationStepRunning(final ReforecastPreparationStepRunningNotice evt)
    {
        if(evt.getSource() != _stepProcessor)
        {
            _runButton.setName("Wait");
            _runButton.setIcon(IconTools.getHSEBIcon("blueHourglass24x24"));
            _runButton.setToolTipText("Waiting for running step to complete.");
            _runButton.setEnabled(false);
        }
        else
        {
            _runButton.setIcon(IconTools.getHSEBIcon("stopRed24x24"));
            _runButton.setName("Stop");
            _runButton.setToolTipText("Stop the currently running step.");
            _runButton.setEnabled(true);
        }
    }

}
