package ohd.hseb.hefs.mefp.sources.plugin;

import ohd.hseb.hefs.mefp.sources.plugin.steps.ReforecastAcquisitionProcessor;
import ohd.hseb.hefs.pe.core.ParameterEstimatorRunInfo;
import ohd.hseb.hefs.utils.jobs.GenericJob;
import ohd.hseb.hefs.utils.jobs.JobMessenger;
import ohd.hseb.hefs.utils.jobs.JobMonitor;
import ohd.hseb.hefs.utils.jobs.JobMonitorAttr;
import ohd.hseb.hefs.utils.log4j.LoggingTools;

import org.apache.logging.log4j.Logger;
import org.apache.logging.log4j.LogManager;

/**
 * Superclass must be extended by any step processor to be called within {@link ReforecastPreparationStepsProcessor}.
 * 
 * @author hankherr
 */
public abstract class ReforecastPreparationStepProcessor extends GenericJob
{
    private PluginForecastSource _source;
    private final Logger _log;
    private final ReforecastPreparationStepInstructions _instructions;
    private ParameterEstimatorRunInfo _runInfo;

    public ReforecastPreparationStepProcessor(final PluginForecastSource source,
                                              final ReforecastPreparationStepInstructions instructions,
                                              final Logger log)
    {
        _source = source;
        _log = log;
        _instructions = instructions;
    }

    public ParameterEstimatorRunInfo getRunInfo()
    {
        return _runInfo;
    }

    public void setRunInfo(final ParameterEstimatorRunInfo runInfo)
    {
        _runInfo = runInfo;
    }

    public void setSource(final PluginForecastSource source)
    {
        _source = source;
    }

    public PluginForecastSource getSource()
    {
        return _source;
    }

    public ReforecastPreparationStepInstructions getInstructions()
    {
        return _instructions;
    }

    /**
     * @return True if the step is already complete.
     */
    public abstract boolean isStepComplete();

    /**
     * @return If there is something to do.
     */
    public abstract boolean canStepBePerformed();

    /**
     * Clears all already made progress.
     * 
     * @param source TODO
     */
    public abstract void clearProgress();

    /**
     * Does the work of the processor.
     * 
     * @param source The forecast source for which the processor is running. The source will contain the step id and
     *            other information needed by the processor.
     * @throws Exception
     */
    public abstract void process() throws Exception;

    /**
     * Override as needed!<br>
     * <br>
     * When called, the processor should call {@link JobMessenger} methods in order to update the text and progress of
     * any listening monitors. This may mean SFTPing to acquire the latest list of files and updating the processed
     * files list, for example. This will ONLY be called when a step is not running (e.g., via clicking the update
     * status button or after steps finish); while it is running whether this method is called or not is determined by
     * the step processor itself.<br>
     * <br>
     * By default, this returns attributes that correspond to a progress that cannot be determined until the step is
     * running. That will result in a disabled progress bar and run button.
     * 
     * @param fullDetermination If true, then if a processor *can* get a new status from an external source, as is the
     *            case for a {@link ReforecastAcquisitionProcessor}, then do so. Otherwise, just provide the information
     *            you have on hand. If false, it should only use what is quickly available to determine the status; it
     *            must run quickly when false is passed in to avoid GUI slowdowns because this method is called often.
     */
    public JobMonitorAttr determineJobStatus(final boolean fullDetermination)
    {
        final JobMonitorAttr attr = new JobMonitorAttr();
        attr.setIndeterminate(true);
        attr.setMaximum(1);
        attr.setMinimum(0);
        attr.setProgress(-1);
        attr.setNote("Progress cannot be determined until process is running.");
        return attr;
    }

    /**
     * Calls the {@link JobMonitor#updateProgress(JobMonitorAttr)} method for the return value of
     * {@link #getJobMonitor()}. This does NOT rely on {@link JobMessenger} so it can work without the thread currently
     * running.
     * 
     * @param fullDetermination If true, then if a processor *can* get a new status from an external source, as is the
     *            case for a {@link ReforecastAcquisitionProcessor}, then do so. Otherwise, just provide the information
     *            you have on hand.
     */
    public void updateJobMonitorStatus(final boolean fullDetermination)
    {
        getJobMonitor().updateProgress(determineJobStatus(fullDetermination));
    }

    /**
     * Wraps the {@link #process()} method, it calls {@link #fireProcessJobFailure(Exception, boolean)} and
     * {@link #fireProcessSuccessfulJobCompletion()} (via {@link #endTask()}) as needed based on whether a
     * {@link Throwable} is thrown out of {@link #process()}. If {@link #getRunInfo()} does not return null, then this
     * will also handle posting {@link ReforecastPreparationStepRunningNotice} and
     * {@link ReforecastPreparationStepCompletedNotice} notices, as well as an
     * {@link UpdateAllPluginReforecastPreparationStepStatusNotice} when the run is complete (whether fail or success).
     */
    @Override
    public void processJob()
    {
        try
        {
            if(getRunInfo() != null)
            {
                getRunInfo().post(new ReforecastPreparationStepRunningNotice(this));
            }
            process();
            endTask();
        }
        catch(final Throwable t)
        {
            LoggingTools.outputStackTraceAsDebug(_log, t);

            Exception exc;
            if(t instanceof Exception)
            {
                exc = (Exception)t;
            }
            else
            {
                exc = new Exception("Failed to complete job; cause: " + t.getMessage());
                exc.setStackTrace(t.getStackTrace());
            }
            fireProcessJobFailure(exc, true);
        }
        finally
        {
            //Make sure any listeners know that the job is done.
            //In addition to other reactions, this will ensure an job monitors are closed.
            setDone(true);
            if(getRunInfo() != null)
            {
                getRunInfo().post(new ReforecastPreparationStepCompletedNotice(this));
                getRunInfo().post(new UpdateAllPluginReforecastPreparationStepStatusNotice(this));
            }
        }
    }

}
