package ohd.hseb.hefs.mefp.sources.plugin;

import java.io.File;
import java.util.ArrayList;
import java.util.LinkedHashMap;
import java.util.List;

import ohd.hseb.hefs.mefp.sources.plugin.steps.CombineReforecastsInstructions;
import ohd.hseb.hefs.mefp.sources.plugin.steps.MoveReforecastFilesInstructions;
import ohd.hseb.hefs.mefp.sources.plugin.steps.MultimodelConstructionInstructions;
import ohd.hseb.hefs.mefp.sources.plugin.steps.ReforecastAcquisitionInstructions;
import ohd.hseb.hefs.mefp.sources.plugin.steps.ReforecastAcquisitionProcessor;
import ohd.hseb.hefs.mefp.sources.plugin.steps.SystemCallInstructions;
import ohd.hseb.hefs.pe.sources.ForecastSourceDefinitionXMLHandler;
import ohd.hseb.hefs.utils.xml.CompositeXMLWriter;
import ohd.hseb.hefs.utils.xml.XMLReader;
import ohd.hseb.hefs.utils.xml.XMLReaderAdapter;
import ohd.hseb.hefs.utils.xml.XMLReaderException;
import ohd.hseb.hefs.utils.xml.XMLVariable;
import ohd.hseb.hefs.utils.xml.XMLWriter;

import org.xml.sax.Attributes;

import com.google.common.collect.Lists;

/**
 * Stores a {@link List} of {@link ReforecastPreparationStepInstructions} and provides a reader and writer to that list.
 * 
 * @author hankherr
 */
public class ReforecastPreparationSteps extends ForecastSourceDefinitionXMLHandler implements Cloneable
{

    /**
     * Records the steps.
     */
    private final List<ReforecastPreparationStepInstructions> _steps = new ArrayList<>();

    /**
     * I cannot make {@link ReforecastPreparationSteps} a {@link LinkedHashMap} because I don't know the id of the step
     * until after I read it and I can't get that information from the readers used. Hence, {@link #_steps} maintains an
     * ordered list and this map is populated during the finalize reading step.
     */
    private final LinkedHashMap<String, ReforecastPreparationStepInstructions> _stepIdToStepInstructionsMap = new LinkedHashMap<>();

    /**
     * The system files directory, where processed file lists files will be updated as reforecasts are processed. See
     * {@link ReforecastAcquisitionProcessor} and {@link ReforecastAcquisitionInstructions}.
     */
    private final File _systemFilesDir;

    private final XMLVariable[] _attributes;

    /**
     * @param systemFilesDir
     * @param attributes Attribute storage objects to be added to the {@link XMLReader} and {@link XMLWriter} returned
     *            by the methods. None of these attributes will be marked as required, so each must have a reasonable
     *            default to fall back on.
     */
    public ReforecastPreparationSteps(final File systemFilesDir, final XMLVariable... attributes)
    {
        _systemFilesDir = systemFilesDir;
        _attributes = attributes;
    }

    /**
     * @return The {@link ReforecastPreparationStepInstructions} associated with the step with the provided id.
     */
    public ReforecastPreparationStepInstructions getStepInstructions(final String id)
    {
        return _stepIdToStepInstructionsMap.get(id);
    }

    /**
     * @return The steps in a list ordered as they were added via reading the XML.
     */
    public List<ReforecastPreparationStepInstructions> getSteps()
    {
        return _steps;
    }

    public void addStep(final ReforecastPreparationStepInstructions step)
    {
        _steps.add(step);
    }

    public void clearSteps()
    {
        _steps.clear();
    }

    public String getXMLTagName()
    {
        return "reforecastPreparationSteps";
    }

    @Override
    public XMLWriter getWriter()
    {
        final CompositeXMLWriter writer = new CompositeXMLWriter(getXMLTagName(), _steps);
        for(final XMLVariable attr: _attributes)
        {
            writer.addAttribute(attr, false);
        }
        return writer;
    }

    @Override
    public XMLReader getReader()
    {
        final XMLReaderAdapter reader = new XMLReaderAdapter(getXMLTagName())
        {

            @Override
            public String getXMLTagName()
            {
                return ReforecastPreparationSteps.this.getXMLTagName();
            }

            @Override
            public XMLReader readInPropertyFromXMLElement(final String elementName, final Attributes attr) throws XMLReaderException
            {
                if(elementName.equals(getXMLTagName()))
                {
                    _steps.clear();
                    super.readInPropertyFromXMLElement(elementName, attr); //Handles attributes.
                }
                else
                {
                    //Loop through the instructions to find a tag match.
                    //XXX Add new create instructions to this list.
                    final List<ReforecastPreparationStepInstructions> possibleInstructions = Lists.newArrayList((ReforecastPreparationStepInstructions)new MultimodelConstructionInstructions(),
                                                                                                                new ReforecastAcquisitionInstructions(_systemFilesDir),
                                                                                                                new MoveReforecastFilesInstructions(),
                                                                                                                new CombineReforecastsInstructions(),
                                                                                                                new SystemCallInstructions());
                    for(final ReforecastPreparationStepInstructions instructions: possibleInstructions)
                    {
                        if(elementName.equals(instructions.getXMLTagName()))
                        {
                            _steps.add(instructions);
                            return instructions.getReader();
                        }
                    }
                }

                return null;
            }

            @Override
            public void finalizeReading() throws XMLReaderException
            {
                super.finalizeReading();

                for(final ReforecastPreparationStepInstructions stepInstructions: ReforecastPreparationSteps.this._steps)
                {
                    _stepIdToStepInstructionsMap.put(stepInstructions.getStepId(), stepInstructions);
                }
            }
        };
        for(final XMLVariable attr: _attributes)
        {
            reader.addAttribute(attr, false);
        }
        return reader;
    }

    @Override
    public boolean equals(final Object o)
    {
        final ReforecastPreparationSteps other = (ReforecastPreparationSteps)o;
        if(_steps.size() != other._steps.size())
        {
            return false;
        }
        for(int i = 0; i < _steps.size(); i++)
        {
            if(!_steps.get(i).equals(other._steps.get(i)))
            {
                return false;
            }
        }
        return true;
    }

    @Override
    public ReforecastPreparationSteps clone()
    {
        File usedDir = null;
        if(_systemFilesDir != null)
        {
            usedDir = new File(_systemFilesDir.getAbsolutePath());
        }
        final ReforecastPreparationSteps cloned = new ReforecastPreparationSteps(usedDir);
        for(final ReforecastPreparationStepInstructions instructions: _steps)
        {
            cloned._steps.add(instructions.clone());
        }
        return cloned;
    }

}
