package ohd.hseb.hefs.mefp.sources.plugin;

import java.io.File;
import java.util.LinkedHashMap;

import ohd.hseb.hefs.mefp.sources.plugin.steps.ReforecastAcquisitionProcessor;
import ohd.hseb.hefs.pe.core.ParameterEstimatorRunInfo;
import ohd.hseb.hefs.utils.jobs.GenericJob;
import ohd.hseb.hefs.utils.jobs.JobListener;
import ohd.hseb.hefs.utils.jobs.JobMessenger;
import ohd.hseb.hefs.utils.tools.FileTools;

import org.apache.logging.log4j.Logger;
import org.apache.logging.log4j.LogManager;

/**
 * Top level processor calls steps specified by {@link ReforecastPreparationSteps} for a {@link #_source}. Each step is
 * run as a separate {@link GenericJob} so that it can be tied to its own monitor. The
 * {@link #_stepInstructionsToProcessor} records the processor used to run each step.
 * 
 * @author hankherr
 */
public class ReforecastPreparationStepsProcessor extends GenericJob
{
    private final ParameterEstimatorRunInfo _runInfo;

    /**
     * Remembered but not actually used anywhere.
     */
    private final PluginForecastSource _source;

    /**
     * Steps to be performed in order.
     */
    private final ReforecastPreparationSteps _steps;

    /**
     * Cross references instructions with the processor created by those instructions via
     * {@link ReforecastPreparationStepInstructions#createProcessor()}.
     */
    private final LinkedHashMap<ReforecastPreparationStepInstructions, ReforecastPreparationStepProcessor> _stepInstructionsToProcessor = new LinkedHashMap<>();

    public ReforecastPreparationStepsProcessor(final ParameterEstimatorRunInfo runInfo,
                                               final ReforecastPreparationSteps steps,
                                               final PluginForecastSource source)
    {
        _runInfo = runInfo;
        _steps = steps;
        _source = source;

        _runInfo.register(this);

        for(final ReforecastPreparationStepInstructions instructions: steps.getSteps())
        {
            final ReforecastPreparationStepProcessor stepProc = instructions.createProcessor(_source);
            stepProc.setRunInfo(runInfo);
            _stepInstructionsToProcessor.put(instructions, stepProc);
        }
    }

    /**
     * @return The {@link ReforecastPreparationStepProcessor} to use for the provided
     *         {@link ReforecastPreparationStepInstructions} instance. This can be used to tied a job monitor dialog to
     *         the processor.
     */
    public ReforecastPreparationStepProcessor getProcessor(final ReforecastPreparationStepInstructions stepInstructions)
    {
        return _stepInstructionsToProcessor.get(stepInstructions);
    }

    public void clearProgress()
    {
        for(final ReforecastPreparationStepInstructions key: _stepInstructionsToProcessor.keySet())
        {
            _stepInstructionsToProcessor.get(key).clearProgress();
        }
        getRunInfo().post(new UpdateAllPluginReforecastPreparationStepStatusNotice(this));
    }

    private ParameterEstimatorRunInfo getRunInfo()
    {
        return _runInfo;
    }

    /**
     * Calls the {@link ReforecastAcquisitionProcessor#process(PluginForecastSource)} method within each processor
     * returned by {@link ReforecastPreparationStepInstructions#createProcessor()} for each step in {@link #_steps}.
     * 
     * @throws Exception
     */
    @Override
    public void processJob()
    {
        //Turn off all cancel and done flags.
        for(final ReforecastPreparationStepInstructions stepInstructions: _steps.getSteps())
        {
            final ReforecastPreparationStepProcessor job = _stepInstructionsToProcessor.get(stepInstructions);
            job.setCanceledDoNotUpdateProgress(false);
            job.setDone(false);
        }

        //Now run...
        for(final ReforecastPreparationStepInstructions stepInstructions: _steps.getSteps())
        {
            final ReforecastPreparationStepProcessor job = _stepInstructionsToProcessor.get(stepInstructions);
            job.setSource(_source);

            //I use an array so that the listener below can set the value of the failure flag.
            //False indicates that there was no failure; true indicates there was a failure.
            final boolean[] failureFlagInSlot0 = new boolean[]{false};

            //Setup the job listener such that it knows to remove itself and, if a failure occurs, it stops altogether.
            final JobListener jobListener = new JobListener()
            {

                //Any processor must fire a job failure if cancelled.
                @Override
                public void processJobFailure(final Exception exc, final GenericJob theJob, final boolean displayMessage)
                {
                    exc.printStackTrace(); //XXX DEBUG!!!
                    job.removeListener(this);
                    fireProcessJobFailure(new Exception("Failed to process step with id "
                                              + stepInstructions.getStepId() + ":\n" + exc.getMessage(), exc),
                                          true);
                    failureFlagInSlot0[0] = true;
                }

                @Override
                public void processSuccessfulJobCompletion(final GenericJob theJob)
                {
                    job.removeListener(this);
                }
            };
            job.addListener(jobListener);

            //Start the job.
            job.startJob();

            //Wait for the job to stop.
            while(job.isJobRunning())
            {
                try
                {
                    Thread.sleep(1);
                }
                catch(final InterruptedException e)
                {
                    job.setCanceledDoNotUpdateProgress(true);
                    e.printStackTrace();
                    fireProcessJobFailure(new Exception("Unexpected interruption waiting for step with id "
                        + stepInstructions.getStepId() + " to finish.", e), true);
                    return;
                }

                //Here, if this job has been canceled, pass it through to the running job and let the listeners failure to stop this
                //wrapper process.
                if(JobMessenger.isCanceled())
                {
                    job.setCanceledDoNotUpdateProgress(true);
                }
            }

            //If the job just run failed out, then exit this noting that we already fired a failure in the listener above.
            if(failureFlagInSlot0[0])
            {
                return;
            }

            //If this job has been canceled, just stop.
            if(JobMessenger.isCanceled())
            {
                fireProcessJobFailure(new Exception("User canceled process."), true);
                return;
            }

            //Update the other steps since this one is done.
            getRunInfo().post(new UpdateAllPluginReforecastPreparationStepStatusNotice(this));
        }

        //Load default estimation options...
        endTask();
    }

//XXX I gave up on this approach for determining the defaults to use.
//
//    /**
//     * @return Array of three values: longest lead time in days (round up) for any provided time series, the earliest
//     *         found year among all T0s, and latest found year.
//     */
//    private int[] determineDefaultEstimationOptions(final TimeSeriesArrays tss)
//    {
//        final int[] results = new int[]{0, Integer.MAX_VALUE, Integer.MIN_VALUE}; //0 - num days; 1 - start year; 2 - end year
//        for(int i = 0; i < tss.size(); i++)
//        {
//            if(!tss.get(i).isEmpty())
//            {
//                results[0] = Math.max(results[0],
//                                      (int)Math.ceil((tss.get(i).getEndTime() - tss.get(i).getStartTime())
//                                          / (HCalendar.MILLIS_IN_HR * 24)));
//
//                final Calendar cal = HCalendar.computeCalendarFromMilliseconds(tss.get(i).getHeader().getForecastTime());
//                results[1] = Math.min(results[1], cal.get(Calendar.YEAR));
//                results[2] = Math.max(results[2], cal.get(Calendar.YEAR));
//            }
//        }
//        return results;
//    }

    /**
     * Utility to be used by anything that implements {@link ReforecastPreparationStepProcessor}. It is used to
     * determine the output directory for the step, and use the source's data handler prepared directory if that output
     * directory is not specified.
     * 
     * @param xmlElement The XML element in the plug-in XML that specifies the output directory.
     * @param instructionSpecifiedDir The directory-specifying {@link File} that indicates the plug-in XML value. May be
     *            null.
     * @param plugInStepId The id of the step involved.
     * @param source The soource involved.
     * @param log A {@link Logger} for generating a debug message indicating when the source default is used.
     * @return The output directory to use.
     * @throws Exception If the output directory cannot be used.
     */
    public static File determineOutputDir(final String xmlElement,
                                          final File instructionSpecifiedDir,
                                          final String plugInStepId,
                                          final PluginForecastSource source,
                                          final Logger log) throws Exception
    {

        File outputDir = instructionSpecifiedDir;
        if(outputDir == null)
        {
            outputDir = FileTools.newFile(source.getSourceDataHandler().getPreparedDataFilesDirectory());
            log.debug("The " + xmlElement + " for plugin step " + plugInStepId
                + " is not specified;, using the default source directory in the run area: "
                + outputDir.getAbsolutePath());
        }
        if(!outputDir.exists())
        {
            outputDir.mkdir();
            if(!outputDir.exists())
            {
                throw new Exception("The value of " + xmlElement + ", " + outputDir.getAbsolutePath()
                    + ", as specified in the plugin step " + plugInStepId + " does not exist and cannot be created.");
            }
        }
        if(!outputDir.canWrite())
        {
            throw new Exception("The value of " + xmlElement + ", " + outputDir.getAbsolutePath()
                + ", as specified in the plugin step " + plugInStepId + " exists but cannot be written to.");
        }

        return outputDir;
    }

}
