package ohd.hseb.hefs.mefp.sources.plugin;

import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.List;

import nl.wldelft.util.timeseries.TimeSeriesArray;
import nl.wldelft.util.timeseries.TimeSeriesArrays;
import ohd.hseb.hefs.pe.tools.TimeSeriesSorter;
import ohd.hseb.hefs.utils.jobs.JobMessenger;
import ohd.hseb.hefs.utils.tools.FileTools;
import ohd.hseb.hefs.utils.tools.ParameterId;
import ohd.hseb.hefs.utils.tsarrays.TimeSeriesArraysTools;
import ohd.hseb.hefs.utils.xml.XMLTools;
import ohd.hseb.util.misc.HStopWatch;

import org.apache.commons.io.FileUtils;
import org.apache.logging.log4j.Logger;
import org.apache.logging.log4j.LogManager;

import com.google.common.collect.ArrayListMultimap;

/**
 * Assumes a directory structure under the base directory like this: [locationId]/[.xml/.fi/.bin files to process].<br>
 * <br>
 * Provided base directory MUST include something separating out precipitation and temperature in its name.
 * 
 * @author Hank.Herr
 */
public class ReforecastTimeSeriesFileCombiner
{
    private static final Logger LOG = LogManager.getLogger(ReforecastTimeSeriesFileCombiner.class);

    private final File _baseDirectory;

    /**
     * Built by constructor, map of locationIds found to {@link File}s that must be combined.
     */
    private final ArrayListMultimap<String, File> _locationToFilesMap = ArrayListMultimap.create();

    //TODO This needs to be drive not by the location, but by the location and parameter, which means a LocationAndDataTypeIdentifier.
    //Modify the map below and then make all other changes as needed!
    //
    //Actually, based on the use of TimeSeriesSorter below, this may work without needing changes.  Have Haksu test it!!!

    /**
     * Map of locationId to corresponding sub-directory to make removing it later easier.
     */
    private final HashMap<String, File> _locationToDirectoryMap = new HashMap<>();

    /**
     * {@link List} of {@link File}s to remove. Populated by {@link #readAndCombineTimeSeries(String)} and removed by
     * {@link #removeFiles(File)}.
     */
    private final List<File> _filesToRemove = new ArrayList<>();

    /**
     * Flag indicates if files are to be removed. Set via {@link #setRemoveFiles(boolean)}.
     */
    private boolean _removeFiles = true;

    public ReforecastTimeSeriesFileCombiner(final File baseDirectory, final boolean logMessages) throws Exception
    {
        this(baseDirectory, logMessages, false);
    }

    /**
     * Constructor populates {@link #_locationToFilesMap}, creating a single, stable listing of files for each location.
     * If another thread continues to populate the directories with more files, those files will be ignored until the
     * next time this is called.
     * 
     * @param baseDirectory {@link File} specifying the base directory, that must include something in it demarcating
     *            precip and temp data, under which directories exist by locationId.
     * @param logMessages Flag indicates if log messages should be generated
     * @param abortAfterFirstFileFoundToCombine If true, then this constructor is being called solely to determine if
     *            there are any files to combine and so it can stop after finding the first one.
     * @throws Exception
     */
    private ReforecastTimeSeriesFileCombiner(final File baseDirectory,
                                             final boolean logMessages,
                                             final boolean abortAfterFirstFileFoundToCombine) throws Exception
    {
        _baseDirectory = baseDirectory;

        if(!_baseDirectory.exists() || !_baseDirectory.canRead())
        {
            throw new Exception("Cannot read base directory for time series combination, "
                + _baseDirectory.getAbsolutePath());
        }

        if(logMessages)
            LOG.info("Building list of files to combine under base directory " + _baseDirectory.getAbsolutePath()
                + "...");

        for(final File subDir: _baseDirectory.listFiles())
        {
            if(subDir.isDirectory() && (subDir.canRead()) && (subDir.canWrite()))
            {
                LOG.debug("Looking into subdirectory " + subDir.getName());
                _locationToDirectoryMap.put(subDir.getName(), subDir);
                for(final File file: subDir.listFiles())
                {
                    if(XMLTools.isFastInfosetFile(file) || XMLTools.isGZIPFile(file) || XMLTools.isXMLFile(file))
                    {
                        _locationToFilesMap.put(subDir.getName(), file);
                        if(abortAfterFirstFileFoundToCombine)
                        {
                            if(logMessages)
                            {
                                LOG.info("Found at least one file to combine.");
                            }
                            return;
                        }
                    }
                }
            }
        }

        if(logMessages)
            LOG.info("Found " + _locationToFilesMap.keySet().size()
                + " locations with reforecast time series to combined into single files.");
    }

    /**
     * @return True if there is anything to combine within {@link #_locationToFilesMap}, which is built when the
     *         constructor is called.
     */
    public boolean areFilesAvailableForCombining()
    {
        return !_locationToFilesMap.isEmpty();
    }

    /**
     * For testing purposes, set this to be false.
     * 
     * @param b Will remove all files as they are processed.
     */
    public void setRemoveFiles(final boolean b)
    {
        _removeFiles = b;
    }

    /**
     * Removes the provided file as well as any bin file that may exist that corresponds to it.
     * 
     * @param file File to remove.
     */
    private void removeFile(final File file)
    {
        file.delete();
        final File binFile = FileTools.replaceExtension(file, ".bin");
        if(binFile.exists())
        {
            binFile.delete();
        }
    }

    /**
     * Calls {@link #_locationToFilesMap} to acquire a listing of files to process for the location. This will read each
     * file, ignore files that are not time series files, and delete those that are after reading.
     * 
     * @param locationId Location id used as a key for {@link #_locationToFilesMap}.
     * @return All time series found in all of the read files, including the base file which will also be read along
     *         with all of the others.
     */
    protected List<TimeSeriesArray> readAndCombineTimeSeries(final String locationId)
    {
        final List<File> files = _locationToFilesMap.get(locationId);
        Collections.sort(files);
        List<TimeSeriesArray> allTS = null;
        _filesToRemove.clear();
        if(files != null)
        {
            LOG.debug("For location " + locationId + ", " + files.size()
                + " many files being combined into a single time series set.");
            for(final File file: files)
            {
                try
                {
//This has been removed due to amount of output it generates.  We'll rely on job status bar to show progress.
//                    LOG.debug("For combination for " + locationId + " processing the file " + file.getAbsolutePath()
//                        + ".");

                    //Read TS file.
                    final TimeSeriesArrays tss = TimeSeriesArraysTools.readFromFile(file);

                    //Remove files if told to.
                    if(_removeFiles)
                    {
                        _filesToRemove.add(file);
                    }

                    //Add to allTS.
                    if(allTS == null)
                    {
                        allTS = TimeSeriesArraysTools.convertTimeSeriesArraysToList(tss);
                    }
                    else
                    {
                        allTS.addAll(TimeSeriesArraysTools.convertTimeSeriesArraysToList(tss));
                    }
                }
                catch(final Exception e)
                {
                    e.printStackTrace();
                    LOG.warn("File found that is not a readable time series file : '" + e.getMessage()
                        + "'. Ignoring the file.");
                }
            }
            LOG.debug("For location " + locationId + ", " + allTS.size() + " many time series found.");
        }

        return allTS;
    }

    /**
     * Output file name will be [location].[precipitation or temperature].reforecasts.fi. It will be combined with
     * {@link #_baseDirectory} to determine the full path.
     */
    protected File constructOutputFile(final String locationId, final boolean precipitation)
    {
        return PluginDataHandler.getPluginDataFile(_baseDirectory, locationId, precipitation);
    }

    /**
     * Removes all files in {@link #_filesToRemove} and then clears the directory. This is called within
     * {@link #combineTimeSeriesForAllLocations()}, but it protected so that it may be used in testing.
     */
    protected void removeFiles(final File directoryToRemoveIfEmpty)
    {
        //Do the files first.
        for(final File file: _filesToRemove)
        {
            removeFile(file);
        }
        try
        {
            //If it is a directory and it is empty, then delete it.
            if((directoryToRemoveIfEmpty.isDirectory()) && (directoryToRemoveIfEmpty.list().length == 0))
            {
                FileUtils.deleteDirectory(directoryToRemoveIfEmpty);
            }
        }
        catch(final IOException e)
        {
            LOG.warn("Attempted to remove the directory " + directoryToRemoveIfEmpty.getAbsolutePath()
                + ", but failed. The directory will be left in place.");
        }
    }

    /**
     * Loops over all locations in {@link #_locationToFilesMap} and combines the time series.
     * 
     * @throws Exception
     */
    public void combineTimeSeriesForAllLocations() throws Exception
    {
        //Do removal in alphabetical order.
        final List<String> locationList = new ArrayList<>(_locationToFilesMap.keySet());
        Collections.sort(locationList);

        //For each location...
        for(final String locationId: locationList)
        {
            if(JobMessenger.isCanceled())
            {
                throw new Exception("Time series T0-reforecast file combination process stopping because of user cancellation.");
            }

            final HStopWatch timer = new HStopWatch();
            LOG.info("Combining " + "time series for location " + locationId + " under directory "
                + _baseDirectory.getAbsolutePath() + "...");

            //Read and combine all input.
            final Collection<TimeSeriesArray> allTS = readAndCombineTimeSeries(locationId);
            LOG.debug("DONE combining time series for location " + locationId + "; writing the output files...");

            //User a sorter to pull out acceptable precipitation files and temperature files, outputting each independently.
            final TimeSeriesSorter sorter = new TimeSeriesSorter(allTS);

            //Precipitation...
            final TimeSeriesSorter precipTS = sorter.restrictViewToParameters(ParameterId.MAP,
                                                                              ParameterId.FMAP,
                                                                              ParameterId.MAPX);
            if(!precipTS.isEmpty())
            {
                final File dataFile = constructOutputFile(locationId, true); //Output file to create.
                TimeSeriesArraysTools.writeToFile(dataFile, addTimeSeriesToThoseInFile(dataFile, precipTS));
                LOG.debug("DONE writing combined precipitation time series file " + dataFile.getAbsolutePath());
            }

            //Temperature...
            final TimeSeriesSorter tempTS = sorter.restrictViewToParameters(ParameterId.MAT,
                                                                            ParameterId.TMIN,
                                                                            ParameterId.TMAX,
                                                                            ParameterId.TFMN,
                                                                            ParameterId.TFMX);
            if(!tempTS.isEmpty())
            {
                final File dataFile = constructOutputFile(locationId, false); //Output file to create.
                TimeSeriesArraysTools.writeToFile(dataFile, addTimeSeriesToThoseInFile(dataFile, tempTS));
                LOG.debug("DONE writing combined temperature time series file " + dataFile.getAbsolutePath());
            }

            //Remove the location-specific directory if the flag is set.
            if(_removeFiles)
            {
                removeFiles(_locationToDirectoryMap.get(locationId));
            }

            LOG.debug("DONE combining time series for location " + locationId + " and writing output files in "
                + timer.getElapsedMillis() + " milliseconds.");
        }
    }

    /**
     * Reads the time series already in the provided file and appends the time series provided to them. It allows only
     * one time series per forecast time, so it makes an assumption that the time series properly correspond to each
     * other in terms of location, parameter, etc.
     * 
     * @return The new list of time series.
     */
    private Collection<TimeSeriesArray> addTimeSeriesToThoseInFile(final File dataFile,
                                                                   final Collection<TimeSeriesArray> tsToAdd) throws Exception
    {
        if(dataFile.exists())
        {
            if(!dataFile.canRead() || !dataFile.canWrite())
            {
                throw new Exception("Target file to write, " + dataFile.getAbsolutePath()
                    + ", exists, but cannot be read from or written to.");
            }

            final TimeSeriesArrays existingTS = TimeSeriesArraysTools.readFromFile(dataFile);

            final LinkedHashMap<Long, TimeSeriesArray> map = TimeSeriesArraysTools.createMapOfForecastTimeToSingleTimeSeries(TimeSeriesArraysTools.convertTimeSeriesArraysToList(existingTS));
            for(final TimeSeriesArray ts: tsToAdd)
            {
                map.put(ts.getHeader().getForecastTime(), ts);
            }
            return map.values();
        }
        return tsToAdd;
    }

    /**
     * Constructs a {@link ReforecastTimeSeriesFileCombiner} that will only check for at least one file to combine. If
     * one file is found, then true is returned. This wraps the method
     * {@link ReforecastTimeSeriesFileCombiner#areFilesAvailableForCombining()}
     * 
     * @param baseDirectory The directory to check, passed through to the constructor.
     * @param logFiles Passed through to the constructor.
     * @return True if there is at least one file that requires processing through combining.
     * @throws Exception
     */
    public static boolean areFilesAvailableForCombining(final File baseDirectory, final boolean logFiles) throws Exception
    {
        ReforecastTimeSeriesFileCombiner combiner;
        combiner = new ReforecastTimeSeriesFileCombiner(baseDirectory, false, true);
        return combiner.areFilesAvailableForCombining();
    }
}
