package ohd.hseb.hefs.mefp.sources.plugin.steps;

import java.io.File;
import java.util.ArrayList;
import java.util.List;

import ohd.hseb.hefs.mefp.sources.plugin.AbstractReforecastPreparationStepInstructions;
import ohd.hseb.hefs.mefp.sources.plugin.PluginDataHandler;
import ohd.hseb.hefs.mefp.sources.plugin.PluginForecastSource;
import ohd.hseb.hefs.mefp.sources.plugin.ReforecastPreparationStepProcessor;
import ohd.hseb.hefs.mefp.sources.plugin.ReforecastPreparationStepsProcessor;
import ohd.hseb.hefs.utils.tools.ListTools;
import ohd.hseb.hefs.utils.xml.CollectionXMLReader;
import ohd.hseb.hefs.utils.xml.CollectionXMLWriter;
import ohd.hseb.hefs.utils.xml.XMLReader;
import ohd.hseb.hefs.utils.xml.XMLReaderFactory;
import ohd.hseb.hefs.utils.xml.XMLWriter;
import ohd.hseb.hefs.utils.xml.vars.XMLFile;
import ohd.hseb.hefs.utils.xml.vars.XMLYesNoBoolean;

/**
 * Instructions for combining reforecasts across several directories into one. All files to combine follow the naming
 * scheme implied by {@link PluginDataHandler#getPluginDataFile(File, String, boolean)}. The output
 * directory can be an input directory (specify a directory of {@link #RESERVED_STRING_FOR_OUTPUT_DIR}, in which case
 * this will append more reforecasts to the previously generated file.
 * 
 * @author hankherr
 */
public class CombineReforecastsInstructions extends AbstractReforecastPreparationStepInstructions
{
    public static final String RESERVED_STRING_FOR_OUTPUT_DIR = "OUTPUTDIR";

    /**
     * Directories in which to find files for combination.
     */
    private final List<XMLFile> _reforecastDirectories = new ArrayList<>();

    /**
     * The directory where the results will be placed. This is not required; {@link CombineReforecastsProcessor} will
     * call
     * {@link ReforecastPreparationStepsProcessor#determineOutputDir(String, File, String, PluginForecastSource, org.apache.log4j.Logger)
     * if this is not specified.
     */
    private final XMLFile _outputDir = new XMLFile("outputDir", false);

    /**
     * Indicates if the combined files should be removed after combining and creating the output directory.
     */
    private final XMLYesNoBoolean _removeFilesAfterCombining = new XMLYesNoBoolean("removeFilesAfterCombining", true);

    public void addReforecastDirectory(final File dir)
    {
        _reforecastDirectories.add(new XMLFile(getListElementTagName(), dir, false));
    }

    public void clearReforecastDirectoryList()
    {
        _reforecastDirectories.clear();
    }

    /**
     * @return True if the directory equals {@link #RESERVED_STRING_FOR_OUTPUT_DIR}.
     */
    public boolean isReforecastDirOutputDir(final File dir)
    {
        return dir.getName().equals(RESERVED_STRING_FOR_OUTPUT_DIR);
    }

    /**
     * @return {@link List} of {@link File} instances converted from the {@link XMLFile} instances in
     *         {@link #_reforecastDirectories}.
     */
    public List<File> getReforecastDirectories()
    {
        final List<File> results = new ArrayList<>();
        for(final XMLFile xmlFile: _reforecastDirectories)
        {
            results.add(xmlFile.get());
        }
        return results;
    }

    private String getListElementTagName()
    {
        return "directory";
    }

    public File getOutputDir()
    {
        return _outputDir.get();
    }

    public void setOutputDir(final File dir)
    {
        _outputDir.set(dir);
    }

    public boolean getRemoveFilesAfterCombining()
    {
        return _removeFilesAfterCombining.get();
    }

    public void setRemoveFilesAfterCombining(final boolean b)
    {
        _removeFilesAfterCombining.set(b);
    }

    @Override
    public String getXMLTagName()
    {
        return "combineReforecastsInstructions";
    }

    @Override
    public XMLWriter getWriter()
    {
        final CollectionXMLWriter writer = new CollectionXMLWriter(getXMLTagName(), _reforecastDirectories);
        addReforecastPreparationStepInstructionsAttributes(writer);
        writer.addAttribute(_outputDir, false);
        writer.addAttribute(_removeFilesAfterCombining, false);
        return writer;
    }

    @Override
    public XMLReader getReader()
    {
        final CollectionXMLReader<XMLFile> reader = new CollectionXMLReader<>(getXMLTagName(),
                                                                              _reforecastDirectories,
                                                                              new XMLReaderFactory<XMLFile>()
                                                                              {
                                                                                  @Override
                                                                                  public XMLFile get()
                                                                                  {
                                                                                      return new XMLFile(getListElementTagName(),
                                                                                                         false); //Specified dirs that do not exist are not used.
                                                                                  }
                                                                              });
        addReforecastPreparationStepInstructionsAttributes(reader);
        reader.addAttribute(_outputDir, false);
        reader.addAttribute(_removeFilesAfterCombining, false);
        return reader;
    }

    @Override
    public boolean equals(final Object obj)
    {
        if(!super.equals(obj))
        {
            return false;
        }
        final CombineReforecastsInstructions other = (CombineReforecastsInstructions)obj;

        if((!_outputDir.equals(other._outputDir))
            || (!_removeFilesAfterCombining.equals(other._removeFilesAfterCombining)))
        {
            return false;
        }

        return ListTools.twoSidedListCheck(_reforecastDirectories, other._reforecastDirectories);
    }

    @Override
    public CombineReforecastsInstructions clone()
    {
        final CombineReforecastsInstructions cloned = new CombineReforecastsInstructions();
        cloned.copyFrom(this);
        for(final XMLFile xmlFile: _reforecastDirectories)
        {
            cloned._reforecastDirectories.add((XMLFile)xmlFile.clone());
        }
        cloned._outputDir.set(_outputDir.get());
        cloned._removeFilesAfterCombining.set(_removeFilesAfterCombining.get());
        return cloned;
    }

    @Override
    public ReforecastPreparationStepProcessor createProcessor(final PluginForecastSource source)
    {
        return new CombineReforecastsProcessor(source, this);
    }

    @Override
    public String getInstructionSummaryForInterface()
    {
        String summary = "";
        for(final XMLFile xmlFile: _reforecastDirectories)
        {
            if(isReforecastDirOutputDir(xmlFile.get()))
            {
                summary += "Source Directory: Default directory in MEFPPE run area\n";
            }
            else
            {
                summary += "Source Directory: " + xmlFile.get().getAbsolutePath() + "\n";
            }
        }

        if(getOutputDir() == null)
        {
            summary += "Output Directory: Default directory in MEFPPE run area\n";
        }
        else
        {
            summary += "Output Directory: " + getOutputDir().getAbsolutePath() + "\n";
        }

        if(getRemoveFilesAfterCombining())
        {
            summary += "Combined files will be removed after combining.\n";
        }
        else
        {
            summary += "Combined files will NOT be removed after combining.\n";
        }
        return summary;
    }
}
