package ohd.hseb.hefs.mefp.sources.plugin.steps;

import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.LinkedHashMap;
import java.util.List;

import nl.wldelft.util.timeseries.TimeSeriesArray;
import nl.wldelft.util.timeseries.TimeSeriesArrays;
import ohd.hseb.hefs.mefp.sources.plugin.PluginDataHandler;
import ohd.hseb.hefs.mefp.sources.plugin.PluginForecastSource;
import ohd.hseb.hefs.mefp.sources.plugin.ReforecastPreparationStepProcessor;
import ohd.hseb.hefs.mefp.sources.plugin.ReforecastPreparationStepsProcessor;
import ohd.hseb.hefs.utils.jobs.JobMessenger;
import ohd.hseb.hefs.utils.jobs.JobMonitorAttr;
import ohd.hseb.hefs.utils.tools.FileTools;
import ohd.hseb.hefs.utils.tools.GeneralTools;
import ohd.hseb.hefs.utils.tsarrays.TimeSeriesArraysTools;
import ohd.hseb.util.misc.HString;

import org.apache.logging.log4j.Logger;
import org.apache.logging.log4j.LogManager;

import com.google.common.collect.LinkedHashMultimap;

/**
 * Uses {@link CombineReforecastsInstructions} to combine reforecasts from multiple directories into one large file.
 * This will gather a list of the files from the first directory that are appropriately named according to
 * {@link PluginDataHandler#isPluginDataFile(File)}. It will then combine the time series in identically named file and
 * write then to a file by the same name under {@link CombineReforecastsInstructions#getOutputDir()}. <br>
 * <br>
 * Repeated time series are allowed, but the last one in is the one used. For the purposes of determining repeats, two
 * time series are considered repeats if their locationId, parameterId, ensembleId, ensembleMemberIndex, forecast time,
 * time step (millis) and qualifiers (in order) are all identical.
 * 
 * @author hankherr
 */
public class CombineReforecastsProcessor extends ReforecastPreparationStepProcessor
{
    private static final Logger LOG = LogManager.getLogger(CombineReforecastsProcessor.class);

    public LinkedHashMultimap<String, File> _fileNameToFilesMap = LinkedHashMultimap.create();

    /**
     * This will check for directory existence for the {@link CombineReforecastsInstructions#getReforecastDirectories()}
     * and readability and then populate the {@link #_fileNameToFilesMap} so that {@link #process()} can be called.
     * 
     * @param instructions Instructions to use.
     */
    public CombineReforecastsProcessor(final PluginForecastSource source,
                                       final CombineReforecastsInstructions instructions)
    {
        super(source, instructions, LOG);

        populateFileNameToFilesMap(true);

        setName(instructions.getStepId());
    }

    /**
     * Loops through the reforecast directories provided in the instructions and looks for files to process. It
     * populates {@link #_fileNameToFilesMap} based on what it finds.
     * 
     * @param generateWarnings If warnings should be generated when directories are provided that cannot be used for
     *            whatever reason.
     */
    private void populateFileNameToFilesMap(final boolean generateWarnings)
    {
        if(this.isJobRunning())
        {
            return;
        }

        _fileNameToFilesMap.clear();

        //Loop through all directories in the instructions.
        for(File dir: getInstructions().getReforecastDirectories())
        {
            //Handle the special-case output dir reference.
            if(getInstructions().isReforecastDirOutputDir(dir))
            {
                try
                {
                    dir = determineOutputDir();
                }
                catch(final Exception e)
                {
                    if(generateWarnings)
                    {
                        LOG.warn("Output directory cannot be determined and will be skipped: " + e.getMessage());
                    }
                    continue;
                }
            }

            //Now do the checks.
            if(!dir.exists())
            {
                if(generateWarnings)
                {
                    LOG.warn("Directory "
                        + dir.getAbsolutePath()
                        + " does not exist and will be skipped; either the XML plug-in configuration is incorrect or the reforecast acquisition failed.");
                }
                continue;
            }
            if(!dir.isDirectory())
            {
                if(generateWarnings)
                {
                    LOG.warn("Directory "
                        + dir.getAbsolutePath()
                        + " exists but is a file, not a directory, and will be skipped; check the XML plug-in configuration.");
                }
                continue;
            }
            if(!dir.canRead())
            {
                if(generateWarnings)
                {
                    LOG.warn("Directory " + dir.getAbsolutePath()
                        + " exists but cannot be read and will be skipped; check the XML plug-in configuration.");
                }
                continue;
            }

            //And add the files.
            final File[] listing = dir.listFiles();
            for(final File file: listing)
            {
                if(PluginDataHandler.isPluginDataFile(file))
                {
                    _fileNameToFilesMap.put(file.getName(), file);
                }
            }
        }
    }

    /**
     * Wraps
     * {@link ReforecastPreparationStepsProcessor#determineOutputDir(String, File, String, PluginForecastSource, Logger)}
     * .
     */
    private File determineOutputDir() throws Exception
    {
        return ReforecastPreparationStepsProcessor.determineOutputDir("outputDir",
                                                                      getInstructions().getOutputDir(),
                                                                      getInstructions().getStepId(),
                                                                      getSource(),
                                                                      LOG);
    }

    /**
     * @return True if any of the files to check is not within the output directory returned by
     *         {@link #determineOutputDir()}.
     */
    private boolean includesAFileNotInTheOutputDir(final Collection<File> filesToCheck)
    {
        File outputDir;
        try
        {
            outputDir = this.determineOutputDir();
        }
        catch(final Exception e)
        {
            //Assume the output directory does not exist so all keys in the map represent files that need combining.
            return !filesToCheck.isEmpty();
        }

        for(final File file: filesToCheck)
        {
            if(!file.getParentFile().equals(outputDir))
            {
                return true;
            }
        }
        return false;
    }

    /**
     * @return True if any of the files to check is not within the output directory returned by
     *         {@link #determineOutputDir()}.
     */
    private boolean includesFileInOutputDir(final Collection<File> filesToCheck)
    {
        File outputDir;
        try
        {
            outputDir = this.determineOutputDir();
        }
        catch(final Exception e)
        {
            //Assume the output directory does not exist so all keys in the map represent files that need combining.
            return !filesToCheck.isEmpty();
        }

        for(final File file: filesToCheck)
        {
            if(file.getParentFile().equals(outputDir))
            {
                return true;
            }
        }
        return false;
    }

    /**
     * @return True if any of the files to combine, specified in {@link #_fileNameToFilesMap}, are in the output
     *         directory.
     */
    private boolean includesCombiningAnyFilesInOutputDir()
    {
        for(final String fileName: _fileNameToFilesMap.keySet())
        {
            if(includesFileInOutputDir(_fileNameToFilesMap.get(fileName)))
            {
                return true;
            }
        }
        return false;
    }

    /**
     * @return A count of the number of unique files that require combining where the combination consists of at least
     *         one file that is not in the output directory. Output directory files will not be combined if they have
     *         nothing that must be combined with them.
     */
    private int countNumberOfNonOutputFilesThatNeedCombining()
    {
        int numFiles = 0;
        for(final String fileName: _fileNameToFilesMap.keySet())
        {
            final boolean doesFileExistOutsideOfOutputDir = includesAFileNotInTheOutputDir(_fileNameToFilesMap.get(fileName));

            //At least one file found for combining, so add to numFiles.
            if(doesFileExistOutsideOfOutputDir)
            {
                numFiles++;
            }
        }

        return numFiles;
    }

    @Override
    public CombineReforecastsInstructions getInstructions()
    {
        return (CombineReforecastsInstructions)super.getInstructions();
    }

    /**
     * Follows the {@link #_instructions} to combine time series in like-named files in the reforecast directories
     * specified in the instructions and generate an output file in the appropriate location.
     * 
     * @throws Exception
     */
    @Override
    public void process() throws Exception
    {
        //Repopulate the map, since it may have been altered if the user ran this process from scratch.  True is used so 
        //that warnings can be output again for bad directories. Since this is only triggered with a user button click, 
        //it should be okay to output warnings.
        populateFileNameToFilesMap(true);

        LOG.info("Combining " + _fileNameToFilesMap.keySet().size() + " files...");
        JobMessenger.setJobStatus(new JobMonitorAttr("Combining " + _fileNameToFilesMap.keySet().size() + " files...",
                                                     0,
                                                     0,
                                                     _fileNameToFilesMap.keySet().size()));
        final File outputDir = determineOutputDir();

        //For each file name (i.e., location)...
        for(final String name: _fileNameToFilesMap.keySet())
        {
            LOG.info("Combining reforecast time series found in files with the name " + name
                + " into a single file under " + outputDir.getAbsolutePath());
            LOG.info("Parameters of combination: files will be removed afterwards = "
                + getInstructions().getRemoveFilesAfterCombining() + ".");

            //Get the list of files to combine.  Do checks involving files in the output directory for appropriate messaging.
            final Collection<File> filesToCombine = _fileNameToFilesMap.get(name);
            if(!includesAFileNotInTheOutputDir(filesToCombine))
            {
                continue; //Skip if there are no files to combine outside the output directory.
            }
            String combineOutputMessageSuffix = "...";
            if(includesFileInOutputDir(filesToCombine))
            {
                combineOutputMessageSuffix = ", including an existing output file...";
            }

            //List of the files that will be removed.
            final List<File> filesToRemove = new ArrayList<>();

            JobMessenger.updateNote("Combining " + filesToCombine.size() + " files with the name " + name
                + combineOutputMessageSuffix);

            //This map will contain the results for the current location, to be combined later when output.
            final LinkedHashMap<TimeSeriesIdentifier, TimeSeriesArray> allSeriesMap = new LinkedHashMap<>();

            /////////////////////////////////////
            //Read each file.
            /////////////////////////////////////
            LOG.debug("Found " + filesToCombine.size() + " many files to combine into a single reforecast file.");
            for(final File file: filesToCombine)
            {
                //Cancel while reading.
                if(this.isCanceled())
                {
                    throw new Exception("Combine process stopping because of user cancellation.");
                }
                if(!file.exists())
                {
                    LOG.debug("Skipping a file that has mysteriously disappeared: " + file.getAbsolutePath());
                    continue;
                }

                LOG.debug("Processing file " + file.getAbsolutePath() + ".");
                try
                {
                    final TimeSeriesArrays resultsForName = TimeSeriesArraysTools.readFromFile(file);
                    int overwriteCount = 0;
                    for(int i = 0; i < resultsForName.size(); i++)
                    {
                        final TimeSeriesIdentifier tsIdent = new TimeSeriesIdentifier(resultsForName.get(i));
                        if(allSeriesMap.get(tsIdent) != null)
                        {
                            overwriteCount++;
                        }
                        allSeriesMap.put(tsIdent, resultsForName.get(i));
                    }
                    LOG.debug("Found " + overwriteCount
                        + " many time series that overwrite a previously found time series within the file "
                        + file.getAbsolutePath() + ".");

                    //Queue the file to be removed if told to do so.
                    if(getInstructions().getRemoveFilesAfterCombining())
                    {
                        filesToRemove.add(file);
                    }
                }
                catch(final Exception e)
                {
                    e.printStackTrace();
                    LOG.warn("Skipping file that could not be processed, " + file.getAbsolutePath() + ": "
                        + e.getMessage());
                }
            }

            //Cancel before writing the one file.
            if(this.isCanceled())
            {
                throw new Exception("Combine process stopping because of user cancellation.");
            }

            /////////////////////////////////////
            //Write the output file.
            /////////////////////////////////////
            final File outputFile = FileTools.newFile(outputDir, name);
            LOG.debug("Writing combined time series to file " + FileTools.newFile(outputDir, name) + "...");
            try
            {
                TimeSeriesArraysTools.writeToFile(outputFile, allSeriesMap.values());
                if(filesToRemove.contains(outputFile))
                {
                    filesToRemove.remove(outputFile);
                }
            }
            catch(final Exception e)
            {
                throw new Exception("Problem encountered writing time series to file.", e);
            }

            //Cancel before removing files (but do NOT cancel in the middle; either all removed or none.
            if(this.isCanceled())
            {
                throw new Exception("Combine process stopping because of user cancellation.");
            }

            /////////////////////////////////////
            //Remove all combined files
            /////////////////////////////////////
            LOG.debug("Removing successfully processed files...");
            for(final File file: filesToRemove)
            {
                if(!file.delete())
                {
                    LOG.warn("After combining reforecast time series files, failed to remove the processed file "
                        + file.getAbsolutePath() + "; file will be left in place.");
                }
            }

            JobMessenger.madeProgress();
            LOG.info("Reforecast combination processor is done combining reforecasts for file with name " + name + ".");
        }

        LOG.info("Reforecast combination processor is done combining reforecasts into multimodel ensembles for "
            + _fileNameToFilesMap.keySet().size() + " reforecast files.");
    }

    @Override
    public boolean isStepComplete()
    {
        return countNumberOfNonOutputFilesThatNeedCombining() == 0;
    }

    @Override
    public boolean canStepBePerformed()
    {
        populateFileNameToFilesMap(false);
        if(_fileNameToFilesMap.isEmpty())
        {
            return false;
        }
        return true;
    }

    @Override
    public JobMonitorAttr determineJobStatus(final boolean fullDetermination)
    {
        populateFileNameToFilesMap(false);

        final int numFiles = countNumberOfNonOutputFilesThatNeedCombining();
        String combineOutputMessageSuffix = "";
        if(includesCombiningAnyFilesInOutputDir())
        {
            combineOutputMessageSuffix = ", possibly with existing output files";
        }

        final JobMonitorAttr attr = new JobMonitorAttr();
        attr.setIndeterminate(false);
        attr.setMaximum(Math.max(1, numFiles));
        attr.setMinimum(0);
        attr.setProgress(0);
        if(numFiles == 0)
        {
            attr.setNote("There are no files the must be combined.");
        }
        else
        {
            attr.setNote(numFiles + " files are ready to be combined" + combineOutputMessageSuffix + ".");
        }
        return attr;
    }

    @Override
    public void clearProgress()
    {
        try
        {
            final File outputDir = determineOutputDir();

            //You can do jack if the to directory does not exist, is not a directory, or cannot be written to.  
            //For this method, do nothing.  Note that the process method will fail.
            if(!outputDir.exists() || !outputDir.isDirectory() || !outputDir.canWrite())
            {
                return;
            }

            final File[] files = outputDir.listFiles();
            for(final File toFile: files)
            {
                if(PluginDataHandler.isPluginDataFile(toFile))
                {
                    try
                    {
                        java.nio.file.Files.delete(toFile.toPath());
                    }
                    catch(final IOException e)
                    {
                        LOG.warn("Step " + getInstructions().getStepId() + ": Skipping file that cannot be removed, "
                            + toFile.getName() + ": " + e.getMessage());
                    }
                }
            }
        }
        catch(final Exception e1)
        {
            LOG.warn("Step " + getInstructions().getStepId() + ": Unexpected problem trying to clear progress: "
                + e1.getMessage());
            return;
        }
    }

    /**
     * Inner class is used as an identifier for a {@link TimeSeriesArray} and is used within a map inside
     * {@link #process()} in order to create a map of unique time series.
     * 
     * @author hankherr
     */
    private class TimeSeriesIdentifier
    {
        private final String _locationId;
        private final String _parameterId;
        private final String _ensembleId;
        private final int _ensembleMemberIndex;
        private final List<String> _qualifiers;
        private final long _forecastTime;
        private final long _timeStepMillis;

        public TimeSeriesIdentifier(final TimeSeriesArray ts)
        {
            _locationId = ts.getHeader().getLocationId();
            _parameterId = ts.getHeader().getParameterId();
            _ensembleId = ts.getHeader().getEnsembleId();
            _ensembleMemberIndex = ts.getHeader().getEnsembleMemberIndex();
            _forecastTime = ts.getHeader().getForecastTime();
            _timeStepMillis = ts.getHeader().getTimeStep().getStepMillis();

            _qualifiers = new ArrayList<>(ts.getHeader().getQualifierCount());
            for(int index = 0; index < ts.getHeader().getQualifierCount(); index++)
            {
                _qualifiers.add(ts.getHeader().getQualifierId(index));
            }
        }

        @Override
        public boolean equals(final Object o)
        {
            if(!(o instanceof TimeSeriesIdentifier))
            {
                return false;
            }
            final TimeSeriesIdentifier other = (TimeSeriesIdentifier)o;

            return (_locationId.equals(other._locationId)) && (_parameterId.equals(other._parameterId))
                && GeneralTools.checkForFullEqualityOfObjects(_ensembleId, other._ensembleId)
                && (_ensembleMemberIndex == other._ensembleMemberIndex) && (_forecastTime == other._forecastTime)
                && (_timeStepMillis == other._timeStepMillis) && _qualifiers.equals(other._qualifiers);
        }

        @Override
        public int hashCode()
        {
            return ("" + _locationId + " " + _parameterId + " " + _ensembleId + " " + _ensembleMemberIndex + " "
                + _forecastTime + " " + _timeStepMillis + " " + HString.buildStringFromList(_qualifiers, ",")).hashCode();
        }
    }
}
