package ohd.hseb.hefs.mefp.sources.plugin.steps;

import java.io.File;

import ohd.hseb.hefs.mefp.sources.plugin.AbstractReforecastPreparationStepInstructions;
import ohd.hseb.hefs.mefp.sources.plugin.PluginForecastSource;
import ohd.hseb.hefs.mefp.sources.plugin.ReforecastPreparationStepProcessor;
import ohd.hseb.hefs.mefp.sources.plugin.ReforecastPreparationStepsProcessor;
import ohd.hseb.hefs.utils.xml.CompositeXMLReader;
import ohd.hseb.hefs.utils.xml.CompositeXMLWriter;
import ohd.hseb.hefs.utils.xml.XMLReader;
import ohd.hseb.hefs.utils.xml.XMLWriter;
import ohd.hseb.hefs.utils.xml.vars.XMLEnum;
import ohd.hseb.hefs.utils.xml.vars.XMLFile;

/**
 * Instructions for moving or copying reforecast files from one directory to another. If {@link #_toDir} contains a null
 * {@link File}, then {@link MoveReforecastFilesProcessor} will call
 * {@link ReforecastPreparationStepsProcessor#determineOutputDir(String, File, String, PluginForecastSource, org.apache.log4j.Logger)}
 * to identify the default directory in the run area.
 * 
 * @author hankherr
 */
public class MoveReforecastFilesInstructions extends AbstractReforecastPreparationStepInstructions
{
    /**
     * Enumeration indicating the type of move to be performed by the {@link MoveReforecastFilesProcessor},
     * {@link MoveActionType#copy} or {@link MoveActionType#move}.
     * 
     * @author hankherr
     */
    public static enum MoveActionType
    {
        copy, move;
    }

    /**
     * Default type is {@link MoveActionType#move}.
     */
    private final XMLEnum<MoveActionType> _type = new XMLEnum(MoveActionType.class, "type", MoveActionType.move);

    /**
     * Directory from which to copy/move files.
     */
    private final XMLFile _fromDir = new XMLFile("fromDir", false);

    /**
     * Directory to which to copy/move files. This does not have to be specified; it can be left as null.
     */
    private final XMLFile _toDir = new XMLFile("toDir", false);

    public File getFromDir()
    {
        return _fromDir.get();
    }

    public void setFromDir(final File fromDir)
    {
        _fromDir.set(fromDir);
    }

    public File getToDir()
    {
        return _toDir.get();
    }

    public void setToDir(final File toDir)
    {
        _toDir.set(toDir);
    }

    /**
     * @return True if {@link #_type} is {@link MoveActionType#move}.
     */
    public boolean isMove()
    {
        return _type.get().equals(MoveActionType.move);
    }

    public void setType(final MoveActionType type)
    {
        _type.set(type);
    }

    @Override
    public String getXMLTagName()
    {
        return "moveReforecastFilesInstructions";
    }

    @Override
    public XMLReader getReader()
    {
        _fromDir.setRequired(true);
        final CompositeXMLReader reader = new CompositeXMLReader(getXMLTagName(), _fromDir, _toDir);
        addReforecastPreparationStepInstructionsAttributes(reader);
        reader.addAttribute(_type, false);
        return reader;
    }

    @Override
    public XMLWriter getWriter()
    {
        final CompositeXMLWriter writer = new CompositeXMLWriter(getXMLTagName(), _fromDir, _toDir);
        addReforecastPreparationStepInstructionsAttributes(writer);
        writer.addAttribute(_type, false);
        return writer;
    }

    @Override
    public MoveReforecastFilesInstructions clone()
    {
        final MoveReforecastFilesInstructions cloned = new MoveReforecastFilesInstructions();
        cloned.copyFrom(this);
        cloned._fromDir.set(_fromDir.get());
        cloned._toDir.set(_toDir.get());
        cloned._type.set(_type.get());
        return cloned;
    }

    @Override
    public boolean equals(final Object obj)
    {
        if(!super.equals(obj))
        {
            return false;
        }

        final MoveReforecastFilesInstructions other = (MoveReforecastFilesInstructions)obj;

        if((!_type.equals(other._type)) || (!_fromDir.equals(other._fromDir)) || (!_toDir.equals(other._toDir)))
        {
            return false;
        }
        return true;
    }

    @Override
    public ReforecastPreparationStepProcessor createProcessor(final PluginForecastSource source)
    {
        return new MoveReforecastFilesProcessor(source, this);
    }

    @Override
    public String getInstructionSummaryForInterface()
    {
        String summary = "";

        if(isMove())
        {
            summary += "Files will be moved...\n";
        }
        else
        {
            summary += "Files will be copied...\n";
        }

        summary += "From Directory: " + getFromDir().getAbsolutePath() + "\n";

        if(getToDir() == null)
        {
            summary += "To Directory: Default directory in MEFPPE run area\n";
        }
        else
        {
            summary += "To Directory: " + getToDir().getAbsolutePath() + "\n";
        }

        return summary;
    }
}
