package ohd.hseb.hefs.mefp.sources.plugin.steps;

import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.List;

import ohd.hseb.hefs.mefp.sources.plugin.PluginDataHandler;
import ohd.hseb.hefs.mefp.sources.plugin.PluginForecastSource;
import ohd.hseb.hefs.mefp.sources.plugin.ReforecastPreparationStepProcessor;
import ohd.hseb.hefs.mefp.sources.plugin.ReforecastPreparationStepsProcessor;
import ohd.hseb.hefs.utils.jobs.JobMessenger;
import ohd.hseb.hefs.utils.jobs.JobMonitorAttr;
import ohd.hseb.hefs.utils.tools.FileTools;

import org.apache.logging.log4j.Logger;
import org.apache.logging.log4j.LogManager;

import com.google.common.io.Files;

/**
 * Follows instructions provided in a {@link MoveReforecastFilesInstructions} instance in order to move the reforecast
 * files that match conditions checked in {@link PluginDataHandler#isPluginDataFile(File)} to a new
 * directory.
 * 
 * @author hankherr
 */
public class MoveReforecastFilesProcessor extends ReforecastPreparationStepProcessor
{
    private static final Logger LOG = LogManager.getLogger(MoveReforecastFilesProcessor.class);

    public MoveReforecastFilesProcessor(PluginForecastSource source, final MoveReforecastFilesInstructions instructions)
    {
        super(source, instructions, LOG);

        setName(instructions.getStepId());
    }

    /**
     * @return A {@link List} of {@link File} instances, one for each file that can be moved. This uses
     *         {@link PluginDataHandler#isPluginDataFile(File)} to determine if a file is one that should be
     *         moved.
     */
    private List<File> getFilesToMove()
    {
        List<File> results = new ArrayList<>();

        if(!getInstructions().getFromDir().exists() || !getInstructions().getFromDir().isDirectory()
            || !getInstructions().getFromDir().canRead())
        {
            return results;
        }
        final File[] files = getInstructions().getFromDir().listFiles();

        for(final File fromFile: files)
        {
            if(PluginDataHandler.isPluginDataFile(fromFile))
            {
                results.add(fromFile);
            }
        }
        return results;
    }

    /**
     * Wraps
     * {@link ReforecastPreparationStepsProcessor#determineOutputDir(String, File, String, PluginForecastSource, Logger)}
     * .
     */
    private File determineOutputDir() throws Exception
    {
        return ReforecastPreparationStepsProcessor.determineOutputDir("toDir",
                                                                      getInstructions().getToDir(),
                                                                      getInstructions().getStepId(),
                                                                      getSource(),
                                                                      LOG);
    }

    @Override
    public MoveReforecastFilesInstructions getInstructions()
    {
        return (MoveReforecastFilesInstructions)super.getInstructions();
    }

    /**
     * @throws Exception If any one of multiple conditions fails.
     */
    @Override
    public void process() throws Exception
    {
        //Check that the from dir and exists and can be read.
        if(!getInstructions().getFromDir().exists())
        {
            throw new Exception("The fromDir, " + getInstructions().getFromDir().getAbsolutePath()
                + ", specified in the plugin " + getInstructions().getStepId() + " does not exist.");
        }
        if(!getInstructions().getFromDir().canRead())
        {
            throw new Exception("The fromDir, " + getInstructions().getFromDir().getAbsolutePath()
                + ", specified in the plugin " + getInstructions().getStepId() + " exists but cannot be read.");
        }

        //Determine the target, to, directory based on either the instructions or the source's data handler prepared directory.
        final File targetDir = determineOutputDir();

        //Loop through all the files under the from dir, but only move those that appear to be gridded plugin data files.
        LOG.info("Moving all files under " + getInstructions().getFromDir().getAbsolutePath() + " to "
            + targetDir.getAbsolutePath() + "...");
        final List<File> files = getFilesToMove();
        if(files.isEmpty())
        {
            LOG.info("No files were found in the directory " + getInstructions().getFromDir() + " to move or copy.");
            JobMessenger.setJobStatus(new JobMonitorAttr("No files to move.", 1, 0, 1));
            return;
        }
        JobMessenger.setJobStatus(new JobMonitorAttr("Moving " + files.size() + " files...", 0, 0, files.size()));

        for(final File fromFile: files)
        {
            if(this.isCanceled())
            {
                throw new Exception("Move process stopping because of user cancellation.");
            }

            JobMessenger.updateNote("Moving (or copying) file " + fromFile.getName());
            final File toFile = FileTools.newFile(targetDir, fromFile.getName());
            try
            {
                if(getInstructions().isMove())
                {
                    FileTools.moveFile(fromFile, toFile);
                    LOG.debug("Moved file " + fromFile.getAbsolutePath() + " to " + toFile.getAbsolutePath() + ".");
                }
                else
                {
                    Files.copy(fromFile, toFile);
                    LOG.debug("Copied file " + fromFile.getAbsolutePath() + " to " + toFile.getAbsolutePath() + ".");
                }
            }
            catch(final Exception e)
            {
                LOG.warn("Skipping the file " + fromFile.getName() + " which cannnot be moved to"
                    + toFile.getAbsolutePath() + " for this reason: " + e.getMessage());
            }
            JobMessenger.madeProgress();
        }
        LOG.info("DONE moving all files under " + getInstructions().getFromDir().getAbsolutePath() + " to "
            + targetDir.getAbsolutePath() + ".");
    }

    @Override
    public boolean isStepComplete()
    {
        return false; //TODO???
    }

    @Override
    public boolean canStepBePerformed()
    {
        if(getFilesToMove().isEmpty())
        {
            return false;
        }
        return true;
    }

    @Override
    public JobMonitorAttr determineJobStatus(final boolean fullDetermination)
    {
        List<File> files = getFilesToMove();

        final JobMonitorAttr attr = new JobMonitorAttr();
        attr.setIndeterminate(false);
        attr.setMaximum(Math.max(1, files.size()));
        attr.setMinimum(0);
        attr.setProgress(0);
        if(files.isEmpty())
        {
            attr.setNote("There are no files ready to be moved.");
        }
        else
        {
            attr.setNote(files.size() + " files are ready to be moved.");
        }
        return attr;
    }

    @Override
    public void clearProgress()
    {
        try
        {
            File outputDir = determineOutputDir();

            //You can do jack if the to directory does not exist, is not a directory, or cannot be written to.  
            //For this method, do nothing.  Note that the process method will fail.
            if(!outputDir.exists() || !outputDir.isDirectory() || !outputDir.canWrite())
            {
                return;
            }

            final File[] files = outputDir.listFiles();
            for(final File toFile: files)
            {
                if(PluginDataHandler.isPluginDataFile(toFile))
                {
                    try
                    {
                        java.nio.file.Files.delete(toFile.toPath());
                    }
                    catch(IOException e)
                    {
                        LOG.warn("Step " + getInstructions().getStepId() + ": Skipping file that cannot be removed, "
                            + toFile.getName() + ": " + e.getMessage());
                    }
                }
            }
        }
        catch(Exception e1)
        {
            LOG.warn("Step " + getInstructions().getStepId() + ": Unexpected problem trying to clear progress: "
                + e1.getMessage());
            return;
        }
    }
}
