package ohd.hseb.hefs.mefp.sources.plugin.steps;

import java.io.File;
import java.util.ArrayList;
import java.util.List;

import ohd.hseb.hefs.mefp.sources.plugin.AbstractReforecastPreparationStepInstructions;
import ohd.hseb.hefs.mefp.sources.plugin.PluginDataHandler;
import ohd.hseb.hefs.mefp.sources.plugin.PluginForecastSource;
import ohd.hseb.hefs.mefp.sources.plugin.ReforecastPreparationStepProcessor;
import ohd.hseb.hefs.mefp.sources.plugin.ReforecastPreparationStepsProcessor;
import ohd.hseb.hefs.utils.tools.ListTools;
import ohd.hseb.hefs.utils.xml.CollectionXMLReader;
import ohd.hseb.hefs.utils.xml.CollectionXMLWriter;
import ohd.hseb.hefs.utils.xml.XMLReader;
import ohd.hseb.hefs.utils.xml.XMLReaderFactory;
import ohd.hseb.hefs.utils.xml.XMLWriter;
import ohd.hseb.hefs.utils.xml.vars.XMLFile;
import ohd.hseb.hefs.utils.xml.vars.XMLYesNoBoolean;

/**
 * Instructions identifying directories containing reforecast files that will be combined into multimodel ensembles. The
 * reforecast files must match the naming convention assumed in
 * {@link PluginDataHandler#isPluginDataFile(File)}. If the {@link #_outputDir} is not specified, the
 * {@link MultimodelContructionProcessor} will call
 * {@link ReforecastPreparationStepsProcessor#determineOutputDir(String, File, String, PluginForecastSource, org.apache.log4j.Logger)}
 * to identify the appropriate directory in the run area to use.
 * 
 * @author hankherr
 */
public class MultimodelConstructionInstructions extends AbstractReforecastPreparationStepInstructions
{

    /**
     * Directories containing files with time series to combine.
     */
    private final List<XMLFile> _modelReforecastDirectories = new ArrayList<>();

    /**
     * The directory where resulting ensembles will be written to files. The {@link File} within can be left as null to
     * use a default.
     */
    private final XMLFile _outputDir = new XMLFile("outputDir", false);

    /**
     * If yes, then files will be removed after the ensembles are created and successfully written to output files.
     */
    private final XMLYesNoBoolean _removeFilesAfterCombining = new XMLYesNoBoolean("removeFilesAfterCombining", true);

    /**
     * If yes, then all time series will have the same forecast length: that of the shortest member. If no, the length
     * may vary between members.
     */
    private final XMLYesNoBoolean _trimTimeSeriesToCommonLength = new XMLYesNoBoolean("trimTimeSeriesToCommonLength",
                                                                                      true);

    public void addMultimodelReforecastDirectory(final File dir)
    {
        _modelReforecastDirectories.add(new XMLFile(getListElementTagName(), dir, false));
    }

    public void clearMultimodelReforecastDirectoryList()
    {
        _modelReforecastDirectories.clear();
    }

    public List<File> getModelReforecastDirectories()
    {
        final List<File> results = new ArrayList<>();
        for(final XMLFile xmlFile: _modelReforecastDirectories)
        {
            results.add(xmlFile.get());
        }
        return results;
    }

    private String getListElementTagName()
    {
        return "directory";
    }

    public boolean getTrimTimeSeriesToCommonLength()
    {
        return _trimTimeSeriesToCommonLength.get();
    }

    public void setTrimTimeSeriesToCommonLength(final boolean b)
    {
        _trimTimeSeriesToCommonLength.set(b);
    }

    public File getOutputDir()
    {
        return _outputDir.get();
    }

    public void setOutputDir(final File dir)
    {
        _outputDir.set(dir);
    }

    public boolean getRemoveFilesAfterCombining()
    {
        return _removeFilesAfterCombining.get();
    }

    public void setRemoveFilesAfterCombining(final boolean b)
    {
        _removeFilesAfterCombining.set(b);
    }

    @Override
    public String getXMLTagName()
    {
        return "multimodelConstructionInstructions";
    }

    @Override
    public XMLWriter getWriter()
    {
        final CollectionXMLWriter writer = new CollectionXMLWriter(getXMLTagName(), _modelReforecastDirectories);
        addReforecastPreparationStepInstructionsAttributes(writer);
        writer.addAttribute(_outputDir, true);
        writer.addAttribute(_removeFilesAfterCombining, false);
        writer.addAttribute(_trimTimeSeriesToCommonLength, false);
        return writer;
    }

    @Override
    public XMLReader getReader()
    {
        final CollectionXMLReader<XMLFile> reader = new CollectionXMLReader<>(getXMLTagName(),
                                                                              _modelReforecastDirectories,
                                                                              new XMLReaderFactory<XMLFile>()
                                                                              {
                                                                                  @Override
                                                                                  public XMLFile get()
                                                                                  {
                                                                                      return new XMLFile(getListElementTagName(),
                                                                                                         false); //Specified dirs that do not exist are not used.
                                                                                  }
                                                                              });
        addReforecastPreparationStepInstructionsAttributes(reader);
        reader.addAttribute(_outputDir, true);
        reader.addAttribute(_removeFilesAfterCombining, false);
        reader.addAttribute(_trimTimeSeriesToCommonLength, false);
        return reader;
    }

    @Override
    public boolean equals(final Object obj)
    {
        if(!super.equals(obj))
        {
            return false;
        }
        final MultimodelConstructionInstructions other = (MultimodelConstructionInstructions)obj;

        if((!_trimTimeSeriesToCommonLength.equals(_trimTimeSeriesToCommonLength))
            || (!_outputDir.equals(other._outputDir))
            || (!_removeFilesAfterCombining.equals(other._removeFilesAfterCombining)))
        {
            return false;
        }

        return ListTools.twoSidedListCheck(_modelReforecastDirectories, other._modelReforecastDirectories);
    }

    @Override
    public MultimodelConstructionInstructions clone()
    {
        final MultimodelConstructionInstructions cloned = new MultimodelConstructionInstructions();
        cloned.copyFrom(this);
        for(final XMLFile xmlFile: _modelReforecastDirectories)
        {
            cloned._modelReforecastDirectories.add((XMLFile)xmlFile.clone());
        }
        cloned._trimTimeSeriesToCommonLength.set(_trimTimeSeriesToCommonLength.get());
        cloned._outputDir.set(_outputDir.get());
        cloned._removeFilesAfterCombining.set(_removeFilesAfterCombining.get());
        return cloned;
    }

    @Override
    public ReforecastPreparationStepProcessor createProcessor(final PluginForecastSource source)
    {
        return new MultimodelContructionProcessor(source, this);
    }

    @Override
    public String getInstructionSummaryForInterface()
    {
        String summary = "";
        for(final XMLFile xmlFile: _modelReforecastDirectories)
        {
            summary += "Model Directory: " + xmlFile.get().getAbsolutePath() + "\n";
        }

        if(getOutputDir() == null)
        {
            summary += "Output Directory: Default directory in MEFPPE run area\n";
        }
        else
        {
            summary += "Output Directory: " + getOutputDir().getAbsolutePath() + "\n";
        }

        if(getRemoveFilesAfterCombining())
        {
            summary += "Combined files will be removed after combining\n";
        }
        else
        {
            summary += "Combined files will NOT be removed after combining\n";
        }

        if(getTrimTimeSeriesToCommonLength())
        {
            summary += "Members will be trimmed in length to the lead time of the shortest member\n";
        }
        else
        {
            summary += "Members will not be trimmed\n";
        }

        return summary;
    }
}
