package ohd.hseb.hefs.mefp.sources.plugin.steps;

import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.LinkedHashMap;
import java.util.List;

import nl.wldelft.util.timeseries.DefaultTimeSeriesHeader;
import nl.wldelft.util.timeseries.TimeSeriesArray;
import nl.wldelft.util.timeseries.TimeSeriesArrays;
import ohd.hseb.hefs.mefp.sources.plugin.PluginDataHandler;
import ohd.hseb.hefs.mefp.sources.plugin.PluginForecastSource;
import ohd.hseb.hefs.mefp.sources.plugin.ReforecastPreparationStepProcessor;
import ohd.hseb.hefs.mefp.sources.plugin.ReforecastPreparationStepsProcessor;
import ohd.hseb.hefs.utils.jobs.JobMessenger;
import ohd.hseb.hefs.utils.jobs.JobMonitorAttr;
import ohd.hseb.hefs.utils.tools.FileTools;
import ohd.hseb.hefs.utils.tools.MapTools;
import ohd.hseb.hefs.utils.tsarrays.TimeSeriesArraysTools;
import ohd.hseb.hefs.utils.tsarrays.TimeSeriesEnsemble;

import org.apache.logging.log4j.Logger;
import org.apache.logging.log4j.LogManager;

import com.google.common.collect.LinkedHashMultimap;

/**
 * Follows {@link MultimodelConstructionInstructions} to building multi-model ensembles and output them to files.
 * 
 * @author hankherr
 */
public class MultimodelContructionProcessor extends ReforecastPreparationStepProcessor
{
    private static final Logger LOG = LogManager.getLogger(MultimodelContructionProcessor.class);

    public LinkedHashMultimap<String, File> _fileNameToFilesMap = LinkedHashMultimap.create();

    /**
     * This will check for directory existence for the
     * {@link MultimodelConstructionInstructions#getModelReforecastDirectories()} and readability and then populate the
     * {@link #_fileNameToFilesMap} so that {@link #process()} can be called.
     * 
     * @param source The {@link PluginForecastSource} associated with this processor.
     * @param instructions Instructions to use.
     */
    public MultimodelContructionProcessor(final PluginForecastSource source,
                                          final MultimodelConstructionInstructions instructions)
    {
        super(source, instructions, LOG);

        populateFileNameToFilesMap(true);

        setName(instructions.getStepId());
    }

    /**
     * Loops through the reforecast directories provided in the instructions and looks for files to process. It
     * populates {@link #_fileNameToFilesMap} based on what it finds.
     * 
     * @param generateWarnings If warnings should be generated when directories are provided that cannot be used for
     *            whatever reason.
     */
    private void populateFileNameToFilesMap(final boolean generateWarnings)
    {
        if(this.isJobRunning())
        {
            return;
        }
        _fileNameToFilesMap.clear();

        //Loop through all directories in the instructions.
        for(final File dir: getInstructions().getModelReforecastDirectories())
        {
            if(!dir.exists())
            {
                if(generateWarnings)
                {
                    LOG.warn("Directory "
                        + dir.getAbsolutePath()
                        + " does not exist and will be skipped; either the XML plug-in configuration is incorrect or the reforecast acquisition failed.");
                }
                continue;
            }
            if(!dir.isDirectory())
            {
                if(generateWarnings)
                {
                    LOG.warn("Directory "
                        + dir.getAbsolutePath()
                        + " exists but is a file, not a directory, and will be skipped; check the XML plug-in configuration.");
                }
                continue;
            }
            if(!dir.canRead())
            {
                if(generateWarnings)
                {
                    LOG.warn("Directory " + dir.getAbsolutePath()
                        + " exists but cannot be read and will be skipped; check the XML plug-in configuration.");
                }
                continue;
            }
            final File[] listing = dir.listFiles();
            for(final File file: listing)
            {
                if(PluginDataHandler.isPluginDataFile(file))
                {
                    _fileNameToFilesMap.put(file.getName(), file);
                }
            }
        }
    }

    /**
     * Wraps
     * {@link ReforecastPreparationStepsProcessor#determineOutputDir(String, File, String, PluginForecastSource, Logger)}
     * .
     */
    private File determineOutputDir() throws Exception
    {
        return ReforecastPreparationStepsProcessor.determineOutputDir("outputDir",
                                                                      getInstructions().getOutputDir(),
                                                                      getInstructions().getStepId(),
                                                                      getSource(),
                                                                      LOG);
    }

    @Override
    public MultimodelConstructionInstructions getInstructions()
    {
        return (MultimodelConstructionInstructions)super.getInstructions();
    }

    /**
     * Follows the {@link #_instructions} to construct multimodel ensembles across the files found in the reforecast
     * directories specified in the instructions.
     * 
     * @throws Exception
     */
    @Override
    public void process() throws Exception
    {
        //Repopulate the map in case files have changed.  True is used so that warnings can be output again for bad directories.
        //Since this is only triggered with a user button click, it should be okay to output warnings.
        populateFileNameToFilesMap(true);

        //Determine the output directory either from the instructions or from the source data handler.
        final File outputDir = determineOutputDir();

        JobMessenger.setJobStatus(new JobMonitorAttr("Creating multimodel ensembles for "
            + _fileNameToFilesMap.keySet().size() + " different files...", 0, 0, _fileNameToFilesMap.keySet().size()));

        //For each file name (i.e., location)...
        for(final String name: _fileNameToFilesMap.keySet())
        {
            LOG.info("Combining reforecast time series found in files with the name " + name
                + " into multimodel ensembles.");
            LOG.info("Parameters of combination: files will be removed afterwards = "
                + getInstructions().getRemoveFilesAfterCombining()
                + "; members will be trimmed to all have same length = "
                + getInstructions().getTrimTimeSeriesToCommonLength());

            //Get the list of files to combined.
            final Collection<File> filesToCombine = _fileNameToFilesMap.get(name);
            final List<File> filesToRemove = new ArrayList<>();

            JobMessenger.updateNote("Combining " + filesToCombine.size() + " reforecast files with the name " + name
                + " into multimodel ensembles...");

            //This map will contain the results for the current location, to be combined later when output.
            final LinkedHashMap<Long, TimeSeriesEnsemble> forecastTimeToEnsembleMap = new LinkedHashMap<>();

            //For each file...
            LOG.debug("Found " + filesToCombine.size()
                + " many files to combine into a single multimodel ensemble reforecast file.");
            for(final File file: filesToCombine)
            {
                //Cancel before reading a file.  The rest of this multimodel construction process is fast until writing 
                //occurs, so this one cancel reaction should be sufficient.
                if(this.isCanceled())
                {
                    throw new Exception("Multimodel construction process stopping because of user cancellation.");
                }

                LOG.debug("Processing file " + file.getAbsolutePath() + ".");
                try
                {
                    final TimeSeriesArrays resultsForName = TimeSeriesArraysTools.readFromFile(file);
                    final List<TimeSeriesArray> tsList = TimeSeriesArraysTools.convertTimeSeriesArraysToList(resultsForName);

                    //Skip files that were empty.
                    if(tsList.isEmpty())
                    {
                        continue;
                    }

                    //First set of times series being processed, remove anything in the map for which that forecast time is not 
                    //found in the list to be processed.  In other words, all multimodel ensembles must have one member from all 
                    //non-empty sources.
                    final boolean firstTimeProcessing = forecastTimeToEnsembleMap.isEmpty();
                    final List<Long> timesTooRemove = new ArrayList<>();
                    if(!firstTimeProcessing)
                    {
                        for(final Long forecastTime: forecastTimeToEnsembleMap.keySet())
                        {
                            if(TimeSeriesArraysTools.searchByForecastTime(tsList, forecastTime) < 0)
                            {
                                timesTooRemove.add(forecastTime);
                            }
                        }
                    }
                    MapTools.removeAll(forecastTimeToEnsembleMap, timesTooRemove);

                    //For every time series to be processed, look for an existing item in the map.
                    //If one is found, add the new member to it, trimming the data if tthe found
                    //item is a TimeSeriesEnsemble.
                    //If one is not found AND this is the first time series being edded to the map, then
                    //create a holder and put it in.  Otherwise, ignore the time series and move on.
                    for(final TimeSeriesArray ts: tsList)
                    {
                        final DefaultTimeSeriesHeader header = (DefaultTimeSeriesHeader)ts.getHeader();
                        header.setEnsembleId("MEFP-MULTIMODEL");
                        TimeSeriesEnsemble ensemble = forecastTimeToEnsembleMap.get(ts.getHeader().getForecastTime());

                        //Time series found in the map.  Add a member appropriately.
                        if(ensemble != null)
                        {
                            ensemble.addMember(ts, true);

                            //Member index is the current size of the time series array; i.e., counting starts at 1.
                            header.setEnsembleMemberIndex(ensemble.size());
                        }

                        //Otherwise, if ths is the first time the map is being populated, create a time series
                        //holder basedo the trim flag.
                        else if(firstTimeProcessing)
                        {
                            ensemble = new TimeSeriesEnsemble();
                            ensemble.setCheckEndTimesEqual(getInstructions().getTrimTimeSeriesToCommonLength());
                            ensemble.add(ts);

                            //Member index counting starts at 1, hence tss.size() will work here because it must be one.
                            header.setEnsembleMemberIndex(ensemble.size());

                            forecastTimeToEnsembleMap.put(ts.getHeader().getForecastTime(), ensemble);
                        }

                        //Otherwise, skip the time series.
                    }

                    if(getInstructions().getRemoveFilesAfterCombining())
                    {
                        filesToRemove.add(file);
                    }
                }
                catch(final Exception e)
                {
                    e.printStackTrace();
                    LOG.warn("Skipping file that could not be processed, " + file.getAbsolutePath() + ": "
                        + e.getMessage());
                }
            }

            //Cancel before writing.
            if(this.isCanceled())
            {
                throw new Exception("Multimodel construction process stopping because of user cancellation.");
            }

            //////////////////////////////////
            //Write the output file.
            //////////////////////////////////
            final File outputFile = FileTools.newFile(outputDir, name);
            LOG.debug("Writing combined multimodel ensembles to file " + outputFile + "...");
            final TimeSeriesArrays tss = TimeSeriesArraysTools.convertTimeSeriesArraysCollections(forecastTimeToEnsembleMap.values());
            try
            {
                TimeSeriesArraysTools.writeToFile(outputFile, tss);
            }
            catch(final Exception e)
            {
                throw new Exception("Problem encountered writing multi-model ensemble to file.", e);
            }

            //Cancel before removing files, but not during (either all or none are removed).
            if(this.isCanceled())
            {
                throw new Exception("Multimodel construction process stopping because of user cancellation.");
            }

            //////////////////////////////////
            //Remove all combined files
            //////////////////////////////////
            LOG.debug("Removing successfully processed files...");
            for(final File file: filesToRemove)
            {
                if(!file.delete())
                {
                    LOG.warn("After combining reforecast time series files, failed to remove the processed file "
                        + file.getAbsolutePath() + "; file will be left in place.");
                }
            }

            JobMessenger.madeProgress();
            LOG.info("Multimodel processor is done combining reforecasts for file with name " + name + ".");
        }

        LOG.info("Multimodel processor is done combining reforecasts into multimodel ensembles for "
            + _fileNameToFilesMap.keySet().size() + " reforecast files.");
    }

    @Override
    public boolean isStepComplete()
    {
        return false; //TODO???
    }

    @Override
    public boolean canStepBePerformed()
    {
        populateFileNameToFilesMap(false);
        if(_fileNameToFilesMap.isEmpty())
        {
            return false;
        }
        return true;
    }

    @Override
    public JobMonitorAttr determineJobStatus(final boolean fullDetermination)
    {
        populateFileNameToFilesMap(false);

        final int numFiles = _fileNameToFilesMap.keySet().size();

        final JobMonitorAttr attr = new JobMonitorAttr();
        attr.setIndeterminate(false);
        attr.setMaximum(Math.max(1, numFiles));
        attr.setMinimum(0);
        attr.setProgress(0);
        if(numFiles == 0)
        {
            attr.setNote("There are no files available for creating multimodel ensemble.");
        }
        else
        {
            attr.setNote(numFiles + " files are ready to combine into a multiemodel ensemble.");
        }
        return attr;
    }

    @Override
    public void clearProgress()
    {
        try
        {
            final File outputDir = determineOutputDir();

            //You can do jack if the to directory does not exist, is not a directory, or cannot be written to.  
            //For this method, do nothing.  Note that the process method will fail.
            if(!outputDir.exists() || !outputDir.isDirectory() || !outputDir.canWrite())
            {
                return;
            }

            final File[] files = outputDir.listFiles();
            for(final File toFile: files)
            {
                if(PluginDataHandler.isPluginDataFile(toFile))
                {
                    try
                    {
                        java.nio.file.Files.delete(toFile.toPath());
                    }
                    catch(final IOException e)
                    {
                        LOG.warn("Step " + getInstructions().getStepId() + ": Skipping file that cannot be removed, "
                            + toFile.getName() + ": " + e.getMessage());
                    }
                }
            }
        }
        catch(final Exception e1)
        {
            LOG.warn("Step " + getInstructions().getStepId() + ": Unexpected problem trying to clear progress: "
                + e1.getMessage());
            return;
        }
    }
}
