package ohd.hseb.hefs.mefp.sources.plugin.steps;

import java.io.File;

import ohd.hseb.hefs.mefp.sources.plugin.AbstractReforecastPreparationStepInstructions;
import ohd.hseb.hefs.mefp.sources.plugin.PluginForecastSource;
import ohd.hseb.hefs.mefp.sources.plugin.ReforecastPreparationStepProcessor;
import ohd.hseb.hefs.mefp.sources.plugin.ReforecastPreparationStepsProcessor;
import ohd.hseb.hefs.utils.tools.FileTools;
import ohd.hseb.hefs.utils.xml.CompositeXMLReader;
import ohd.hseb.hefs.utils.xml.CompositeXMLWriter;
import ohd.hseb.hefs.utils.xml.XMLReader;
import ohd.hseb.hefs.utils.xml.XMLWriter;
import ohd.hseb.hefs.utils.xml.vars.XMLFile;
import ohd.hseb.hefs.utils.xml.vars.XMLYesNoBoolean;

/**
 * Instructions for use with {@link ReforecastAcquisitionProcessor}. Includes {@link SFTPInstructions},
 * {@link WorkflowInstructions}, and a {@link #_reforecastOutputBaseDirectory} that specifies where the reforecast time
 * series files output by the CHPS workflow can be found.<br>
 * <br>
 * NOTE: The {@link #_reforecastOutputBaseDirectory} MUST match where time series files are exported to by the CHPS
 * workflow. Furthermore the directory structure under that directory MUST be [locationId]/[.xml/.fi (.bin as
 * appropriate) files providing data]. Note that the data type does not play a role at this point. See
 * {@link ReforecastAcquisitionProcessor}.<br>
 * <br>
 * If {@link #_reforecastOutputBaseDirectory} does not specify a file (i.e, its {@link File} is null), the processor
 * will call
 * {@link ReforecastPreparationStepsProcessor#determineOutputDir(String, File, String, PluginForecastSource, org.apache.log4j.Logger)}
 * to determine the default output directory to use under the run area.
 * 
 * @author hankherr
 */
public class ReforecastAcquisitionInstructions extends AbstractReforecastPreparationStepInstructions
{
    /**
     * This cannot be final; clone will set it.
     */
    private SFTPInstructions _sftpInstructions = new SFTPInstructions();

    /**
     * This cannot be final; clone will set it.
     */
    private WorkflowInstructions _workflowInstructions = new WorkflowInstructions();

    /**
     * Attribute specifies if the reforecast files should be combined into one after they are acquired.
     */
    private final XMLYesNoBoolean _combineReforecasts = new XMLYesNoBoolean("combineReforecasts", true);

    /**
     * Attribute specifies the location where output reforecast files can be found. The structure under this directory
     * must be [locationId]/[XML/fi/bin files specifying data].
     */
    private final XMLFile _reforecastOutputBaseDirectory = new XMLFile("reforecastOutputBaseDirectory",
                                                                       new File(""),
                                                                       false);

    /**
     * The system files directory under the run area, to be passed into the constructor for
     * {@link ReforecastAcquisitionProcessor}. This is not considered part of the instructions: it is not output to XML
     * and not checked via {@link #equals(Object)}. It will be included via {@link #clone()}, however.
     */
    private final File _systemFilesDir;

    public ReforecastAcquisitionInstructions(final File systemFilesDir)
    {
        _systemFilesDir = systemFilesDir;
    }

    public SFTPInstructions getSftpInstructions()
    {
        return _sftpInstructions;
    }

    public WorkflowInstructions getWorkflowInstructions()
    {
        return _workflowInstructions;
    }

    public File getReforecastOutputBaseDirectory()
    {
        return _reforecastOutputBaseDirectory.get();
    }

    public void setReforecastOutputBaseDirectory(final File dir)
    {
        _reforecastOutputBaseDirectory.set(dir);
    }

    public boolean getCombineReforecasts()
    {
        return _combineReforecasts.get();
    }

    public void setCombineReforecasts(final boolean b)
    {
        _combineReforecasts.set(b);
    }

    @Override
    public String getXMLTagName()
    {
        return "reforecastAcquisitionInstructions";
    }

    @Override
    public XMLWriter getWriter()
    {
        final CompositeXMLWriter writer = new CompositeXMLWriter(getXMLTagName(),
                                                                 _sftpInstructions,
                                                                 _workflowInstructions);
        addReforecastPreparationStepInstructionsAttributes(writer);
        writer.addAttribute(_reforecastOutputBaseDirectory, false);
        writer.addAttribute(_combineReforecasts, false);
        return writer;
    }

    @Override
    public XMLReader getReader()
    {
        final CompositeXMLReader reader = new CompositeXMLReader(getXMLTagName(),
                                                                 _sftpInstructions,
                                                                 _workflowInstructions);
        addReforecastPreparationStepInstructionsAttributes(reader);
        reader.addAttribute(_reforecastOutputBaseDirectory, false);
        reader.addAttribute(_combineReforecasts, false);
        return reader;
    }

    @Override
    public boolean equals(final Object obj)
    {
        if(!super.equals(obj))
        {
            return false;
        }
        final ReforecastAcquisitionInstructions other = (ReforecastAcquisitionInstructions)obj;
        return ((_sftpInstructions.equals(other._sftpInstructions))
            && (_workflowInstructions.equals(other._workflowInstructions))
            && (_reforecastOutputBaseDirectory.equals(other._reforecastOutputBaseDirectory)) && (_combineReforecasts.equals(other._combineReforecasts)));
    }

    @Override
    public ReforecastAcquisitionInstructions clone()
    {
        File usedDir = null;
        if(_systemFilesDir != null)
        {
            usedDir = new File(_systemFilesDir.getAbsolutePath());
        }
        final ReforecastAcquisitionInstructions cloned = new ReforecastAcquisitionInstructions(usedDir);
        cloned.copyFrom(this);
        cloned._sftpInstructions = _sftpInstructions.clone();
        cloned._workflowInstructions = _workflowInstructions.clone();
        cloned._reforecastOutputBaseDirectory.set(_reforecastOutputBaseDirectory.get());
        cloned._combineReforecasts.set(_combineReforecasts.get());
        return cloned;
    }

    @Override
    public ReforecastPreparationStepProcessor createProcessor(final PluginForecastSource source)
    {
        return new ReforecastAcquisitionProcessor(source, this, FileTools.newFile(_systemFilesDir, source.getSourceId()
                                                                                                         .toLowerCase()
            + "." + getStepId() + ".processedReforecasts.xml.gz"));
    }

    @Override
    public String getInstructionSummaryForInterface()
    {
        String summary = "";

        summary += _sftpInstructions.getSummary();
        summary += "\n";
        summary += "Workflow Executed: " + _workflowInstructions.getWorkflowID() + "\n";

        if(getReforecastOutputBaseDirectory() == null)
        {
            summary += "Expected Workflow Output Directory: Source directory under MEFPPE run area\n";
        }
        else
        {
            summary += "Expected Workflow Output Directory " + getReforecastOutputBaseDirectory() + "\n";
        }

        if(getCombineReforecasts())
        {
            summary += "Acquired and processed files will be combined into a single file.\n";
        }
        else
        {
            summary += "Acquired and processed files will NOT be combined into a single file.\n";
        }

        return summary;
    }
}
