package ohd.hseb.hefs.mefp.sources.plugin.steps;

import java.io.File;
import java.io.IOException;

import ohd.hseb.hefs.mefp.sources.plugin.PluginDataHandler;
import ohd.hseb.hefs.mefp.sources.plugin.PluginForecastSource;
import ohd.hseb.hefs.mefp.sources.plugin.ProcessedFilesList;
import ohd.hseb.hefs.mefp.sources.plugin.ReforecastPreparationStepProcessor;
import ohd.hseb.hefs.mefp.sources.plugin.ReforecastPreparationStepsProcessor;
import ohd.hseb.hefs.mefp.sources.plugin.ReforecastTimeSeriesFileCombiner;
import ohd.hseb.hefs.pe.core.ParameterEstimatorRunInfo;
import ohd.hseb.hefs.utils.jobs.GenericJob;
import ohd.hseb.hefs.utils.jobs.JobListener;
import ohd.hseb.hefs.utils.jobs.JobMessenger;
import ohd.hseb.hefs.utils.jobs.JobMonitorAttr;
import ohd.hseb.hefs.utils.tools.FileTools;
import ohd.hseb.hefs.utils.xml.XMLTools;
import ohd.hseb.util.misc.HCalendar;
import ohd.hseb.util.misc.HStopWatch;

import org.apache.logging.log4j.Logger;
import org.apache.logging.log4j.LogManager;

import com.google.common.eventbus.EventBus;
import com.google.common.eventbus.Subscribe;

/**
 * Processor directly calls {@link WorkflowProcessor#process(File)} while executing {@link SFTPProcessor} in a separate
 * thread. The SFTP and workflow processors send messages along an internal {@link EventBus}; it does not use the event
 * bus within the {@link ParameterEstimatorRunInfo} returned by {@link #getRunInfo()}.
 * 
 * @author Hank.Herr
 */
public class ReforecastAcquisitionProcessor extends ReforecastPreparationStepProcessor implements JobListener,
SFTPFilePutInPlaceNotice.Subscriber
{
    private static final Logger LOG = LogManager.getLogger(SFTPProcessor.class);

    private final SFTPProcessor _sftpProcessor;
    private final WorkflowProcessor _workflowProcessor;

    /**
     * Made true by {@link #reactToSFTPFilePutInPlace(SFTPFilePutInPlaceNotice)}, when true, it tells this job that it
     * can now call {@link WorkflowProcessor#process(File)} to execute the workflow for one file.
     */
    private final boolean _readyToCombine = false;

    /**
     * Maintains a list of {@link #_processedFilesList}, which is also handed off to {@link #_sftpProcessor}. The list
     * is written to a system file with each change, but that writing is done manually by calling
     * {@link ProcessedFilesList#writeToSystemFile()}.
     */
    private final ProcessedFilesList _processedFilesList;

    /**
     * If not null, then {@link #_sftpProcessor} failed and the exception explains why.
     */
    private Exception _sftpProcessorJobFailedException = null;

    /**
     * Records the file being processed as notified via {@link SFTPFilePutInPlaceNotice}. This is a server-side file and
     * is the one that should be recorded in {@link #_processedFiles} after the workflow executes, if successful.
     */
    private File _fileBeingProcessed = null;

    /**
     * When true, execute the workflow.
     */
    private boolean _fileReadyToBeProcessed = false;

    /**
     * Constructor will read in {@link #_processedFilesList} or create a new file as needed and then hand it off to both
     * {@link #_sftpProcessor} and {@link #_workflowProcessor} for updating as needed.
     * 
     * @param source The {@link PluginForecastSource} associated with this processor.
     * @param instructions Instructions to follow.
     * @param processedFilesListFile The name of the {@link ProcessedFilesList} file must be constructed based on the
     *            forecast source identifier and step id and located under the system files directory.
     */
    public ReforecastAcquisitionProcessor(final PluginForecastSource source,
                                          final ReforecastAcquisitionInstructions instructions,
                                          final File processedFilesListFile)
    {
        super(source, instructions, LOG);

        //Load the processed files list if possible.
        _processedFilesList = new ProcessedFilesList();
        _processedFilesList.setOutputSystemFile(processedFilesListFile);

        //If the fix exists, can it be read and written to?  If not, that's bad.  If it can be read/written,
        //but the existing file is badly formatted, try to delete it.  If it cannot be removed, that's bad.
        if(processedFilesListFile.exists())
        {
            if(processedFilesListFile.canRead() && processedFilesListFile.canWrite())
            {
                try
                {
                    XMLTools.readXMLFromFile(processedFilesListFile, _processedFilesList);
                }
                catch(final Exception e)
                {
                    if(!processedFilesListFile.delete())
                    {
                        _processedFilesList.setOutputSystemFile(null);
                        LOG.warn("The processed list file " + processedFilesListFile.getAbsolutePath()
                            + " exists, but was invalid and cannot be removed, so it will not be used.  Cause: "
                            + e.getMessage());
                    }
                    else
                    {
                        LOG.warn("The processed list file " + processedFilesListFile.getAbsolutePath()
                            + " exists, but was invalid and was removed; cause: " + e.getMessage());
                    }
                }
            }
            else
            {
                LOG.warn("The processed list file " + processedFilesListFile.getAbsolutePath()
                    + " exists, but cannot be read from and/or written to; check permissions.");
                _processedFilesList.setOutputSystemFile(null);
            }
        }
        if(!processedFilesListFile.exists())
        {
            try
            {
                processedFilesListFile.createNewFile();
            }
            catch(final Exception e)
            {
                _processedFilesList.setOutputSystemFile(null);
                LOG.warn("An exception occurred trying to create the processed list file "
                    + processedFilesListFile.getAbsolutePath() + " so it will not be used; cause: " + e.getMessage()
                    + ".");
            }
            if(!processedFilesListFile.exists() || !processedFilesListFile.canWrite())
            {
                _processedFilesList.setOutputSystemFile(null);
                LOG.warn("The processed list file " + processedFilesListFile.getAbsolutePath()
                    + " cannot be created and will not be used.");
            }
        }

        //Initialize the sub-jobs using a mini-EventBus.
        final EventBus bus = new EventBus();
        bus.register(this);
        _sftpProcessor = new SFTPProcessor(instructions.getSftpInstructions(), bus, _processedFilesList);
        _sftpProcessor.addListener(this);
        _workflowProcessor = new WorkflowProcessor(instructions.getWorkflowInstructions(), bus, _processedFilesList);

        setName(instructions.getStepId());
    }

    /**
     * This should only be useful for testing.
     */
    protected SFTPProcessor getSFTPProcessor()
    {
        return _sftpProcessor;
    }

    /**
     * Wraps
     * {@link ReforecastPreparationStepsProcessor#determineOutputDir(String, File, String, PluginForecastSource, Logger)}
     * .
     */
    private File determineOutputDir() throws Exception
    {
        return ReforecastPreparationStepsProcessor.determineOutputDir("reforecastOutputBaseDirectory",
                                                                      getInstructions().getReforecastOutputBaseDirectory(),
                                                                      getInstructions().getStepId(),
                                                                      getSource(),
                                                                      LOG);
    }

    @Override
    public ReforecastAcquisitionInstructions getInstructions()
    {
        return (ReforecastAcquisitionInstructions)super.getInstructions();
    }

    /**
     * Run this processor to acquire reforecasts by spawning and {@link SFTPProcessor} job and {@link WorkflowProcessor}
     * , followed by a {@link ReforecastTimeSeriesFileCombiner} job. The combiner will be run even if the jobs fail in
     * order to get rid of the possibly huge number of files.
     * 
     * @throws Exception
     */
    @Override
    public void process() throws Exception
    {
        final HStopWatch timer = new HStopWatch();
        LOG.info("Starting reforecast acquisition for plugin " + getInstructions().getStepId() + ".");
        JobMessenger.updateNote("Waiting for SFTP processor to acquire file...");
        _sftpProcessor.setCanceledDoNotUpdateProgress(false);
        _sftpProcessor.setAcquireListOfFilesOnly(false);
        _sftpProcessor.setPutFileInPlace(true);
        _sftpProcessor.clearFilesToProcess();
        _sftpProcessor.startJob();

        boolean firstTimeThrough = true;
        final int combineProgressAdder = getInstructions().getCombineReforecasts() ? 1 : 0;

        //Loop until the SFTP processor stops.
        int waitTime = 0;
        while(_sftpProcessor.isJobRunning() || _fileReadyToBeProcessed)
        {
            if(JobMessenger.isCanceled())
            {
                _sftpProcessor.setCanceledAndUpdateProgress(true);
                break;
            }

            //If a file is ready to be processed.
            if(_fileReadyToBeProcessed)
            {
                //First time through, reset the progress.  All other times, just update the note.
                if(firstTimeThrough)
                {
                    //Progress is made one time for each file processed via workflow, one time for combining the forecasts (if specified),
                    //and one final time marking completion.
                    final String message = "Processing file " + _fileBeingProcessed.getName() + "...";
                    LOG.info(message);
                    JobMessenger.setJobStatus(new JobMonitorAttr(message,
                                                                 _processedFilesList.getNumberOfProcessedFiles(),
                                                                 0,
                                                                 _processedFilesList.getExpectedNumberOfFiles()
                                                                     + combineProgressAdder + 1));

                    firstTimeThrough = false;
                }
                else
                {
                    final String message = "Processing file " + _fileBeingProcessed.getName() + " (step "
                        + _processedFilesList.getNumberOfProcessedFiles() + " of "
                        + (_processedFilesList.getExpectedNumberOfFiles() + combineProgressAdder) + ")...";
                    LOG.info(message);
                    JobMessenger.updateNote(message);
                }

                //process the job.
                try
                {
                    waitTime = 0;
                    _fileReadyToBeProcessed = false;
                    _workflowProcessor.process(_fileBeingProcessed);
                }
                catch(final Exception e)
                {
                    _sftpProcessor.setCanceledDoNotUpdateProgress(true);
                    throw new Exception("Workflow processor failed; stopping reforecast acquisition process.", e);
                }

                //Made progress.
                _sftpProcessor.setPutFileInPlace(true);
                JobMessenger.madeProgress();
            }

            //Wait for one millisecond and then check to see if we've waited long enough.
            try
            {
                Thread.sleep(1L);
            }
            catch(final InterruptedException e)
            {
                e.printStackTrace();
                _sftpProcessor.setCanceledDoNotUpdateProgress(true);
                throw new Exception("Unexpected exception waiting (via sleep) for next workflow execution.");
            }
            waitTime += 1;
            if(waitTime > getInstructions().getWorkflowInstructions().getMaxWaitToExecute())
            {
                _sftpProcessor.setCanceledDoNotUpdateProgress(true);
                fireProcessJobFailure(new Exception("Wait time for next CHPS workflow execution exceeded the maximum allowed, "
                                          + getInstructions().getWorkflowInstructions().getMaxWaitToExecute()
                                          + " milliseconds."),
                                      true);
                return;
            }
        }

        //If SFTP processor failed with an error, throw an exception so that this process stops.
        if(_sftpProcessorJobFailedException != null)
        {
            throw new Exception("Stopping data acquisition process because SFTP processor failed unexpectedly: "
                + _sftpProcessorJobFailedException.getMessage());
        }

        //Job was canceled, so quit.
        if(JobMessenger.isCanceled())
        {
            throw new Exception("Reforecast acquisition processing stopping because of user cancellation.");
        }

        ////////////////////////////////////////////////
        //Process combine instruction if present.
        ////////////////////////////////////////////////
        if(getInstructions().getCombineReforecasts())
        {
            LOG.info("Combining T0-specific reforecast files into one reforecast file per location.");
            JobMessenger.updateNote("Combining T0-specific reforecast files into one reforecast file per location.");
            runTimeSeriesCombiner();
            JobMessenger.updateNote("Done combining T0-specific reforecast files into one reforecast file per location.");
            LOG.info("Done combining T0-specific reforecast files into one reforecast file per location.");
        }

        //Made progress.
        LOG.info("Done reforecast acquisition process.");
        JobMessenger.madeProgress("Done reforecast acquisition process.");

        LOG.info("Completed reforecast acquisition for step with id " + getInstructions().getStepId() + " in "
            + ((double)timer.getElapsedMillis() / HCalendar.MILLIS_IN_MIN) + " minutes.");
        endTask();
    }

    private void runTimeSeriesCombiner() throws Exception
    {
        //Determine the output directory either from the instructions or from the source data handler.
        final File reforecastOutputBaseDir = determineOutputDir();

        LOG.debug("Starting ts combine process; _readyToCombine - " + _readyToCombine);
        LOG.info("Combining output reforecast time series files, if any were generated.");

        //Combine the time series.
        try
        {
            final ReforecastTimeSeriesFileCombiner combiner = new ReforecastTimeSeriesFileCombiner(reforecastOutputBaseDir,
                                                                                                   true);
            combiner.combineTimeSeriesForAllLocations();
        }
        catch(final Exception e)
        {
            throw new Exception("Problem encountered combining time series into one file.", e);
        }

        //Made progress.
        JobMessenger.madeProgress();
    }

    @Override
    public synchronized void processJobFailure(final Exception exc,
                                               final GenericJob theJob,
                                               final boolean displayMessage) //note it is synchronized
    {
        LOG.debug("ReforecastAcquisitionProcessor received a FAILURE message from SFTPProcessor: " + exc.getMessage());
        _sftpProcessorJobFailedException = exc;
    }

    @Override
    public synchronized void processSuccessfulJobCompletion(final GenericJob theJob) //note it is synchronized
    {
        LOG.debug("ReforecastAcquisitionProcessor received a success message from SFTPProcessor.");
        _sftpProcessorJobFailedException = null;
    }

    @Override
    public boolean isStepComplete()
    {
        //Check to see if all files are processed.
        if(!_processedFilesList.haveAllFilesBeenProcessed())
        {
            return false;
        }

        //If results are to be combined, check to see if any are available for combining.
        if(getInstructions().getCombineReforecasts())
        {
            try
            {
                final File reforecastOutputBaseDir = determineOutputDir();
                if(ReforecastTimeSeriesFileCombiner.areFilesAvailableForCombining(reforecastOutputBaseDir, false))
                {
                    return false;
                }
            }
            catch(final Exception e)
            {
                //Do not log anything!  It would lead to too many messages.  Let the run action do it.
            }
        }

        //Its complete!
        return true;
    }

    @Override
    public boolean canStepBePerformed()
    {
        //Step can always be performed.
        return true;
    }

    @Override
    public JobMonitorAttr determineJobStatus(final boolean fullDetermination)
    {
        if(fullDetermination)
        {
            _sftpProcessor.clearFilesToProcess();
            _sftpProcessor.setAcquireListOfFilesOnly(true);
            _sftpProcessor.processJob(); //Do not run in a thread.
            _sftpProcessor.setAcquireListOfFilesOnly(false);
        }

        int combineStepAdder = 0;
        int progressAdder = 0;
        String combineStepAddText = "";
        if(getInstructions().getCombineReforecasts())
        {
            combineStepAdder++;
            combineStepAddText = "; includes combining results as last step";

            //Have files been combined? Only do this check if all of hte files to process have been processed.
            if(_processedFilesList.haveAllFilesBeenProcessed())
            {
                try
                {
                    //This creates a combiner solely to check for the presence of output files already created.
                    final File reforecastOutputBaseDir = determineOutputDir();
                    if(!ReforecastTimeSeriesFileCombiner.areFilesAvailableForCombining(reforecastOutputBaseDir, false))
                    {
                        progressAdder++;
                    }
                }
                catch(final Exception e)
                {
                    //IGNORE THIS...
                }
            }
        }

        final JobMonitorAttr attr = new JobMonitorAttr();
        attr.setIndeterminate(false);
        attr.setMaximum(_processedFilesList.getExpectedNumberOfFiles() + combineStepAdder);
        attr.setMinimum(0);
        attr.setProgress(_processedFilesList.getNumberOfProcessedFiles() + progressAdder);
        attr.setNote("Progress from earlier runs: ("
            + (_processedFilesList.getNumberOfProcessedFiles() + progressAdder) + " of "
            + (_processedFilesList.getExpectedNumberOfFiles() + combineStepAdder) + combineStepAddText + ")");
        return attr;
    }

    @Override
    @Subscribe
    public void reactToSFTPFilePutInPlace(final SFTPFilePutInPlaceNotice evt)
    {
        _fileReadyToBeProcessed = true;
        _fileBeingProcessed = evt.getServerFile();
    }

    @Override
    public void clearProgress()
    {
        //Clear processed file list and update the system file.
        _processedFilesList.clearProcessedFiles();
        _processedFilesList.writeToSystemFile();

        //Remove all files in the output directory that are valid files.
        try
        {
            final File outputDir = determineOutputDir();

            final File[] files = outputDir.listFiles();
            for(final File file: files)
            {
                if(file.isDirectory() && file.canWrite())
                {
                    FileTools.deleteFilesRecursively(file);
                }
                if(PluginDataHandler.isPluginDataFile(file) || file.isDirectory())
                {
                    try
                    {
                        java.nio.file.Files.delete(file.toPath());
                    }
                    catch(final IOException e)
                    {
                        LOG.warn("Step " + getInstructions().getStepId() + ": Skipping file that cannot be removed, "
                            + file.getName() + ": " + e.getMessage());
                    }
                }
            }
        }
        catch(final Exception e1)
        {
            LOG.warn("Step " + getInstructions().getStepId() + ": Unexpected problem trying to clear progress: "
                + e1.getMessage());
            return;
        }
    }
}
