package ohd.hseb.hefs.mefp.sources.plugin.steps;

import java.io.File;

import ohd.hseb.hefs.utils.xml.CompositeXMLReader;
import ohd.hseb.hefs.utils.xml.CompositeXMLWriter;
import ohd.hseb.hefs.utils.xml.XMLReadable;
import ohd.hseb.hefs.utils.xml.XMLReader;
import ohd.hseb.hefs.utils.xml.XMLReaderException;
import ohd.hseb.hefs.utils.xml.XMLWritable;
import ohd.hseb.hefs.utils.xml.XMLWriter;
import ohd.hseb.hefs.utils.xml.vars.XMLInteger;
import ohd.hseb.hefs.utils.xml.vars.XMLString;

/**
 * Instructions to be used with {@link SFTPProcessor}. Includes specifying the maximum allowable number of reconnection
 * attempts if a connection fails or is cut off, and a maximum wait time to putting an acquired file (in the tmp
 * directory) into place (i.e., waiting for the {@link WorkflowProcessor} to complete an execution of the workflow).
 * Also, server and server-file information must be specified.
 * 
 * @author hankherr
 */
public class SFTPInstructions implements XMLReadable, XMLWritable, Cloneable
{

    private final XMLInteger _maxReconnects = new XMLInteger("maxReconnects", 10, 0, null);
    private final XMLInteger _maxWaitForPuttingFileInPlace = new XMLInteger("maxWaitForPuttingFileInPlace",
                                                                            1000000,
                                                                            0,
                                                                            null);

    private final XMLString _serverURL = new XMLString("serverURL");
    private final XMLString _userName = new XMLString("userName");
    private final XMLString _password = new XMLString("password");

    /**
     * This should always be processed {@link File#getPath()}, NOT {@link File#getAbsolutePath()}.
     */
    private final XMLString _serverDirectory = new XMLString("serverDirectory");
    private final XMLString _fileNamePattern = new XMLString("fileNamePattern");

    /**
     * This should always be processed {@link File#getAbsolutePath()}.
     */
    private final XMLString _localDirectory = new XMLString("localDirectory");

    public SFTPInstructions()
    {
        _serverURL.setRequired(true);
        _userName.setRequired(true);
        _password.setRequired(true);
        _serverDirectory.setRequired(true);
        _fileNamePattern.setRequired(true);
        _localDirectory.setRequired(true);
    }

    /**
     * Used within {@link ReforecastAcquisitionInstructions#getCombineReforecasts()}.
     * 
     * @return
     */
    public String getSummary()
    {
        String summary = "Server: " + _serverURL.get() + "\n";
        summary += "Login: " + _userName.get() + " (password will not be shown)\n";
        summary += "Directory on Server: " + _serverDirectory.get() + "\n";
        summary += "File Name Pattern: " + _fileNamePattern.get() + "\n";
        summary += "Local Directory: " + _localDirectory.get() + "\n";
        return summary;
    }

    public String getServerURL()
    {
        return _serverURL.get();
    }

    public void setServerURL(final String serverURL)
    {
        _serverURL.set(serverURL);
    }

    public String getUserName()
    {
        return _userName.get();
    }

    public void setUserName(final String userName)
    {
        _userName.set(userName);
    }

    public String getPassword()
    {
        return _password.get();
    }

    public void setPassword(final String password)
    {
        _password.set(password);
    }

    public String getServerDirectory()
    {
        return _serverDirectory.get();
    }

    public void setServerDirectory(final String serverDirectory)
    {
        _serverDirectory.set(serverDirectory);
    }

    public String getLocalDirectory()
    {
        return _localDirectory.get();
    }

    public void setLocalDirectory(final String dir)
    {
        _localDirectory.set(dir);
    }

    public String getFileNamePattern()
    {
        return _fileNamePattern.get();
    }

    public void setFileNamePattern(final String fileNamePattern)
    {
        _fileNamePattern.set(fileNamePattern);
    }

    public String getXMLTagName()
    {
        return "sftpInstructions";
    }

    public int getMaxReconnects()
    {
        return _maxReconnects.get();
    }

    public void setMaxReconnects(final int n)
    {
        _maxReconnects.set(n);
    }

    public int getMaxWaitForPuttingFileInPlace()
    {
        return _maxWaitForPuttingFileInPlace.get();
    }

    public void setMaxWaitForPuttingFileInPlace(final int n)
    {
        _maxWaitForPuttingFileInPlace.set(n);
    }

    @Override
    public XMLWriter getWriter()
    {
        final CompositeXMLWriter writer = new CompositeXMLWriter(getXMLTagName(),
                                                                 _serverURL,
                                                                 _userName,
                                                                 _password,
                                                                 _serverDirectory,
                                                                 _fileNamePattern,
                                                                 _localDirectory);
        writer.addAttribute(_maxReconnects, false);
        writer.addAttribute(_maxWaitForPuttingFileInPlace, false);
        return writer;
    }

    @Override
    public XMLReader getReader()
    {
        final CompositeXMLReader reader = new CompositeXMLReader(getXMLTagName(),
                                                                 _serverURL,
                                                                 _userName,
                                                                 _password,
                                                                 _serverDirectory,
                                                                 _fileNamePattern,
                                                                 _localDirectory)
        {
            @Override
            public void finalizeReading() throws XMLReaderException
            {
                super.finalizeReading();
            }
        };
        reader.addAttribute(_maxReconnects, false);
        reader.addAttribute(_maxWaitForPuttingFileInPlace, false);
        return reader;
    }

    @Override
    public boolean equals(final Object obj)
    {
        final SFTPInstructions other = (SFTPInstructions)obj;
        return ((_serverURL.equals(other._serverURL)) && (_userName.equals(other._userName))
            && (_password.equals(other._password)) && (_serverDirectory.equals(other._serverDirectory))
            && (_fileNamePattern.equals(other._fileNamePattern)) && (_localDirectory.equals(other._localDirectory))
            && (_maxReconnects.equals(other._maxReconnects)) && (_maxWaitForPuttingFileInPlace.equals(other._maxWaitForPuttingFileInPlace)));
    }

    @Override
    public SFTPInstructions clone()
    {
        final SFTPInstructions cloned = new SFTPInstructions();
        cloned._serverURL.set(getServerURL());
        cloned._userName.set(getUserName());
        cloned._password.set(getPassword());
        cloned._serverDirectory.set(getServerDirectory());
        cloned._fileNamePattern.set(getFileNamePattern());
        cloned._localDirectory.set(getLocalDirectory());
        cloned._maxReconnects.set(getMaxReconnects());
        cloned._maxWaitForPuttingFileInPlace.set(getMaxWaitForPuttingFileInPlace());
        return cloned;
    }
}
