package ohd.hseb.hefs.mefp.sources.plugin.steps;

import java.io.File;
import java.util.List;

import ohd.hseb.hefs.mefp.sources.plugin.ProcessedFilesList;
import ohd.hseb.hefs.utils.TempFile;
import ohd.hseb.hefs.utils.ftp.SFTPConductor;
import ohd.hseb.hefs.utils.jobs.GenericJob;
import ohd.hseb.hefs.utils.jobs.JobListener;
import ohd.hseb.hefs.utils.log4j.LoggingTools;

import org.apache.logging.log4j.Logger;
import org.apache.logging.log4j.LogManager;

import com.google.common.eventbus.EventBus;
import com.google.common.eventbus.Subscribe;

/**
 * Process the SFTP process specified by {@link SFTPInstructions}. This must be made a listener of a
 * {@link WorkflowProcessor} in order for it to shutdown if the workflow processor dies.
 * 
 * @author Hank.Herr
 */
public class SFTPProcessor extends GenericJob implements WorkflowProcessorReadyForFileNotice.Subscriber, JobListener
{
    private static final Logger LOG = LogManager.getLogger(SFTPProcessor.class);

    /**
     * Instructions dictating what will be done within this processor.
     */
    private final SFTPInstructions _instructions;

    /**
     * Used for signalling between this and a job processing the CHPS workflow execution.
     */
    private final EventBus _bus;

    /**
     * Maintains a list of the files already processed, so that work is not repeated. Also, this list is updated
     * whenever the {@link #_filesToProcess} object is updated; i.e., at the beginning of a run.
     */
    private final ProcessedFilesList _processedFilesList;

    /**
     * Populated first by the {@link #processJob()} method. If {@link #_acquireListOfFilesOnly} is set to true, then the method
     * will stop after populating this object. Call {@link #getFilesToProcess()} to get the resulting list.<br>
     * <br>
     * If this list is null, then the files to process has not been determined yet.
     */
    private List<File> _filesToProcess = null;

    /**
     * Set to true to indicate all you want is a list of the files to process.
     */
    private boolean _acquireListOfFilesOnly = false;

    /**
     * Set to true to let this know its okay to put a {@link TempFile} acquired via SFTP into its permanent location.
     * This is true to start with so that the first file can be acquired and put in place.
     */
    private boolean _putFileInPlace = true;

    private int _testFileCount = -1;

    /**
     * @param instructions Instructions specifying the SFTP to perform.
     * @param bus {@link EventBus} to which {@link SFTPFilePutInPlaceNotice} instances are posted and from which
     *            {@link WorkflowProcessorReadyForFileNotice} notices are received.
     */
    public SFTPProcessor(final SFTPInstructions instructions,
                         final EventBus bus,
                         final ProcessedFilesList processedFilesList)
    {
        _instructions = instructions;
        _bus = bus;
        _bus.register(this);
        _processedFilesList = processedFilesList;
        this.setName("SFTPProcessor");
    }

    /**
     * Sets {@link #_acquireListOfFilesOnly}; see the attribute description.
     * 
     * @param b True if you only want a list of files (call {@link #getFilesToProcess()} after the job is done) or false
     *            if you want the full run.
     */
    public void setAcquireListOfFilesOnly(final boolean b)
    {
        _acquireListOfFilesOnly = b;
    }

    public List<File> getFilesToProcess()
    {
        return _filesToProcess;
    }

    /**
     * Call to force a full re-acquisition of files list; useful when updating status.
     */
    public void clearFilesToProcess()
    {
        _filesToProcess = null;
    }

    /**
     * Sets a count on the number of files to acquire. Useful only for testing purposes.
     * 
     * @param count The number of files to acquire.
     */
    public void setTestFileCount(final int count)
    {
        _testFileCount = count;
    }

    /**
     * @param b If true, then the {@link TempFile} acquired by {@link SFTPConductor} will be put in its permanent
     *            (presumably import) location.
     */
    public void setPutFileInPlace(final boolean b)
    {
        _putFileInPlace = b;
    }

    @Override
    public void processJob()
    {
        int failureCount = 0;
        SFTPConductor conductor = null;
        TempFile acquiredFile = null;
        while(failureCount < _instructions.getMaxReconnects())
        {
            try
            {
                //Establish the connection.
                conductor = new SFTPConductor(_instructions.getServerURL(),
                                              _instructions.getUserName(),
                                              _instructions.getPassword());
                if(failureCount == _instructions.getMaxReconnects())
                {
                    throw new IllegalStateException("THIS SHOULD NOT HAVE HAPPENED: Failure count exception should have been caught and resulted in a job failure.");
                }

                //Acquire the files.
                //Only acquire the files to process if they have not been acquired yet.
                if(_filesToProcess == null)
                {
                    _filesToProcess = conductor.listFilesInDirectory(_instructions.getServerDirectory(),
                                                                     _instructions.getFileNamePattern());
                    _processedFilesList.updateProcessedFilesListAgainstFilesToProcess(_filesToProcess);
                    _processedFilesList.writeToSystemFile();
                }

                //If the flag is set, then we are just acquiring a list of the files to process.
                if(_acquireListOfFilesOnly)
                {
                    endTask();
                    return;
                }

                //For each file to process.  Note that the file's path will provide the complete path on the server, by construction.  
                //Do NOT call file.getAbsolutePath() or it will locate the path on this machine.
                int processedFileCount = 0;
                for(int index = 0; index < _filesToProcess.size(); index++)
                {
                    final File file = _filesToProcess.get(index);
                    final boolean lastFile = (index == _filesToProcess.size() - 1);

                    //Standard checks to stop the process.
                    if(this.isCanceled())
                    {
                        endTask();
                        return;
                    }

                    //Skip the file if it has already been processed.
                    if(_processedFilesList.containsFile(file))
                    {
                        continue;
                    }

                    LOG.info("Processing server file " + file.getPath() + "...");
                    acquiredFile = conductor.getTempFileFromServer(new File(_instructions.getLocalDirectory()).getAbsolutePath()
                                                                       + "/" + file.getName(),
                                                                   file.getPath());

                    //Wait for a signal to put the file in place.  
                    int waitCountMillis = 0;
                    while((!_putFileInPlace) && (waitCountMillis < _instructions.getMaxWaitForPuttingFileInPlace()))
                    {
                        Thread.sleep(1); //hundredth of a second wait time between checks.
                        waitCountMillis += 1;

                        //Standard checks to stop the process.
                        if(this.isCanceled())
                        {
                            endTask();
                            return;
                        }
                    }

                    //Waited too long.  Failing out altogether.
                    if(waitCountMillis >= _instructions.getMaxWaitForPuttingFileInPlace())
                    {
                        fireProcessJobFailure(new Exception("File was acquired but signal to put the file in place for import was never caught after waiting "
                                                  + waitCountMillis + " milliseconds.  Failing out."),
                                              true);
                        return;
                    }

                    //Put the file in place and send out a notice on the event bus.
                    LOG.info("Copying temporary SFTP file into place as " + acquiredFile.getTarget());
                    acquiredFile.move();
                    _putFileInPlace = false;
                    processedFileCount++;
                    _bus.post(new SFTPFilePutInPlaceNotice(this, file, lastFile));

                    //TESTING PURPOSES ONLY!!!
                    //Exit out if test file count is set and count meets or exceeds it.
                    if((_testFileCount >= 0) && (processedFileCount >= _testFileCount))
                    {
                        endTask();
                        return;
                    }
                }

                //Successful finish.  All files processed.
                endTask();
                return;
            }
            //Too many connection failures.
            catch(final IllegalStateException e)
            {
                throw e;
            }
            //Any other exception should increase the failureCount, but otherwise allow for processing to commence once a new connection is established.
            catch(final Exception e)
            {
                failureCount++;
                LOG.warn("Failed to connect to SFTP server: " + e.getMessage());
                LoggingTools.outputStackTraceAsDebug(LOG, e);

                //Close any connection.  A new connection will be attempted to see if that fixes things.
                if(conductor != null)
                {
                    conductor.closeConnection();
                }

                if(failureCount < _instructions.getMaxReconnects())
                {
                    LOG.warn("This is failure # " + failureCount + ".  Data acquisition will stop after "
                        + _instructions.getMaxReconnects() + " failures.");
                    _putFileInPlace = true; //Assume that a file can be put in place.
                }
                else
                {
                    fireProcessJobFailure(new Exception("SFTP processor failed " + failureCount
                        + " times.  Quitting SFTP process.  Message: " + e.getMessage()), true);
                    return;
                }
            }
            //ALWAYS: Close the SFTP connection and delete any hanging temp file.
            finally
            {
                LOG.info("Closing SFTP connection.");
                if(conductor != null)
                {
                    conductor.closeConnection();
                }
                if(acquiredFile != null)
                {
                    acquiredFile.delete();
                }
            }

        }
    }

    /**
     * React to the workflow component posting a {@link WorkflowProcessorReadyForFileNotice}.
     */
    @Override
    @Subscribe
    public void reactToWorkflowProcessorReadyForFile(final WorkflowProcessorReadyForFileNotice evt)
    {
        _putFileInPlace = true;
    }

    @Override
    public void processJobFailure(final Exception exc, final GenericJob theJob, final boolean displayMessage)
    {
        setCanceledDoNotUpdateProgress(true);
    }

    @Override
    public void processSuccessfulJobCompletion(final GenericJob theJob)
    {
        setCanceledDoNotUpdateProgress(true);
    }
}
