package ohd.hseb.hefs.mefp.sources.plugin.steps;

import java.util.ArrayList;
import java.util.List;

import ohd.hseb.hefs.mefp.sources.plugin.AbstractReforecastPreparationStepInstructions;
import ohd.hseb.hefs.mefp.sources.plugin.PluginForecastSource;
import ohd.hseb.hefs.mefp.sources.plugin.ReforecastPreparationSteps;
import ohd.hseb.hefs.utils.xml.CollectionXMLReader;
import ohd.hseb.hefs.utils.xml.CollectionXMLWriter;
import ohd.hseb.hefs.utils.xml.XMLReader;
import ohd.hseb.hefs.utils.xml.XMLReaderFactory;
import ohd.hseb.hefs.utils.xml.XMLWriter;
import ohd.hseb.hefs.utils.xml.vars.XMLString;

import com.google.common.base.Function;
import com.google.common.collect.Lists;

/**
 * Instructions specify a system call to make as a step within a plugin {@link ReforecastPreparationSteps}. It stores a
 * command, {@link #_systemCommand}, and a list of command line arguments, {@link #_arguments}.
 * 
 * @author hankherr
 */
public class SystemCallInstructions extends AbstractReforecastPreparationStepInstructions
{

    /**
     * The command to execute.
     */
    private final XMLString _systemCommand = new XMLString("systemCommand");

    /**
     * The arguments; order is important, obviously, since the arguments are passed to a command.
     */
    private final List<XMLString> _arguments = new ArrayList<>();

    public String getSystemCommand()
    {
        return _systemCommand.get();
    }

    public void setSystemCommand(final String systemCommand)
    {
        _systemCommand.set(systemCommand);
    }

    /**
     * @return A {@link List} of {@link String} instances acquired from the stored {@link #_arguments}.
     */
    public List<String> getArguments()
    {
        return Lists.transform(_arguments, new Function<XMLString, String>()
        {
            @Override
            public String apply(final XMLString arg0)
            {
                return arg0.get();
            }
        });
    }

    public void addArgument(final String argument)
    {
        _arguments.add(new XMLString(getListXMLTagName(), argument));
    }

    public void clearArguments()
    {
        _arguments.clear();
    }

    private String getListXMLTagName()
    {
        return "argument";
    }

    @Override
    public String getXMLTagName()
    {
        return "systemCallInstructions";
    }

    @Override
    public SystemCallProcessor createProcessor(final PluginForecastSource source)
    {
        return new SystemCallProcessor(source, this);
    }

    @Override
    public String getInstructionSummaryForInterface()
    {
        String summary = "";
        summary += "System Command: " + _systemCommand.get() + "\n";
        for(final XMLString arg: _arguments)
        {
            summary += "Argument: " + arg.get() + "\n";
        }
        return summary;
    }

    @Override
    public XMLReader getReader()
    {
        final CollectionXMLReader<XMLString> reader = new CollectionXMLReader<>(getXMLTagName(),
                                                                                _arguments,
                                                                                new XMLReaderFactory<XMLString>()
                                                                                {
                                                                                    @Override
                                                                                    public XMLString get()
                                                                                    {
                                                                                        return new XMLString(getListXMLTagName());
                                                                                    }
                                                                                });
        addReforecastPreparationStepInstructionsAttributes(reader);
        reader.addAttribute(_systemCommand, true);
        return reader;
    }

    @Override
    public XMLWriter getWriter()
    {
        final CollectionXMLWriter writer = new CollectionXMLWriter(getXMLTagName(), _arguments);
        addReforecastPreparationStepInstructionsAttributes(writer);
        writer.addAttribute(_systemCommand, true);
        return writer;
    }

    @Override
    public boolean equals(final Object obj)
    {
        if(!super.equals(obj))
        {
            return false;
        }
        final SystemCallInstructions other = (SystemCallInstructions)obj;

        if(!_systemCommand.equals(other._systemCommand))
        {
            return false;
        }

        return _arguments.equals(other._arguments);
    }

    @Override
    public SystemCallInstructions clone()
    {
        final SystemCallInstructions cloned = new SystemCallInstructions();
        cloned.copyFrom(this);
        for(final XMLString arg: _arguments)
        {
            cloned.addArgument(arg.get());
        }
        cloned._systemCommand.set(_systemCommand.get());
        return cloned;
    }

}
