package ohd.hseb.hefs.mefp.sources.plugin.steps;

import java.io.File;
import java.util.Calendar;
import java.util.Date;

import nl.wldelft.fews.system.pi.FewsPiService;
import ohd.hseb.hefs.mefp.sources.plugin.ProcessedFilesList;
import ohd.hseb.hefs.utils.piservice.FewsPiServiceProvider;
import ohd.hseb.util.misc.HCalendar;
import ohd.hseb.util.misc.HStopWatch;
import ohd.hseb.util.misc.SegmentedLine;

import org.apache.logging.log4j.Logger;
import org.apache.logging.log4j.LogManager;

import com.google.common.eventbus.EventBus;

/**
 * Used to call the workflow to process within a {@link ReforecastAcquisitionProcessor}.
 * 
 * @author Hank.Herr
 */
public class WorkflowProcessor
{
    private static final Logger LOG = LogManager.getLogger(WorkflowProcessor.class);

    private final WorkflowInstructions _instructions;

    /**
     * Used for signalling between this and a job processing the SFTP.
     */
    private final EventBus _bus;

    /**
     * A list of the files already processed.
     */
    private final ProcessedFilesList _processedFiles;

    /**
     * @param instructions Instructions supplying the workflow id.
     * @param bus The {@link EventBus} on which an {@link SFTPProcessor} will be posting notices letting this know when
     *            it call the workflow. Also, when this is done with a workflow, it will post
     *            {@link WorkflowProcessorReadyForFileNotice} notices to the bus letting the SFTP tool know it can now
     *            put a file in place.
     * @param processedFiles List of files processed, which is stored externally because it will be written to a file as
     *            it is modified.
     */
    public WorkflowProcessor(final WorkflowInstructions instructions,
                             final EventBus bus,
                             final ProcessedFilesList processedFiles)
    {
        _instructions = instructions;
        _bus = bus;
        _bus.register(this);
        _processedFiles = processedFiles;
    }

    /**
     * Method calls
     * {@link FewsPiService#runTask(String, String, String, Date, Date, Date, String, String, String, String)} for the
     * return of {@link FewsPiServiceProvider#getDefaultService()}, and other methods as needed. After success, it will
     * post a {@link WorkflowProcessorReadyForFileNotice} notice to the {@link #_bus} so that the {@link SFTPProcessor}
     * knows it can put the next file into place.
     * 
     * @param fileBeingProcessed The file being processed, mainly used for messaging.
     * @throws Exception If a problem occurs.
     */
    public void process(final File fileBeingProcessed) throws Exception
    {
        final FewsPiService piService = FewsPiServiceProvider.getDefaultService();
        if(piService == null)
        {
            throw new Exception("Unable to connect to FEWS PI-service.");
        }

        //The file name must have multiple components separated by '_' or '.'...
        final SegmentedLine segLine = new SegmentedLine(fileBeingProcessed.getName(),
                                                        "_.",
                                                        SegmentedLine.MODE_NO_EMPTY_SEGS);
        if(segLine.getNumberOfSegments() >= 2)
        {
            //The next to last component must be a date string of format 'CCYYMMDDhh', with the last component being the file extension.
            final String dateStr = segLine.getSegment(segLine.getNumberOfSegments() - 2);
            final Calendar cal = HCalendar.convertStringToCalendar(dateStr, "CCYYMMDDhh");
            if(cal != null)
            {
                try
                {
                    LOG.debug("Executing CHPS workflow to process file " + fileBeingProcessed.getPath()
                        + " with T0 of " + HCalendar.buildDateTimeTZStr(cal) + "...");
                    final HStopWatch timer = new HStopWatch();
                    final Date date = cal.getTime(); //GraphGen is filler and should matter, I hope.
                    final String taskId = FewsPiServiceProvider.getDefaultService().createTask("GraphGen"); //GraphGen is filler and should matter, I hope.
                    final String runId = FewsPiServiceProvider.getDefaultService()
                                                              .runTask("GraphGen", //FILLER -- shouldn't matter, I hope
                                                                       taskId,
                                                                       _instructions.getWorkflowID(),
                                                                       date,
                                                                       date,
                                                                       date,
                                                                       "",
                                                                       "", //FILLER -- shouldn't matter
                                                                       "",
                                                                       "Plug-in Import of WPC Forecast");
                    final boolean success = FewsPiServiceProvider.getDefaultService()
                                                                 .waitForTask("GraphGen",
                                                                              runId,
                                                                              _instructions.getMaxWaitForExecutionToComplete()); //GraphGen is filler and should matter, I hope.
                    if(success)
                    {
                        LOG.info("**** CHPS Workflow executed successfully in " + timer.getElapsedMillis()
                            + " milliseconds for the file " + fileBeingProcessed.getPath() + ".");
                        _processedFiles.addProcessedFile(fileBeingProcessed);
                        timer.reset();
                        _processedFiles.writeToSystemFile();
                        LOG.debug("Updated system file with new processed files list in " + timer.getElapsedMillis()
                            + " milliseconds.");

                    }
                    else
                    {
                        LOG.warn("CHPS Workflow FAILED to execute in " + timer.getElapsedMillis() + " for the file "
                            + fileBeingProcessed.getPath() + ".  Continuing to the next file when available.");
                    }
                }
                catch(final Throwable t)
                {
                    throw new Exception("Unexpected problem occurred within PI-service.", t);
                }
            }
            else
            {
                LOG.warn("File to process "
                    + fileBeingProcessed.getName()
                    + " is not properly named and will be skipped: the last component of the name BEFORE the extension (where components are separated by '.' or '_')"
                    + " is not a date string of format 'CCYYMMDDhh'.");
            }
        }
        else
        {
            LOG.warn("File to process "
                + fileBeingProcessed.getName()
                + " is not properly named and will be skipped: the name must include multiple parts separated by '.' or '_'.");
        }

        //Post the notice that the next file can now be processed.
        _bus.post(new WorkflowProcessorReadyForFileNotice(this));
    }

}
