package ohd.hseb.hefs.mefp.sources.rfcfcst;

import java.util.concurrent.TimeUnit;

import net.jcip.annotations.Immutable;
import nl.wldelft.util.timeseries.TimeSeriesHeader;
import ohd.hseb.hefs.pe.tools.LocationAndDataTypeIdentifier;
import ohd.hseb.hefs.utils.Duration;
import ohd.hseb.hefs.utils.tools.ParameterId;
import ohd.hseb.hefs.utils.xml.XMLReader;
import ohd.hseb.hefs.utils.xml.XMLReaderException;

import org.xml.sax.Attributes;

import com.google.common.base.Supplier;

/**
 * SHEF pedtsep info holder. Uses '_' to specify unset values.
 * 
 * @author alexander.garbarino
 */
@Immutable
public class PedtsepCode extends PedtsepCodeBase implements Cloneable
{
    public static final String XML_TAG = "pedtsepCode";

    private final String _ts;

    private PedtsepCode()
    {
        super(null, null, null, null);
        _ts = null;
    }

    public PedtsepCode(final String pe, final String d, final String ts, final String e, final String p)
    {
        super(processPE(pe), processD(d), processE(e), processP(p));
        _ts = processTS(ts);
    }

    /**
     * Create a code from the given code string.
     * 
     * @param string the string to make the code from
     */
    public PedtsepCode(final String string)
    {
        this(processPE(string.substring(0, 2)),
             processD(string.substring(2, 3)),
             processTS(string.substring(3, 5)),
             processE(string.substring(5, 6)),
             processP(string.substring(6, 7)));
    }

    /**
     * Create a code from the given code string
     * 
     * @param code the string to make the code from
     * @throws IllegalArgumentException if the code is not 7 characters long
     */
    public static PedtsepCode make(final String code)
    {
        if(code.length() != 7)
        {
            throw new IllegalArgumentException("Code must be 7 characters long, not " + code.length() + ".");
        }

        return new PedtsepCode(code);
    }

    /**
     * Makes a pedtsep code for the given series header.
     * 
     * @param header the series header
     */
    public static PedtsepCode make(final TimeSeriesHeader header)
    {
        PedtsepCode code = new PedtsepCode();

        final ParameterId parameterId = ParameterId.of(header);
        if(parameterId.isPrecipitation())
        {
            code = code.withPE("PP");
        }
        else if(parameterId.isTemperature())
        {
            code = code.withPE("TA");
        }
        else
        {
            throw new IllegalArgumentException("Unknown Parameter Type.");
        }

        try
        {
            final Duration duration = new Duration(header.getTimeStep().getStepMillis(), TimeUnit.MILLISECONDS);
            final Pedtsep.D d = Pedtsep.D.forDuration(duration);
            if(d != null)
            {
                code = code.withD(d.toString());
            }
            else
            {
                throw new IllegalArgumentException("Illegal duration.");
            }
        }
        catch(final UnsupportedOperationException e) // Irregular Timestep.
        {
        }

        if(parameterId.isObserved())
        {
            code = code.withTS("RA");
        }
        else
        {
            code = code.withTS("FA");
        }

        if(parameterId.isMax())
        {
            code = code.withE("X");
        }
        else if(parameterId.isMin())
        {
            code = code.withE("N");
        }
        else
        {
            code = code.withE("Z");
        }

        code = code.withP("Z");

        return code;
    }

    public static PedtsepCode make(final LocationAndDataTypeIdentifier identifier)
    {
        return make(identifier.makeHeader());
    }

    public PedtsepCode withPE(final String pe)
    {

        return new PedtsepCode(processPE(pe), getD(), _ts, getE(), getP());
    }

    public PedtsepCode withD(final String d)
    {
        return new PedtsepCode(getPE(), processD(d), _ts, getE(), getP());
    }

    public PedtsepCode withTS(final String ts)
    {
        return new PedtsepCode(getPE(), getD(), processTS(ts), getE(), getP());
    }

    public PedtsepCode withE(final String e)
    {
        return new PedtsepCode(getPE(), getD(), _ts, processE(e), getP());
    }

    public PedtsepCode withP(final String p)
    {
        return new PedtsepCode(getPE(), getD(), _ts, getE(), processP(p));
    }

    /**
     * Gives the actual SHEF code.
     */
    @Override
    public String toString()
    {
        String rep = "";
        rep += getPE() != null ? getPE() : "__";
        rep += getD() != null ? getD() : "_";
        rep += getTS() != null ? getTS() : "__";
        rep += getE() != null ? getE() : "_";
        rep += getP() != null ? getP() : "_";
        return rep;
    }

    /**
     * @return the parameter id that this code would have
     */
    public ParameterId asParameterId()
    {
        final ParameterId.Type type = getType();
        final ParameterId.Extremum extremum = getExtremum();
        return ParameterId.get(type, isForecast(), extremum);
    }

    public boolean isForecast()
    {
        return _ts != null && _ts.startsWith("F");
    }

    public boolean isObserved()
    {
        return _ts != null && _ts.startsWith("R");
    }

    public String getTS()
    {
        return _ts;
    }

    @Override
    public String getXMLTagName()
    {
        return "pedtsepCode";
    }

    public static class Reader implements XMLReader, Supplier<PedtsepCode>
    {
        private PedtsepCode _code;

        public Reader()
        {
            _code = new PedtsepCode();
        }

        @Override
        public PedtsepCode get()
        {
            return _code;
        }

        @Override
        public XMLReader getReader()
        {
            return this;
        }

        @Override
        public String getXMLTagName()
        {
            return XML_TAG;
        }

        @Override
        public void setValueOfElement(final String elementName, final String value) throws XMLReaderException
        {
            if(!elementName.equals(getXMLTagName()))
            {
                throw new XMLReaderException("Unknown tag " + elementName);
            }
            _code = PedtsepCode.make(value);
        }

        @Override
        public XMLReader readInPropertyFromXMLElement(final String elementName, final Attributes attr) throws XMLReaderException
        {
            return null;
        }

        @Override
        public void finalizeReading() throws XMLReaderException
        {
        }

        @Override
        public void validate() throws XMLReaderException
        {
        }
    }
}
