package ohd.hseb.hefs.mefp.sources.rfcfcst;

import java.io.File;
import java.io.FileWriter;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.List;
import java.util.NoSuchElementException;
import java.util.Properties;
import java.util.Scanner;

import nl.wldelft.util.timeseries.DefaultTimeSeriesHeader;
import nl.wldelft.util.timeseries.ParameterType;
import nl.wldelft.util.timeseries.SimpleEquidistantTimeStep;
import nl.wldelft.util.timeseries.TimeSeriesArray;
import ohd.hseb.hefs.pe.tools.LocationAndDataTypeIdentifier;
import ohd.hseb.util.misc.HCalendar;
import ohd.hseb.util.misc.SegmentedLine;

import org.apache.logging.log4j.Logger;
import org.apache.logging.log4j.LogManager;

public abstract class PreparedRFCFileTools
{
    private static final Logger LOG = LogManager.getLogger(PreparedRFCFileTools.class);

    private static final float MISSING_PRECIP = -99.99f;
    private static final float MISSING_TEMP = -99.0f;

    private static void setHeaderLocation(final DefaultTimeSeriesHeader header, final String location)
    {
        header.setLocationDescription(location);
        header.setLocationId(location);
        header.setLocationName(location);
    }

    private static void setHeaderParameter(final DefaultTimeSeriesHeader header, final String parameter)
    {
        header.setParameterId(parameter);
        header.setParameterName(parameter);
        header.setParameterType(ParameterType.ACCUMULATIVE);
    }

    private static DefaultTimeSeriesHeader makePrecipitationHeader()
    {
        final DefaultTimeSeriesHeader header = new DefaultTimeSeriesHeader();
        header.setTimeStep(SimpleEquidistantTimeStep.getInstance(HCalendar.MILLIS_IN_HR * 6));
        return header;
    }

    private static DefaultTimeSeriesHeader makeTemperatureHeader()
    {
        final DefaultTimeSeriesHeader header = new DefaultTimeSeriesHeader();
        header.setTimeStep(SimpleEquidistantTimeStep.getInstance(HCalendar.MILLIS_IN_HR * 24));
        header.setUnit("DEGC");
        return header;
    }

    /**
     * @param scan Scanner connected to the file being read.
     * @return A Java {@link Properties} instance that contains the properties found. Keys will be in all lower case and
     *         the value will be trimmed of white space before and after.
     */
    private static Properties readHeader(final Scanner scan) throws Exception
    {
        scan.useDelimiter("\n");
        boolean done = false;
        String nextLine;
        final SegmentedLine segLine = new SegmentedLine("=", SegmentedLine.MODE_NO_EMPTY_SEGS);
        final Properties props = new Properties();

        //Process until "end" is found or scan returns null.
        //Each line provides a property: key = value.
        while(!done)
        {
            try
            {
                nextLine = scan.next();
            }
            catch(final NoSuchElementException e)
            {
                throw new Exception("Unexpected end of elements in header; header may not include an 'end': "
                    + e.getMessage());
            }
            if((nextLine == null) || (nextLine.trim().equalsIgnoreCase("end")))
            {
                done = true;
            }
            else
            {
                segLine.segmentLine(nextLine);
                if(segLine.getNumberOfSegments() == 2)
                {
                    props.setProperty(segLine.getSegment(0).trim().toLowerCase(), segLine.getSegment(1).trim());
                }
            }
        }
        return props;
    }

    public static List<TimeSeriesArray> readPrecipitationForecastFile(final File forecastFile,
                                                                      final LocationAndDataTypeIdentifier identifier,
                                                                      final String paramId) throws Exception
    {
        final List<TimeSeriesArray> seriesList = new ArrayList<TimeSeriesArray>();

        final String location = identifier.getLocationId();

        // Read File Header.
        final Scanner scan = new Scanner(forecastFile);
        try
        {
            scan.useDelimiter("\n");
            final Properties props = readHeader(scan);
            final String unit = props.getProperty("units");
            float unitConversionFactor = 1.0f;
            if(unit.trim().equalsIgnoreCase("in"))
            {
                unitConversionFactor = 25.4f;
            }

            Scanner lineScanner;
            String nextLine;

            // Read File Body.
            while(scan.hasNext()) // Start each entry with the date. 
            {
                nextLine = scan.next();
                lineScanner = new Scanner(nextLine);
                lineScanner.reset();

                // Prepare Header
                final String date = lineScanner.next(); // Read date.
                final Calendar t0 = HCalendar.convertStringToCalendar(date + " 12:00:00", "CCYYMMDD hh:mm:ss");
                final DefaultTimeSeriesHeader header = makePrecipitationHeader();
                setHeaderLocation(header, location);
                setHeaderParameter(header, paramId);
                header.setUnit("MM");
                header.setForecastTime(t0.getTimeInMillis());

                // Create Time Series.
                final TimeSeriesArray ts = new TimeSeriesArray(header);
                ts.setForecastTime(header.getForecastTime());

                int i = 1;
                while(lineScanner.hasNextFloat()) // Starts at next timestep from start time.
                {
                    float value = lineScanner.nextFloat(); //account for unit conv.
                    if((value != MISSING_PRECIP) && (!Float.isNaN(value)))
                    {
                        value = value * unitConversionFactor;
                        ts.putValue(header.getForecastTime() + i * header.getTimeStep().getStepMillis(), value);
                    }
                    else
                    {
                        ts.putValue(header.getForecastTime() + i * header.getTimeStep().getStepMillis(), Float.NaN);
                    }
                    i++;
                }

                if((ts.size() > 0) && (!ts.isCompletelyMissing()))
                {
                    seriesList.add(ts);
                }
            }
        }
        catch(final Exception e)
        {
            e.printStackTrace();
            throw new Exception("Failure while reading file: " + e.getMessage());
        }
        finally
        {
            scan.close();
        }

        return seriesList;
    }

    public static List<TimeSeriesArray> readTemperatureForecastFile(final File forecastFile,
                                                                    final LocationAndDataTypeIdentifier identifier,
                                                                    final String paramId) throws Exception
    {
        final List<TimeSeriesArray> seriesList = new ArrayList<TimeSeriesArray>();

        // Ignore Header
        final Scanner scan = new Scanner(forecastFile);

        try
        {
            scan.nextLine();
            scan.nextLine();

            // Read File Body.
            while(scan.hasNext("\\d{8}")) // Start each entry with the date
            {
                // Prepare Header
                final String date = scan.next();
                final Calendar t0 = HCalendar.convertStringToCalendar(date + " 12:00:00", "CCYYMMDD hh:mm:ss");
                final DefaultTimeSeriesHeader header = makeTemperatureHeader();
                setHeaderLocation(header, identifier.getLocationId());
                setHeaderParameter(header, paramId);
                header.setForecastTime(t0.getTimeInMillis());

                // Create Time Series.
                final TimeSeriesArray ts = new TimeSeriesArray(header);
                ts.setForecastTime(header.getForecastTime());
                for(int i = 1; i <= 7; i++)
                {
                    final float value = scan.nextFloat();
                    if((value != MISSING_TEMP) && (!Float.isNaN(value)))
                    {
                        ts.putValue(header.getForecastTime() + i * header.getTimeStep().getStepMillis(), value);
                    }
                    else
                    {
                        ts.putValue(header.getForecastTime() + i * header.getTimeStep().getStepMillis(), Float.NaN);
                    }
                }

                if((ts.size() > 0) && (!ts.isCompletelyMissing()))
                {
                    seriesList.add(ts);
                }
            }
        }
        catch(final Exception e)
        {
            e.printStackTrace();
            throw new Exception("Failure while reading file: " + e.getMessage());
        }
        finally
        {
            scan.close();
        }

        return seriesList;
    }

    public static TimeSeriesArray readPrecipitationObservationFile(final File observationFile,
                                                                   final LocationAndDataTypeIdentifier identifier,
                                                                   final String paramId) throws Exception
    {
        // Create Time Series to be returned.
        final DefaultTimeSeriesHeader header = makePrecipitationHeader();
        setHeaderLocation(header, identifier.getLocationId());
        setHeaderParameter(header, paramId);
        header.setUnit("MM");
        header.setForecastTime(Long.MIN_VALUE);
        final TimeSeriesArray ts = new TimeSeriesArray(header);

        //If the file does not exist, return the empty time series.
        if(!observationFile.exists() || !observationFile.canRead())
        {
            return ts;
        }

        // Read File Header.
        final Scanner scan = new Scanner(observationFile);
        try
        {
            //Start scanning.
            final Properties props = readHeader(scan);
            final String unit = props.getProperty("units");
            float unitConversionFactor = 1.0f;
            if(unit.trim().equalsIgnoreCase("in"))
            {
                unitConversionFactor = 25.4f;
            }

            // Read File Body.
            scan.reset(); // Set delimiter back to any whitespace.
            while(scan.hasNext("\\d{8}")) // Start each entry with the date. 
            {
                final Calendar date = HCalendar.convertStringToCalendar(scan.next() + " 12:00:00", "CCYYMMDD hh:mm:ss");

                //Subtract 1 day to account for the goofiness in the rfc obs file.  Namely, the date on a line
                //specifies the observation day of the LAST value on the line, which is at 12Z.
                date.add(Calendar.HOUR_OF_DAY, -24);

                for(int i = 1; i <= 4; i++) // Starts at 18:00.
                {
                    float value = scan.nextFloat();
                    if((value != MISSING_PRECIP) && (!Float.isNaN(value)))
                    {
                        value = value * unitConversionFactor;
                        ts.putValue(date.getTimeInMillis() + i * header.getTimeStep().getStepMillis(), value);
                    }
                    else
                    {
                        ts.putValue(date.getTimeInMillis() + i * header.getTimeStep().getStepMillis(), Float.NaN);
                    }
                }
            }

            return ts;
        }
        catch(final Exception e)
        {
            e.printStackTrace();
            throw new Exception("Failure while reading file: " + e.getMessage());
        }
        finally
        {
            scan.close();
        }
    }

    public static TimeSeriesArray readTemperatureObservationFile(final File observationFile,
                                                                 final LocationAndDataTypeIdentifier identifier,
                                                                 final String paramId) throws Exception
    {
        // Create Time Series.
        final DefaultTimeSeriesHeader header = makeTemperatureHeader();
        setHeaderLocation(header, identifier.getLocationId());
        setHeaderParameter(header, paramId);
        header.setForecastTime(Long.MIN_VALUE);
        final TimeSeriesArray ts = new TimeSeriesArray(header);

        //If the observed file does not exist, return the empty time series.
        if(!observationFile.exists() || !observationFile.canRead())
        {
            return ts;
        }

        // Ignore Header.
        final Scanner scan = new Scanner(observationFile);

        try
        {
            scan.nextLine();
            scan.nextLine();

            // Read File Body.
            scan.reset(); // Set delimiter back to any whitespace.
            while(scan.hasNext("\\d{8}")) // Start each entry with the date. 
            {
                final Calendar date = HCalendar.convertStringToCalendar(scan.next() + " 12:00:00", "CCYYMMDD hh:mm:ss");

                final float value = scan.nextFloat();
                if((value != MISSING_TEMP) && (!Float.isNaN(value)))
                {
                    ts.putValue(date.getTimeInMillis(), value);
                }
                else
                {
                    ts.putValue(date.getTimeInMillis(), Float.NaN);
                }
            }

            return ts;
        }
        catch(final Exception e)
        {
            e.printStackTrace();
            throw new Exception("Failure while reading file: " + e.getMessage());
        }
        finally
        {
            scan.close();
        }
    }

    public static void writePrecipitationForecastFile(final File forecastFile, final List<TimeSeriesArray> data) throws IOException
    {
        if(data.isEmpty())
        {
            throw new IllegalArgumentException("Must write at least 1 time series.");
        }

        // Write header.
        final FileWriter writer = new FileWriter(forecastFile);

        try
        {
            writer.write("vtime = 12Z\n");
            writer.write("ctime = 12Z\n");
            writer.write("units = " + data.get(0).getHeader().getUnit() + "\n");
            writer.write("dt = 6\n");
            writer.write("nfcstdays = 5\n");
            writer.write("end\n");

            for(final TimeSeriesArray series: data)
            {
                // Start at 12Z.
                final Calendar firstDate = HCalendar.computeCalendarFromMilliseconds(series.getHeader()
                                                                                           .getForecastTime());
                final int hour = firstDate.get(Calendar.HOUR_OF_DAY);
                if(hour != 12)
                {
                    LOG.warn("Time series with computed forecast time "
                        + HCalendar.buildDateTimeTZStr(series.getHeader().getForecastTime())
                        + " will be ignored because it is not a 12Z time.");
                    continue;
                }

                writer.write(HCalendar.buildDateStr(firstDate, "CCYYMMDD "));

                long time = firstDate.getTimeInMillis();
                for(int j = 0; j < 20; j++)
                {
                    time += HCalendar.MILLIS_IN_HR * 6; // Increment time beforehand.
                    final int index = series.indexOfTime(time);
                    if(index == -1)
                    {
                        writer.write(String.format(" %6.2f", MISSING_PRECIP));
                    }
                    else
                    {
                        final float value = series.getValue(index);
                        if((value >= 0) && (!Float.isNaN(value)))
                        {
                            writer.write(String.format(" %6.2f", value));
                        }
                        else
                        {
                            writer.write(String.format(" %6.2f", MISSING_PRECIP));
                        }
                    }
                }

                writer.write("\n");
                writer.flush();
            }
        }
        finally
        {
            writer.close();
        }
    }

    public static void writeTemperatureForecastFile(final File forecastFile, final List<TimeSeriesArray> data) throws IOException
    {
        if(data.isEmpty())
        {
            throw new IllegalArgumentException("Must write at least 1 time series.");
        }

        // Write header.
        final FileWriter writer = new FileWriter(forecastFile);
        try
        {
            String location = data.get(0).getHeader().getLocationId().toUpperCase();
            if(location.length() > 8)
            {
                location = location.substring(0, 8);
            }
            writer.write(String.format("%-8s", location));
            writer.write(String.format(" %6d", 7));
            writer.write(String.format(" %8.4f", 0f));
            writer.write(String.format(" %8.4f", 0f));
            writer.write(String.format(" %5d", -9999));
            writer.write(String.format("  %-20s\n", "UNKNOWN"));
            writer.write(HCalendar.buildDateStr(data.get(0).getHeader().getForecastTime(), "CCYYMMDD "));
            writer.write(HCalendar.buildDateStr(data.get(data.size() - 1).getHeader().getForecastTime(), " CCYYMMDD\n"));

            for(final TimeSeriesArray series: data)
            {
                // Start at 12Z.
                final Calendar firstDate = HCalendar.computeCalendarFromMilliseconds(series.getHeader()
                                                                                           .getForecastTime());
                final int hour = firstDate.get(Calendar.HOUR_OF_DAY);
                if(hour != 12)
                {
                    LOG.warn("Time series with computed forecast time "
                        + HCalendar.buildDateStr(series.getHeader().getForecastTime(), HCalendar.DEFAULT_DATETZ_FORMAT)
                        + " will be ignored because it is not a 12Z time.");
                    continue;
                }

                writer.write(HCalendar.buildDateStr(firstDate, "CCYYMMDD"));

                long time = firstDate.getTimeInMillis();
                for(int j = 0; j < 7; j++)
                {
                    time += HCalendar.MILLIS_IN_HR * 24; // Increment time beforehand.
                    final int index = series.indexOfTime(time);
                    if(index == -1)
                    {
                        writer.write(" -99.0");
                    }
                    else
                    {
                        writer.write(String.format(" %5.1f", series.getValue(index)));
                    }
                }

                writer.write("\n");
                writer.flush();
            }
        }
        finally
        {
            writer.close();
        }
    }

    public static void writePrecipitationObservedFile(final File observedFile, final TimeSeriesArray series) throws IOException
    {
        if(series == null)
        {
            throw new IllegalArgumentException("Must write a time series.");
        }

        // Write header.
        final FileWriter writer = new FileWriter(observedFile);
        try
        {
            writer.write("vtime = 12Z\n");
            writer.write("units = " + series.getHeader().getUnit() + "\n");
            writer.write("dt = 6\n");
            writer.write("end\n");
            writer.flush();

            // Start at 18Z: Find the first 18Z value.
            int checkIndex = 0;
            Calendar checkDate = HCalendar.computeCalendarFromMilliseconds(series.getTime(checkIndex));
            int hour = checkDate.get(Calendar.HOUR_OF_DAY);
            while((hour != 18) && (checkIndex < series.size()))
            {
                checkIndex++;
                checkDate = HCalendar.computeCalendarFromMilliseconds(series.getTime(checkIndex));
                hour = checkDate.get(Calendar.HOUR_OF_DAY);
            }
            if(checkIndex == series.size())
            {
                throw new IllegalArgumentException("Provided series do not have any 18Z times; the first observed value output to RFC file must be at 18Z.");
            }

            for(long time = series.getTime(checkIndex); time < series.getEndTime();) // Incremented in inner loop.
            {
                //Adjust it forward one day, because the date on a line is the date of the LAST value on that
                //line, not the first value!
                writer.write(HCalendar.buildDateStr(time + 24 * HCalendar.MILLIS_IN_HR, "CCYYMMDD "));

                for(int i = 0; i < 4; i++)
                {
                    final int index = series.indexOfTime(time);
                    if(index == -1)
                    {
                        writer.write(String.format(" %6.2f", MISSING_PRECIP));
                    }
                    else
                    {
                        final float value = series.getValue(index);
                        if((value >= 0) && (!Float.isNaN(value)))
                        {
                            writer.write(String.format(" %6.2f", value));
                        }
                        else
                        {
                            writer.write(String.format(" %6.2f", MISSING_PRECIP));
                        }
                    }
                    time += HCalendar.MILLIS_IN_HR * 6; // Loop incremented here.
                }

                writer.write("\n");
                writer.flush();
            }
        }
        finally
        {
            writer.close();
        }
    }

    public static void writeTemperatureObservedFile(final File observedFile, final TimeSeriesArray series) throws IOException
    {
        if(series == null)
        {
            throw new IllegalArgumentException("Must write a time series.");
        }

        // Write header.
        final FileWriter writer = new FileWriter(observedFile);
        try
        {
            String location = series.getHeader().getLocationId().toUpperCase();
            if(location.length() > 8)
            {
                location = location.substring(0, 8);
            }
            writer.write(String.format("%-8s  ", location));
            writer.write(String.format(" %8.4f", 0f));
            writer.write(String.format(" %8.4f", 0f));
            writer.write(String.format(" %5d", -9999));
            writer.write(String.format("  %-20s\n", "UNKNOWN"));
            writer.write(HCalendar.buildDateStr(series.getStartTime(), "CCYYMMDD "));
            writer.write(HCalendar.buildDateStr(series.getEndTime(), " CCYYMMDD\n"));
            writer.flush();

            // Start at 18Z.
            final Calendar firstDate = HCalendar.computeCalendarFromMilliseconds(series.getStartTime());
            final int hour = firstDate.get(Calendar.HOUR_OF_DAY);
            if(hour != 12)
            {
                throw new IllegalArgumentException("Time Series to write is not at 12Z.");
            }

            for(long time = series.getStartTime(); time <= series.getEndTime(); time += HCalendar.MILLIS_IN_HR * 24)
            {
                writer.write(HCalendar.buildDateStr(time, "CCYYMMDD"));

                final int index = series.indexOfTime(time);
                if(index == -1)
                {
                    writer.write(" -99.0\n");
                }
                else
                {
                    writer.write(String.format(" %5.1f\n", series.getValue(index)));
                }

                writer.flush();
            }
        }
        finally
        {
            writer.close();
        }
    }
}
