package ohd.hseb.hefs.mefp.sources.rfcfcst;

import java.util.Map;

import ohd.hseb.hefs.mefp.sources.rfcfcst.database.DatabaseConnectionSpecification;
import ohd.hseb.hefs.mefp.sources.rfcfcst.database.DatabaseConnectionSpecificationReader;
import ohd.hseb.hefs.mefp.sources.rfcfcst.database.DatabaseConnectionTester;
import ohd.hseb.hefs.mefp.sources.rfcfcst.database.VfyPairFilter;
import ohd.hseb.hefs.mefp.sources.rfcfcst.database.VfyPairSource;
import ohd.hseb.hefs.pe.notice.StepUnitsUpdatedNotice;
import ohd.hseb.hefs.pe.sources.SourceOptions;
import ohd.hseb.hefs.pe.tools.LocationAndDataTypeIdentifier;
import ohd.hseb.hefs.pe.tools.LocationAndDataTypeIdentifierSupplier;
import ohd.hseb.hefs.utils.notify.ObjectModifiedNotice;
import ohd.hseb.hefs.utils.xml.CompositeXMLReader;
import ohd.hseb.hefs.utils.xml.CompositeXMLWriter;
import ohd.hseb.hefs.utils.xml.MapXMLReader;
import ohd.hseb.hefs.utils.xml.MapXMLWriter;
import ohd.hseb.hefs.utils.xml.XMLReader;
import ohd.hseb.hefs.utils.xml.XMLReaderException;
import ohd.hseb.hefs.utils.xml.XMLWriter;

import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.util.concurrent.ListeningExecutorService;

public class RFCDataOptions extends SourceOptions
{
    private DatabaseConnectionSpecification _defaultConnection;

    /**
     * This attribute may be null if the {@link ListeningExecutorService} passed into the constructors is null.
     */
    private DatabaseConnectionTester _connectionTester;

    /**
     * Map of identifiers to their specified source parameters.
     */
    private Map<LocationAndDataTypeIdentifier, VfyPairSource> _specifiedSources = Maps.newHashMap();

    /**
     * Map of identifiers to their current (last loaded) source parameters.
     */
    private final Map<LocationAndDataTypeIdentifier, VfyPairSource> _retrievedSources = Maps.newHashMap();

    private XMLReader _compositeReader;

    public RFCDataOptions(final ListeningExecutorService executor)
    {
        _connectionTester = null;
        if(executor != null)
        {
            _connectionTester = new DatabaseConnectionTester(executor);
            _connectionTester.register(this);
        }
        initializeReaders();
    }

    public RFCDataOptions(final DatabaseConnectionSpecification defaultConnection,
                          final Map<LocationAndDataTypeIdentifier, VfyPairSource> sourcesMap,
                          final ListeningExecutorService executor)
    {
        _connectionTester = null;
        if(executor != null)
        {
            _connectionTester = new DatabaseConnectionTester(executor);
            _connectionTester.register(this);
        }
        setDefaultConnection(defaultConnection);
        _specifiedSources = sourcesMap;
        _connectionTester.test(_defaultConnection);
    }

    /**
     * Creates and sets the default source for the given identifier.
     * 
     * @param identifier the identifier to add the default source for
     * @return the newly created default source
     */
    private VfyPairSource makeDefaultSource(final LocationAndDataTypeIdentifier identifier)
    {
        VfyPairSource source;
        if(identifier.isPrecipitationDataType())
        {
            final PedtsepCode code = PedtsepCode.make(identifier).withD("Q").withE("Z");
            final VfyPairFilter filter = new VfyPairFilter(identifier.getLocationId(), code);
            source = new VfyPairSource(_defaultConnection, filter);
        }
        else
        {
            final PedtsepCode code0 = PedtsepCode.make(identifier).withD("D").withE("X");
            final PedtsepCode code1 = PedtsepCode.make(identifier).withD("D").withE("N");
            final VfyPairFilter filter = new VfyPairFilter(identifier.getLocationId(), code0, code1);
            source = new VfyPairSource(_defaultConnection, filter);
        }
        _specifiedSources.put(identifier, source);
        return source;
    }

    public boolean hasSourceFor(final LocationAndDataTypeIdentifier identifier)
    {
        return _specifiedSources.containsKey(identifier);
    }

    public DatabaseConnectionSpecification getDefaultConnection()
    {
        return _defaultConnection;
    }

    public void setDefaultConnection(final DatabaseConnectionSpecification conn)
    {
        _defaultConnection = conn;
        post(new ObjectModifiedNotice(this));
    }

    public void setAllConnections(final DatabaseConnectionSpecification conn)
    {
        _defaultConnection = conn;
        for(final VfyPairSource source: _specifiedSources.values())
        {
            source.setConnection(conn);
        }
        post(new ObjectModifiedNotice(this));
    }

    /**
     * @return A {@link DatabaseConnectionTester} that can be used by the interface. It is possible for the return value
     *         to be null, but that should never be the case for an operationally, properly loaded set of RFC data
     *         options.
     */
    public DatabaseConnectionTester getTester()
    {
        return _connectionTester;
    }

    /**
     * Never call this method in any situation where the {@link #_connectionTester} may be null. This will null except.
     * I'm not going to prevent that exception, because, if it occurs, then it is a programmatic error and should be
     * fixed.
     */
    public void resendDefaultConnectionTest()
    {
        _connectionTester.resendConnectionEvent(_defaultConnection);
    }

    public boolean canConnect()
    {
        if(_connectionTester == null)
        {
            return false;
        }
        while(true)
        {
            try
            {
                return _connectionTester.canConnect(_defaultConnection);
            }
            catch(final InterruptedException e)
            {
                // Try again.
            }
        }
    }

    /**
     * Gets the source for the specified identifier, creating it if needed.
     * 
     * @param identifier the identifier we are retrieving the source for
     * @return the source for the identifier, or a new source if there isn't one specified
     */
    public VfyPairSource getSourceFor(final LocationAndDataTypeIdentifier identifier)
    {
        VfyPairSource source = _specifiedSources.get(identifier);
        if(source == null)
        {
            source = makeDefaultSource(identifier);
        }
        return source;
    }

//    public void setSourceFor(LocationAndDataTypeIdentifier identifier, VfyPairSource source)
//    {
//        _specifiedSources.put(identifier, source);
//    }
//
//    // Convenience function.
//    public void setConnectionFor(LocationAndDataTypeIdentifier identifier, DatabaseConnectionSpecification connection)
//    {
//        VfyPairSource source = new VfyPairSource(connection, _specifiedSources.get(identifier).getFilter());
//        _specifiedSources.put(identifier, source);
//    }

    /**
     * Set the location for the source of the given identifier
     * 
     * @param identifier identifier to modify the source of
     * @param locationId the new location id to grab the source from
     */
    public void setSourceLocation(final LocationAndDataTypeIdentifier identifier, final String locationId)
    {
        _specifiedSources.get(identifier).setLocation(locationId);
        post(new ObjectModifiedNotice(this));
    }

    /**
     * Set the given pedtsep code for the identifier's source.
     * 
     * @param identifier the identifier to modify the source of
     * @param index the index of the code to replace
     * @param code the new first pedtsep code for the source
     */
    public void setSourceCode(final LocationAndDataTypeIdentifier identifier, final int index, final PedtsepCode code)
    {
        _specifiedSources.get(identifier).getFilter().getCodes().set(index, VfyPairPedtsepCode.make(code));
        post(new ObjectModifiedNotice(this));
    }

    /**
     * Get the given pedtsep code for the identifier's source.
     * 
     * @param identifier the identifier to retrieve the source of
     * @param index the index of the code to get
     */
    public VfyPairPedtsepCode getSourceCode(final LocationAndDataTypeIdentifier identifier, final int index)
    {
        return _specifiedSources.get(identifier).getFilter().getCodes().get(index);
    }

    /**
     * Notifies the options that the source was prepared, so it can store its mapping for that identifier as the last
     * one that was actually performed.
     * 
     * @param identifier the identifier which was prepared
     */
    public void notifySourceWasPrepared(final LocationAndDataTypeIdentifier identifier)
    {
        notifySourceWasPrepared(identifier, this);
    }

    public void notifySourceWasPrepared(final LocationAndDataTypeIdentifier identifier, final Object source)
    {
        _retrievedSources.put(identifier, _specifiedSources.get(identifier));
        // This is called by the step runner.
//        post(new StepUnitUpdatedEvent(source, RFCForecastPEStepProcessor.class, identifier));
    }

    public void notifySourceWasImported(final LocationAndDataTypeIdentifier identifier)
    {
        notifySourceWasImported(identifier, this);
    }

    public void notifySourceWasImported(final LocationAndDataTypeIdentifier identifier, final Object source)
    {
        _retrievedSources.remove(identifier);

        //Added on 7/16/14 to fix a problem where the Location Summary Panel was not updating after importing RFC data.
//        post(new StepUpdatedNotice(source, RFCForecastPEStepProcessor.class));

        //I don't know that this is necessary, but it existed before and didn't appear to break anything.
        post(new StepUnitsUpdatedNotice(source, RFCForecastPEStepProcessor.class, Lists.newArrayList(identifier)));
    }

    /**
     * If the sources for the specified identifier are current
     * 
     * @param identifier
     * @return
     */
    public boolean isCurrent(final LocationAndDataTypeIdentifier identifier)
    {
        return _specifiedSources.containsKey(identifier)
            && _specifiedSources.get(identifier).equals(_retrievedSources.get(identifier));
    }

    public boolean isImported(final LocationAndDataTypeIdentifier identifier)
    {
        return !_retrievedSources.containsKey(identifier);
    }

    private void initializeReaders()
    {
        try
        {
            final Map<LocationAndDataTypeIdentifierSupplier, VfyPairSource> tempActiveSources = Maps.newHashMap();
            final Map<LocationAndDataTypeIdentifierSupplier, VfyPairSource> tempCurrentSources = Maps.newHashMap();

            MapXMLReader<LocationAndDataTypeIdentifierSupplier, VfyPairSource> activeSourceReader;
            activeSourceReader = new MapXMLReader<LocationAndDataTypeIdentifierSupplier, VfyPairSource>("activeSourceMap",
                                                                                                        tempActiveSources,
                                                                                                        LocationAndDataTypeIdentifierSupplier.DEFAULT_FACTORY,
                                                                                                        VfyPairSource.DEFAULT_FACTORY);
            MapXMLReader<LocationAndDataTypeIdentifierSupplier, VfyPairSource> currentSourceReader;
            currentSourceReader = new MapXMLReader<LocationAndDataTypeIdentifierSupplier, VfyPairSource>("currentSourceMap",
                                                                                                         tempCurrentSources,
                                                                                                         LocationAndDataTypeIdentifierSupplier.DEFAULT_FACTORY,
                                                                                                         VfyPairSource.DEFAULT_FACTORY);

            _defaultConnection = new DatabaseConnectionSpecification("DATABASENAME");
            final DatabaseConnectionSpecificationReader connectionReader = new DatabaseConnectionSpecificationReader(_defaultConnection);
            _compositeReader = new CompositeXMLReader(getXMLTagName(),
                                                      activeSourceReader,
                                                      currentSourceReader,
                                                      connectionReader)
            {
                @Override
                public void finalizeReading() throws XMLReaderException
                {
                    super.finalizeReading();

                    for(final Map.Entry<LocationAndDataTypeIdentifierSupplier, VfyPairSource> entry: tempActiveSources.entrySet())
                    {
                        _specifiedSources.put(entry.getKey().get(), entry.getValue());
                    }
                    for(final Map.Entry<LocationAndDataTypeIdentifierSupplier, VfyPairSource> entry: tempCurrentSources.entrySet())
                    {
                        _retrievedSources.put(entry.getKey().get(), entry.getValue());
                    }
                    setDefaultConnection(connectionReader.get());
                    if(_connectionTester != null)
                    {
                        _connectionTester.test(_defaultConnection);
                    }
                }
            };
        }
        catch(final Exception e)
        {
            throw new RuntimeException(e);
        }
    }

    public String getXMLTagName()
    {
        return "rfcDataOptions";
    }

    @Override
    public XMLReader getReader()
    {
        return _compositeReader;
    }

    @Override
    public XMLWriter getWriter()
    {
        return new CompositeXMLWriter(getXMLTagName(),
                                      _defaultConnection,
                                      new MapXMLWriter("activeSourceMap", _specifiedSources),
                                      new MapXMLWriter("currentSourceMap", _retrievedSources)

        );
    }
}
