package ohd.hseb.hefs.mefp.sources.rfcfcst;

import java.io.File;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Map;

import nl.wldelft.util.timeseries.TimeSeriesArray;
import ohd.hseb.hefs.mefp.sources.AbstractMEFPSourceDataHandler;
import ohd.hseb.hefs.mefp.sources.rfcfcst.database.DatabaseConnectionSpecification;
import ohd.hseb.hefs.mefp.tools.QuestionableMessageMap;
import ohd.hseb.hefs.mefp.tools.QuestionableStatus;
import ohd.hseb.hefs.mefp.tools.QuestionableTools;
import ohd.hseb.hefs.pe.tools.HEFSTools;
import ohd.hseb.hefs.pe.tools.LocationAndDataTypeIdentifier;
import ohd.hseb.hefs.pe.tools.TimeSeriesSorter;
import ohd.hseb.hefs.utils.status.StatusIndicator;
import ohd.hseb.hefs.utils.tools.FileTools;
import ohd.hseb.hefs.utils.tools.ListTools;
import ohd.hseb.hefs.utils.tools.MapTools;
import ohd.hseb.hefs.utils.tools.ParameterId;

import org.apache.logging.log4j.Logger;
import org.apache.logging.log4j.LogManager;

import com.google.common.collect.ArrayListMultimap;
import com.google.common.collect.ImmutableMap;
import com.google.common.collect.Multimap;

/**
 * Can be used to load either precip or temperature RFC data. The time series loaded by this handler use HEFSTools
 * constants to identify parameter id strings.
 * 
 * @author hank.herr
 */
public class RFCForecastDataHandler extends AbstractMEFPSourceDataHandler implements ConnectionChangeListener
{
    private static final Logger LOG = LogManager.getLogger(RFCForecastDataHandler.class);

    public static int TMAX_INDEX = 0;
    public static int TMIN_INDEX = 1;

    /**
     * Maps the file extension to its {@link ParameterId}. Reverses {@link #PARM_ID_EXTENSION}.
     */
    public static final Map<String, ParameterId> EXTENSION_PARM_ID;
    static
    {
        EXTENSION_PARM_ID = ImmutableMap.<String, ParameterId>builder()
                                        .put("pfcst06", ParameterId.FMAP)
                                        .put("rfctmxfcst", ParameterId.TFMX)
                                        .put("rfctmnfcst", ParameterId.TFMN)
                                        .put("pobs06", ParameterId.MAP)
                                        .put("rfctmxobs", ParameterId.TMAX)
                                        .put("rfctmnobs", ParameterId.TMIN)
                                        .build();
    }

    /**
     * Maps a {@link ParameterId} to its appropriate file extension. Reverses {@link #EXTENSION_PARM_ID}.
     */
    public static final Map<ParameterId, String> PARM_ID_EXTENSION;
    static
    {
        PARM_ID_EXTENSION = ImmutableMap.<ParameterId, String>builder()
                                        .put(ParameterId.FMAP, "pfcst06")
                                        .put(ParameterId.TFMX, "rfctmxfcst")
                                        .put(ParameterId.TFMN, "rfctmnfcst")
                                        .put(ParameterId.MAP, "pobs06")
                                        .put(ParameterId.TMAX, "rfctmxobs")
                                        .put(ParameterId.TMIN, "rfctmnobs")
                                        .build();
    }

    private File _preparedDataFileDirectory = null;

    private final Multimap<LocationAndDataTypeIdentifier, TimeSeriesArray> _forecastTimeSeries = ArrayListMultimap.create();
    private final Multimap<LocationAndDataTypeIdentifier, TimeSeriesArray> _observedTimeSeries = ArrayListMultimap.create();

    private RFCDataOptions _options;

    public RFCForecastDataHandler()
    {
    }

    public RFCForecastDataHandler(final File baseDirectory) throws Exception
    {
        setDataHandlerBaseDirectory(baseDirectory);
        initialize();
    }

    @Override
    public void initialize() throws Exception
    {
        LOG.info("Initializing RFC Forecast data handler...");
        LOG.info("Completed initialization.");
    }

    /**
     * @param identifier Identifier for which to build the directory name for RFC files.
     * @param observed True to get the directory for observed data, false for forecast data.
     * @return The complete path name for the directory that contains data files for the identifier and type of time
     *         series: observed or forecast.
     */
    public String buildDirectoryName(final LocationAndDataTypeIdentifier identifier, final boolean observed)
    {
        if(identifier.isPrecipitationDataType())
        {
            if(observed)
            {
                return _preparedDataFileDirectory.getAbsolutePath() + "/rfc_pobs06";
            }
            else
            {
                return _preparedDataFileDirectory.getAbsolutePath() + "/rfc_pfcst06";
            }
        }
        else
        // temp
        {
            if(observed)
            {
                return _preparedDataFileDirectory.getAbsolutePath() + "/rfc_tobs";
            }
            else
            {
                return _preparedDataFileDirectory.getAbsolutePath() + "/rfc_tfcst";
            }
        }
    }

    public RFCDataOptions getOptions()
    {
        return _options;
    }

    public void setOptions(final RFCDataOptions options)
    {
        _options = options;
    }

    // Returns max, then min.
    public List<File> getPreparedForecastSourcesForIdentifier(final LocationAndDataTypeIdentifier identifier)
    {
        final List<File> results = new ArrayList<File>();
        if(identifier.isPrecipitationDataType())
        {
            results.add(new File(buildDirectoryName(identifier, false) + "/" + identifier.getUsedLocationId() + "."
                + PARM_ID_EXTENSION.get(ParameterId.FMAP)));
        }
        else if(identifier.isTemperatureDataType())
        {
            results.add(new File(buildDirectoryName(identifier, false) + "/" + identifier.getUsedLocationId() + "."
                + PARM_ID_EXTENSION.get(ParameterId.TFMX)));
            results.add(new File(buildDirectoryName(identifier, false) + "/" + identifier.getUsedLocationId() + "."
                + PARM_ID_EXTENSION.get(ParameterId.TFMN)));
        }
        return results;
    }

    // Returns max, then min.
    public List<File> getPreparedObservedSourcesForIdentifier(final LocationAndDataTypeIdentifier identifier)
    {
        final List<File> results = new ArrayList<File>();
        if(identifier.isPrecipitationDataType())
        {
            results.add(new File(buildDirectoryName(identifier, true) + "/" + identifier.getUsedLocationId() + "."
                + PARM_ID_EXTENSION.get(ParameterId.MAP)));
        }
        else if(identifier.isTemperatureDataType())
        {
            results.add(new File(buildDirectoryName(identifier, true) + "/" + identifier.getUsedLocationId() + "."
                + PARM_ID_EXTENSION.get(ParameterId.TMAX)));
            results.add(new File(buildDirectoryName(identifier, true) + "/" + identifier.getUsedLocationId() + "."
                + PARM_ID_EXTENSION.get(ParameterId.TMIN)));
        }
        return results;
    }

    /**
     * @param file The file for which to determine its pair.
     * @return The name of the file that can be considered as a pair for the provided file. Specifically, temperature
     *         files always come in min/max pairs, so if the min file is given, then the max file will be returned. Null
     *         is returned if there is no paired file, such as for precip.
     */
    public File getPairedFile(final File file)
    {
        final String location = FileTools.getBaseName(file);
        final String extension = FileTools.getExtension(file);

        final ParameterId parmId = EXTENSION_PARM_ID.get(extension);

        //Precipitation obs and forecast files are singletons.
        if(parmId.isPrecipitation())
        {
            return null;
        }
        if(parmId == ParameterId.TMAX)
        {
            return FileTools.newFile(file.getParent(), location + "." + PARM_ID_EXTENSION.get(ParameterId.TMIN));
        }
        if(parmId == ParameterId.TMIN)
        {
            return FileTools.newFile(file.getParent(), location + "." + PARM_ID_EXTENSION.get(ParameterId.TMAX));
        }
        if(parmId == ParameterId.TFMX)
        {
            return FileTools.newFile(file.getParent(), location + "." + PARM_ID_EXTENSION.get(ParameterId.TFMN));
        }
        if(parmId == ParameterId.TFMN)
        {
            return FileTools.newFile(file.getParent(), location + "." + PARM_ID_EXTENSION.get(ParameterId.TFMX));
        }
        return null;
    }

    /**
     * @param source
     * @return A LocationAndDataTypeIdentifier for the given file, with its location and parameter info determined from
     *         the file name.
     */
    public LocationAndDataTypeIdentifier getIdentifierForSource(final File source)
    {
        final String location = FileTools.getBaseName(source);
        final String parameter = EXTENSION_PARM_ID.get(FileTools.getExtension(source)).toString();
        return LocationAndDataTypeIdentifier.get(location, parameter);
    }

    public File getPreparedDirectory()
    {
        return _preparedDataFileDirectory.getAbsoluteFile();
    }

    /**
     * Only call this for testing!
     */
    protected void setPreparedDirectory(final File dir)
    {
        _preparedDataFileDirectory = dir;
    }

    @Override
    public void setDataHandlerBaseDirectory(final File directory)
    {
        super.setDataHandlerBaseDirectory(directory);
        _preparedDataFileDirectory = new File(directory.getAbsolutePath() + "/rfcForecastData");
    }

    @Override
    public void loadOriginalTimeSeries(final List<LocationAndDataTypeIdentifier> identifiers) throws Exception
    {
        for(final LocationAndDataTypeIdentifier identifier: identifiers)
        {
            final VfyPairSet set = _options.getSourceFor(identifier).getPairs();
            if(set.isEmpty())
            {
                throw new Exception("No forecast-observed pairs were found for "
                    + identifier.buildStringToDisplayInTree() + ". RFC forecast files could not be constructed.");
            }
            final TimeSeriesSorter sorter = set.constructTimeSeries(false);

            _forecastTimeSeries.putAll(identifier, sorter.restrictViewToForecast());
            _observedTimeSeries.putAll(identifier, sorter.restrictViewToObserved());
        }
    }

    @Override
    public void loadPreparedTimeSeries(final List<LocationAndDataTypeIdentifier> identifiers) throws Exception
    {
        clearLoadedTimeSeries();
        for(final LocationAndDataTypeIdentifier identifier: identifiers)
        {
            loadPreparedTimeSeries(identifier);
        }
    }

    public void loadPreparedTimeSeries(final LocationAndDataTypeIdentifier identifier) throws Exception
    {
        List<File> sources;

        if(HEFSTools.isPrecipitationDataType(identifier))
        {
            // Observed.
            sources = getPreparedObservedSourcesForIdentifier(identifier);
            _observedTimeSeries.put(identifier,
                                    PreparedRFCFileTools.readPrecipitationObservationFile(sources.get(0),
                                                                                          identifier,
                                                                                          HEFSTools.DEFAULT_PRECIPITATION_IDENTIFIER_PARAMETER_ID));

            // Forecast.
            sources = getPreparedForecastSourcesForIdentifier(identifier);
            _forecastTimeSeries.putAll(identifier,
                                       PreparedRFCFileTools.readPrecipitationForecastFile(sources.get(0),
                                                                                          identifier,
                                                                                          HEFSTools.FORECAST_PRECIP_PARAMETER_ID));
        }
        else if(HEFSTools.isTemperatureDataType(identifier))
        {
            // Observed.
            sources = getPreparedObservedSourcesForIdentifier(identifier);
            _observedTimeSeries.put(identifier,
                                    PreparedRFCFileTools.readTemperatureObservationFile(sources.get(0),
                                                                                        identifier,
                                                                                        HEFSTools.DEFAULT_TMAX_PARAMETER_ID));
            _observedTimeSeries.put(identifier,
                                    PreparedRFCFileTools.readTemperatureObservationFile(sources.get(1),
                                                                                        identifier,
                                                                                        HEFSTools.DEFAULT_TMIN_PARAMETER_ID));

            // Forecast.
            sources = getPreparedForecastSourcesForIdentifier(identifier);
            _forecastTimeSeries.putAll(identifier,
                                       PreparedRFCFileTools.readTemperatureForecastFile(sources.get(0),
                                                                                        identifier,
                                                                                        HEFSTools.FORECAST_TMAX_PARAMETER_ID));
            _forecastTimeSeries.putAll(identifier,
                                       PreparedRFCFileTools.readTemperatureForecastFile(sources.get(1),
                                                                                        identifier,
                                                                                        HEFSTools.FORECAST_TMIN_PARAMETER_ID));
        }
        else
        {
            throw new Exception("Unknown parameter: " + identifier.getParameterId());
        }
    }

    @Override
    public void clearLoadedTimeSeries()
    {
        _forecastTimeSeries.clear();
        _observedTimeSeries.clear();
    }

    @Override
    public Collection<TimeSeriesArray> getLoadedForecastTimeSeries(final LocationAndDataTypeIdentifier identifier)
    {
        return _forecastTimeSeries.get(identifier);
    }

    @Override
    public Collection<TimeSeriesArray> getLoadedObservedTimeSeries(final LocationAndDataTypeIdentifier identifier)
    {
        return _observedTimeSeries.get(identifier);
    }

    @Override
    public List<LocationAndDataTypeIdentifier> getIdentifiersWithData()
    {
        final List<LocationAndDataTypeIdentifier> identifiers = new ArrayList<LocationAndDataTypeIdentifier>();
        ListTools.addAllUniqueItemsToList(identifiers, _observedTimeSeries.keySet());
        ListTools.addAllUniqueItemsToList(identifiers, _forecastTimeSeries.keySet());
        return identifiers;
    }

    @Override
    public void prepareDataFiles() throws Exception
    {
        TimeSeriesSorter sorter;
        List<File> files;
        LocationAndDataTypeIdentifier identifier;

        // Prepare forecast series.

        for(final Map.Entry<LocationAndDataTypeIdentifier, Collection<TimeSeriesArray>> entry: _forecastTimeSeries.asMap()
                                                                                                                  .entrySet())
        {
            identifier = entry.getKey();
            sorter = new TimeSeriesSorter(entry.getValue());

            if(identifier.isPrecipitationDataType())
            {
                files = getPreparedForecastSourcesForIdentifier(identifier);
                PreparedRFCFileTools.writePrecipitationForecastFile(files.get(0), sorter.toList());
            }
            else
            {
                files = getPreparedForecastSourcesForIdentifier(identifier);
                PreparedRFCFileTools.writeTemperatureForecastFile(files.get(0),
                                                                  sorter.restrictViewToParameters(HEFSTools.FORECAST_TMAX_PARAMETER_ID)
                                                                        .toList());
                PreparedRFCFileTools.writeTemperatureForecastFile(files.get(1),
                                                                  sorter.restrictViewToParameters(HEFSTools.FORECAST_TMIN_PARAMETER_ID)
                                                                        .toList());
            }

            try
            {
                QuestionableTools.deleteQuestionableFile(files.get(0).getParent(), getIdentifierForSource(files.get(0)));
                QuestionableTools.createRFCQuestionableFile(files.get(0), LOG);
                // uncomment for debugging
                // final HashMap<String, HashMap<Long, HashMap<Long, String>>> testHash = loadQuestionableHash(identifier);
                // System.err.println(testHash);
            }
            catch(final Throwable t)
            {
                LOG.warn("Problem creating questionable file: " + t.getMessage());
            }
        }

        // Prepare observed series.

        for(final Map.Entry<LocationAndDataTypeIdentifier, Collection<TimeSeriesArray>> entry: _observedTimeSeries.asMap()
                                                                                                                  .entrySet())
        {
            identifier = entry.getKey();
            sorter = new TimeSeriesSorter(entry.getValue());

            if(identifier.isPrecipitationDataType())
            {
                files = getPreparedObservedSourcesForIdentifier(identifier);
                try
                {
                    PreparedRFCFileTools.writePrecipitationObservedFile(files.get(0), sorter.getOnly());
                }
                catch(final IllegalStateException e)
                {
                    throw new Exception("While writing observed file, an exception occured extracting single observed time series from archive data: "
                        + e.getMessage());
                }
            }
            else
            {
                files = getPreparedObservedSourcesForIdentifier(identifier);
                PreparedRFCFileTools.writeTemperatureObservedFile(files.get(0),
                                                                  sorter.restrictViewToParameters(HEFSTools.FORECAST_TMAX_PARAMETER_ID)
                                                                        .getOnly());
                PreparedRFCFileTools.writeTemperatureObservedFile(files.get(1),
                                                                  sorter.restrictViewToParameters(HEFSTools.FORECAST_TMIN_PARAMETER_ID)
                                                                        .getOnly());
            }

            try
            {
                QuestionableTools.deleteQuestionableFile(files.get(0).getParent(), getIdentifierForSource(files.get(0)));
                QuestionableTools.createRFCQuestionableFile(files.get(0), LOG);
                // uncomment for debugging
                // final HashMap<String, HashMap<Long, HashMap<Long, String>>> testHash = loadQuestionableHash(identifier);
                // System.err.println(testHash);
            }
            catch(final Throwable t)
            {
                LOG.warn("Problem creating questionable file: " + t.getMessage());
            }
        }
//      post(new StepUpdatedEvent(this, RFCForecastPEStepProcessor.class));
    }

    public void notifySourceWasImported(final File fileSource, final Object eventSource)
    {
        getOptions().notifySourceWasImported(getIdentifierForSource(fileSource), eventSource);
    }

    @Override
    public Collection<TimeSeriesArray> getAllLoadedForecastTimeSeries()
    {
        return _forecastTimeSeries.values();
    }

    @Override
    public Collection<TimeSeriesArray> getAllLoadedObservedTimeSeries()
    {
        return _observedTimeSeries.values();
    }

    @Override
    public boolean havePreparedDataFilesBeenCreatedAlready(final LocationAndDataTypeIdentifier identifier)
    {
        // Both files must exist.
        final List<File> preparedForecastDataFiles = getPreparedForecastSourcesForIdentifier(identifier);
//        final List<File> preparedObservedDataFiles = getPreparedObservedSourcesForIdentifier(identifier);

        for(final File file: preparedForecastDataFiles)
        {
            if(!file.exists())
            {
                return false;
            }
        }
//        for(final File file: preparedObservedDataFiles)
//        {
//            if(!file.exists())
//            {
//                return false;
//            }
//        }
        return true;
    }

    public boolean havePreparedObservedFilesBeenCreated(final LocationAndDataTypeIdentifier identifier)
    {
        // Both files must exist.
        final List<File> preparedObservedDataFiles = getPreparedObservedSourcesForIdentifier(identifier);

        for(final File file: preparedObservedDataFiles)
        {
            if(!file.exists())
            {
                return false;
            }
        }
        return true;
    }

    @Override
    public boolean arePreparedDataFilesUpToDate(final LocationAndDataTypeIdentifier identifier)
    {
        return havePreparedDataFilesBeenCreatedAlready(identifier) && _options.isCurrent(identifier);
    }

    public StatusIndicator getStatus(final LocationAndDataTypeIdentifier identifier)
    {
        if(!havePreparedDataFilesBeenCreatedAlready(identifier))
        {
            return RFCStatus.MISSING;
        }

        if(QuestionableTools.isQuestionable(buildDirectoryName(identifier, false), identifier)
            || QuestionableTools.isQuestionable(buildDirectoryName(identifier, true), identifier))
        {
            return QuestionableStatus.QUESTIONABLE;
        }
        else if(_options.isImported(identifier))
        {
            if(!havePreparedObservedFilesBeenCreated(identifier))
            {
                return RFCStatus.NO_OBS;
            }
            return RFCStatus.IMPORTED;
        }
        else if(_options.isCurrent(identifier))
        {
            if(!havePreparedObservedFilesBeenCreated(identifier))
            {
                return RFCStatus.NO_OBS;
            }
            return RFCStatus.READY;
        }
        else if(!havePreparedObservedFilesBeenCreated(identifier))
        {
            return RFCStatus.NO_OBS;
        }
        else
        {
            return RFCStatus.NEEDS_UPDATE;
        }
    }

    @Override
    public File getPreparedDataFilesDirectory()
    {
        return this._preparedDataFileDirectory;
    }

    @Override
    public List<File> generateListOfPreparedDataFiles(final LocationAndDataTypeIdentifier identifier)
    {
        final List<File> results = new ArrayList<File>();
        results.addAll(this.getPreparedObservedSourcesForIdentifier(identifier));
        results.addAll(this.getPreparedForecastSourcesForIdentifier(identifier));
        return results;
    }

    @Override
    public void setConnection(final DatabaseConnectionSpecification connection)
    {
        getOptions().setAllConnections(connection);
    }

    /**
     * @param identifier - identifier used to construct the questionableFile
     * @return The map created by {@link QuestionableTools#toHash(java.io.File)}.
     * @throws Exception
     */
    @Override
    public QuestionableMessageMap loadQuestionableHash(final LocationAndDataTypeIdentifier identifier) throws Exception
    {
        File questionableFile;

        // Get the fcstHash
        QuestionableMessageMap fcstHash = null;
        questionableFile = QuestionableTools.getFile(buildDirectoryName(identifier, false), identifier);
        if(questionableFile.exists())
        {
            fcstHash = new QuestionableMessageMap(questionableFile);
        }

        // Get the obsHash
        QuestionableMessageMap obsHash = null;
        questionableFile = QuestionableTools.getFile(buildDirectoryName(identifier, true), identifier);
        if(questionableFile.exists())
        {
            obsHash = new QuestionableMessageMap(questionableFile);
        }

        if(fcstHash != null)
        {
            if(obsHash != null) // put obsHash into fcstHash
            {
                MapTools.concatenateMaps(fcstHash, obsHash);
            }

            return (fcstHash);
        }
        else if(obsHash != null)
        {
            return (obsHash);
        }

        // If you got here, both fcstHash and obsHash are null
        return null;
    }

    /**
     * @return A file that pairs with the given RFC forecast file, if the provided file is a temperature file. For
     *         example, if provided a rfctmxfcst file, it will return an rfctmnfcst file (and vice versa), whereas
     *         rfctmxobs will pair with rfctmnobs (and vice versa). If given a non-temperature file, null is returned.
     */
    public static File findPairedTemperatureFile(final File file)
    {
        final String fileExt = FileTools.getExtension(file);
        if(fileExt.equals(PARM_ID_EXTENSION.get(ParameterId.TMIN)))
        {
            return FileTools.replaceExtension(file, PARM_ID_EXTENSION.get(ParameterId.TMAX));
        }
        if(fileExt.equals(PARM_ID_EXTENSION.get(ParameterId.TFMN)))
        {
            return FileTools.replaceExtension(file, PARM_ID_EXTENSION.get(ParameterId.TFMX));
        }
        if(fileExt.equals(PARM_ID_EXTENSION.get(ParameterId.TMAX)))
        {
            return FileTools.replaceExtension(file, PARM_ID_EXTENSION.get(ParameterId.TMIN));
        }
        if(fileExt.equals(PARM_ID_EXTENSION.get(ParameterId.TFMX)))
        {
            return FileTools.replaceExtension(file, PARM_ID_EXTENSION.get(ParameterId.TFMN));
        }
        return null;
    }
}
