package ohd.hseb.hefs.mefp.sources.rfcfcst;

import java.awt.Color;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.List;

import nl.wldelft.util.timeseries.TimeSeriesArray;
import ohd.hseb.charter.ChartEngine;
import ohd.hseb.charter.datasource.DefaultXYChartDataSource;
import ohd.hseb.charter.panel.ChartEngineTableModel;
import ohd.hseb.hefs.mefp.pe.core.GraphGenForDiagnosticsEditingRunner;
import ohd.hseb.hefs.mefp.tools.QuestionableMessageMap;
import ohd.hseb.hefs.pe.tools.DiagnosticChartBuilder;
import ohd.hseb.hefs.pe.tools.LocationAndDataTypeIdentifier;
import ohd.hseb.hefs.pe.tools.TimeSeriesChartDiagnosticPanel;
import ohd.hseb.hefs.pe.tools.TimeSeriesChartDiagnosticTableModel;
import ohd.hseb.hefs.utils.gui.help.HelpFile;
import ohd.hseb.hefs.utils.tsarrays.TimeSeriesArrayTools;

import com.google.common.base.Supplier;

/**
 * This diagnostic display is built based on the system resources rfcPrecipitationDisplayDiagnostics.xml and
 * rfcTemperatureDisplayDiagnostics.xml under mefppe/diagnostics. It is a Graphics Generator product which displays each
 * GFS forecast time series (prate for precip and tmax/tmin for temp) in a rotating color scheme based on 9 member
 * espadp color palette (blue to cyan to green to yellow in 9 steps).<br>
 * <br>
 * To modify the base product, run the {@link GraphGenForDiagnosticsEditingRunner} under ohd.hseb.mefp.tools as a java
 * app. Select the product you want to edit under nonsrc/mefppe/diagnostics, and modify it. When done, copy the
 * parameters created and viewed via the Output panel preview button (at the bottom) into the product by hand. Add any
 * needed required arguments to the execute parameter section of the XML.
 * 
 * @author alexander.garbarino
 */
@HelpFile("PDFHELP:helpManual.pdf#RFCDiagnosticsPanelRef")
@SuppressWarnings("serial")
public class RFCForecastDiagnosticPanel extends TimeSeriesChartDiagnosticPanel
{
    private final static Color QUESTIONABLE_MARK = new Color(255, 0, 0, 64); // red
    private final static Color QUESTIONABLE_ZONE = new Color(255, 0, 0, 64); // red

    private final Collection<TimeSeriesArray> _observedSeries;

    /**
     * The map of questionable data; must be available by the time {@link #buildTableModelSupplier()} is called.
     */
    private final QuestionableMessageMap _questionableMap;

    /**
     * Kept sorted on forecast time.
     */
    private final List<TimeSeriesArray> _forecastSeries;

    private DiagnosticChartBuilder _chartBuilder;

    @Override
    protected boolean hasYearSpinner()
    {
        return true;
    }

    @Override
    protected boolean hasForecastList()
    {
        return true;
    }

    /**
     * This override is needed in order to ensure that {@link #_questionableMap} gets to the table's
     * {@link TimeSeriesChartDiagnosticTableModel}.
     */
    @Override
    protected Supplier<ChartEngineTableModel> buildTableModelSupplier()
    {
        return new Supplier<ChartEngineTableModel>()
        {
            @Override
            public ChartEngineTableModel get()
            {
                final TimeSeriesChartDiagnosticTableModel model = new TimeSeriesChartDiagnosticTableModel();
                model.setQuestionableMap(_questionableMap);
                return model;
            }
        };
    }

    @SuppressWarnings("unchecked")
    public RFCForecastDiagnosticPanel(final LocationAndDataTypeIdentifier identifier,
                                      final Collection<TimeSeriesArray> observedSeries,
                                      final Collection<TimeSeriesArray> forecastSeries,
                                      final QuestionableMessageMap questionableMap)
    {
        super(identifier, observedSeries, forecastSeries);
        _questionableMap = questionableMap;

        _observedSeries = observedSeries;
        _forecastSeries = new ArrayList<TimeSeriesArray>(forecastSeries);
        Collections.sort(_forecastSeries, TimeSeriesArrayTools.FORECAST_TIME_COMPARATOR);
        try
        {
            final String parameterFile;

            if(getIdentifier().isPrecipitationDataType())
            {
                parameterFile = "mefppe/diagnostics/rfcPrecipitationDisplayDiagnostics.xml";
            }
            else if(getIdentifier().isTemperatureDataType())
            {
                parameterFile = "mefppe/diagnostics/rfcTemperatureDisplayDiagnostics.xml";
            }
            else
            {
                throw new Exception("Unknown parameter type.");
            }
            _chartBuilder = new DiagnosticChartBuilder(getIdentifier(),
                                                       parameterFile,
                                                       RFCForecastSource.SOURCE_ID,
                                                       _observedSeries,
                                                       _forecastSeries);
        }
        catch(final Exception e)
        {
            error(e);
        }

        updateChart();
    }

    @Override
    protected ChartEngine buildChart() throws Exception
    {
        _chartBuilder.setYear(getSelectedYear());
        _chartBuilder.setQuestionableMap(_questionableMap);
        _chartBuilder.setEmphasizedDates(getSelectedDates());
        _chartBuilder.addDomainThresholdsForQuestionableValues(QUESTIONABLE_MARK, QUESTIONABLE_ZONE);

        final ChartEngine engine = _chartBuilder.buildChartEngine();

        ((DefaultXYChartDataSource)engine.getDataSources().get(0)).setSourceNameInTable("Emphasized Ensemble Mean");
        ((DefaultXYChartDataSource)engine.getDataSources().get(1)).setSourceNameInTable("Historical Observed");
        ((DefaultXYChartDataSource)engine.getDataSources().get(2)).setSourceNameInTable("Reforecasts");

        return engine;
    }

}
