package ohd.hseb.hefs.mefp.sources.rfcfcst;

import java.awt.BorderLayout;
import java.awt.Dimension;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.io.File;
import java.util.Collection;
import java.util.List;

import javax.swing.JButton;
import javax.swing.JOptionPane;
import javax.swing.event.ListSelectionEvent;
import javax.swing.event.ListSelectionListener;

import nl.wldelft.util.timeseries.TimeSeriesArray;
import ohd.hseb.hefs.mefp.pe.core.MEFPParameterEstimatorRunInfo;
import ohd.hseb.hefs.mefp.sources.rfcfcst.database.ConnectionEditorPanel;
import ohd.hseb.hefs.mefp.tools.QuestionableTools;
import ohd.hseb.hefs.pe.core.ParameterEstimatorStepOptionsPanel;
import ohd.hseb.hefs.pe.core.StepUnit;
import ohd.hseb.hefs.pe.notice.SelectedIdentifiersChangedNotice;
import ohd.hseb.hefs.pe.notice.StepUpdatedNotice;
import ohd.hseb.hefs.pe.tools.GenericSummaryTablePanel;
import ohd.hseb.hefs.pe.tools.LocationAndDataTypeIdentifier;
import ohd.hseb.hefs.utils.gui.components.ImportDialog;
import ohd.hseb.hefs.utils.gui.help.HelpFile;
import ohd.hseb.hefs.utils.gui.tools.HSwingFactory;
import ohd.hseb.hefs.utils.gui.tools.SwingTools;
import ohd.hseb.hefs.utils.jobs.GenericJob;
import ohd.hseb.hefs.utils.jobs.HJobMonitorDialog;
import ohd.hseb.hefs.utils.jobs.JobListener;
import ohd.hseb.hefs.utils.tools.ListTools;
import ohd.hseb.hefs.utils.tools.StringTools;

import com.google.common.collect.ImmutableList;

/**
 * Design: Display a table of location ids based on RFC forecast read in (i.e., Locations Table). For each location,
 * allow for specifying either archive db input (forecast lid and SHEF pedstep and observed SHEF pedstep and maybe lid)
 * or synthetic station computation (MOS-driven synthetic station location id).
 * 
 * @author hank.herr
 */
@SuppressWarnings("serial")
@HelpFile("PDFHELP:helpManual.pdf#RFCForecastsSubpanelRef")
public class RFCForecastPEStepOptionsPanel extends ParameterEstimatorStepOptionsPanel implements ActionListener
{
    private final RFCForecastDataHandler _handler;
    private final RFCDataOptions _options;

    private GenericSummaryTablePanel _locationPanel;
    private RFCLocationTableModel _locationModel;
    private final JButton _viewButton = HSwingFactory.createJButtonWithIcon("hefsIcons/chart20x20.png",
                                                                            "View the RFC forecasts and observed data for a single, selected location and data type.",
                                                                            this);
    private final JButton _removeQuestionableButton = HSwingFactory.createJButtonWithIcon("hefsIcons/removeQuestionableFile20x20.png",
                                                                                          "Remove Questionable Data",
                                                                                          this);

    public RFCForecastPEStepOptionsPanel(final MEFPParameterEstimatorRunInfo runInfo,
                                         final RFCForecastPEStepProcessor stepProcessor)
    {
        super(runInfo, stepProcessor);
        _handler = runInfo.getRFCForecastDataHandler();
        _options = runInfo.getRfcDataOptions();

        _handler.setOptions(_options);
        initializeDisplay();
        setupListeners();

        updateEnablednessOfViewButton();
        updateEnablednessOfRemoveQuestionableButton();
        //updateShefEditor();
    }

    @Override
    public MEFPParameterEstimatorRunInfo getRunInfo()
    {
        return (MEFPParameterEstimatorRunInfo)super.getRunInfo();
    }

    @Override
    protected void initializeDisplay()
    {
        final RFCImportAction action = new RFCImportAction(_handler);
        final JButton importButton = new JButton(action);

        _locationModel = new RFCLocationTableModel(getRunInfo(), getStepProcessor());
        _locationPanel = new GenericSummaryTablePanel(getRunInfo(),
                                                      "Summary of Available RFC Forecast Data",
                                                      _locationModel,
                                                      ImmutableList.of(new JButton(ConnectionEditorPanel.createModalAction(getRunInfo(),
                                                                                                                           SwingTools.getGlobalDialogParent(this))),
                                                                       GenericSummaryTablePanel.SEPARATOR,
                                                                       GenericSummaryTablePanel.DEFAULT_BUTTONS_LEFT,
                                                                       _viewButton,
                                                                       _removeQuestionableButton,
                                                                       importButton,
                                                                       GenericSummaryTablePanel.FILLER,
                                                                       GenericSummaryTablePanel.REFRESH_BUTTON));

        setLayout(new BorderLayout());
        add(_locationPanel, BorderLayout.CENTER);

        _viewButton.setEnabled(false);
        _removeQuestionableButton.setEnabled(false);

        _locationPanel.getTable().getRowSelectionBus().register(importButton);
    }

    @Override
    public RFCForecastPEStepProcessor getStepProcessor()
    {
        return (RFCForecastPEStepProcessor)super.getStepProcessor();
    }

    protected void openImportDialog()
    {
        int row = _locationPanel.getTable().getSelectedRow();
        final LocationAndDataTypeIdentifier identifier = _locationModel.getRow(row);
        final List<File> sources = ListTools.concat(_handler.getPreparedForecastSourcesForIdentifier(identifier),
                                                    _handler.getPreparedObservedSourcesForIdentifier(identifier));
        final ImportDialog dialog = new ImportDialog("Import prepared files for "
            + identifier.buildStringToDisplayInTree(), _handler.getDataHandlerBaseDirectory(), sources);
        dialog.setModal(true);
        dialog.setVisible(true);

        row = _locationPanel.getTable().convertRowIndexToModel(row);
        _handler.getOptions().notifySourceWasImported(identifier, this);
        _locationModel.fireTableRowsUpdated(row, row);
    }

    private void setupListeners()
    {
        final ListSelectionListener listListener = new ListSelectionListener()
        {
            @Override
            public void valueChanged(final ListSelectionEvent e)
            {
                updateEnablednessOfViewButton();
                updateEnablednessOfRemoveQuestionableButton();
                //updateShefEditor();
            }
        };

        _locationPanel.addListSelectionListener(listListener);
        _locationPanel.addColumnSelectionListener(listListener);
    }

    /**
     * Gets the selected identifiers and, if any of them include questionable files, enables the
     * {@link #_removeQuestionableButton}. Otherwise, the button is disabled.
     */
    private void updateEnablednessOfRemoveQuestionableButton()
    {
        // get the selected identifiers

        final List<LocationAndDataTypeIdentifier> identifiers = _locationPanel.getSelectedIdentifiers();

        if(identifiers != null && !identifiers.isEmpty()) // identifiers have been selected
        {
            final String fcstDirectory = _handler.buildDirectoryName(identifiers.get(0), false);
            final String obsDirectory = _handler.buildDirectoryName(identifiers.get(0), true);
            _removeQuestionableButton.setEnabled(QuestionableTools.isQuestionable(fcstDirectory, identifiers)
                || QuestionableTools.isQuestionable(obsDirectory, identifiers));
            if(_removeQuestionableButton.isEnabled())
            {
                _removeQuestionableButton.setToolTipText("<html>Remove files identifying questionable data for selected locations,"
                    + "<br>so locations are no longer marked as questionable." + "<br>Does not change any data.</html>");
            }
        }
        else
        // no identifiers have been selected
        {
            _removeQuestionableButton.setEnabled(false);
            _removeQuestionableButton.setToolTipText("Select rows for the locations for which to remove the questionable data mark.");
        }
    }

    /**
     * Gets a single selected identifier and, if the step has been performed, enables the {@link #_viewButton}. If more
     * than one is selected or none are selected, the step has not be performed for the selected identifier, then the
     * {@link #_viewButton} is disabled.
     */
    private void updateEnablednessOfViewButton()
    {
        final LocationAndDataTypeIdentifier identifier = _locationPanel.getSelectedIdentifier();
        if(identifier != null)
        {
            _viewButton.setEnabled(_handler.havePreparedDataFilesBeenCreatedAlready(identifier));
            if(_viewButton.isEnabled())
            {
                _viewButton.setToolTipText("View the past RFC forecasts for " + identifier.buildStringToDisplayInTree()
                    + ".");
            }
            else
            {
                _viewButton.setToolTipText("<html>No RFC forecasts available for "
                    + identifier.buildStringToDisplayInTree()
                    + ".<br>Specify archive parameters and click the Prepare RFC Forecast Pairs Button to construct them.</html>");
            }
        }
        else
        {
            _viewButton.setEnabled(false);
            _viewButton.setToolTipText("Select a single row to view the past RFC forecasts for that location and data type.");
        }
    }

    private void loadIdentifier(final LocationAndDataTypeIdentifier identifier) throws Exception
    {
        _handler.clearLoadedTimeSeries();
        _handler.loadPreparedTimeSeries(identifier);
    }

    @Override
    public List<LocationAndDataTypeIdentifier> getStepUnitsToPerform()
    {
        return _locationPanel.getSelectedIdentifiers();
    }

    @Override
    public void reactToSelectedIdentifiersChanged(final SelectedIdentifiersChangedNotice evt)
    {
        _locationModel.setIdentifiers(evt.getIdentifiers());
        _locationPanel.refreshTable();

        // Also refresh the Estimation Location Summary Panel
        post(new StepUpdatedNotice(this, RFCForecastPEStepProcessor.class));
    }

    @Override
    public void gotoUnit(final Collection<StepUnit> units)
    {
        _locationPanel.selectIdentifiers(ListTools.convertCollection(units, (LocationAndDataTypeIdentifier)null));
    }

    @Override
    public void actionPerformed(final ActionEvent event)
    {
        if(event.getSource() == this._viewButton)
        {
            this.viewSelectedRFCReforecasts();
        }
        else if(event.getSource() == this._removeQuestionableButton)
        {
            this.removeQuestionableFilesForSelectedIdentifiers();
        }
    }

    /**
     * Removes the questionable files associated with the selected identifiers and posts a {@link StepUpdatedNotice}.
     */

    private void removeQuestionableFilesForSelectedIdentifiers()
    {
        final List<LocationAndDataTypeIdentifier> identifiers = _locationPanel.getSelectedIdentifiers();

        try
        {
            QuestionableTools.deleteQuestionableFiles(_handler.buildDirectoryName(identifiers.get(0), false),
                                                      identifiers);
            QuestionableTools.deleteQuestionableFiles(_handler.buildDirectoryName(identifiers.get(0), true),
                                                      identifiers);
        }
        catch(final Throwable t)
        {
            t.printStackTrace();
            return;
        }

        _locationPanel.refreshTable();

        // Also refresh the Estimation Location Summary Panel
        post(new StepUpdatedNotice(this, RFCForecastPEStepProcessor.class));

    }

    /**
     * Execute a job to load the reforecast and observed time series for a single selected identifier and display them
     * within an {@link RFCForecastDiagnosticPanel}.
     */
    private void viewSelectedRFCReforecasts()
    {
        final LocationAndDataTypeIdentifier selectedIdent = _locationPanel.getSelectedIdentifier();
        if(selectedIdent == null)
        {
            return;
        }

        final GenericJob buildPanelJob = new GenericJob()
        {
            @Override
            public void processJob()
            {
                setIndeterminate(true);
                updateNote("Loading time series from source...");

                try
                {
                    loadIdentifier(selectedIdent);
                }
                catch(final Exception e)
                {
                    e.printStackTrace();
                    fireProcessJobFailure(new Exception("Error loading RFC forecast and observed time series for location "
                                              + selectedIdent.buildStringToDisplayInTree() + ": " + e.getMessage()),
                                          true);
                    return;
                }

                updateNote("Constructing chart...");
                final Collection<TimeSeriesArray> observedSeries = _handler.getLoadedObservedTimeSeries(selectedIdent);
                final Collection<TimeSeriesArray> forecastSeries = _handler.getLoadedForecastTimeSeries(selectedIdent);

                try
                {
                    final RFCForecastDiagnosticPanel panel = new RFCForecastDiagnosticPanel(selectedIdent,
                                                                                            observedSeries,
                                                                                            forecastSeries,
                                                                                            _handler.loadQuestionableHash(selectedIdent));
                    fireDiagnostic(panel);

                }
                catch(final Exception e)
                {
                    e.printStackTrace();
                }
                endTask();
            }
        };
        final HJobMonitorDialog jobDialog = new HJobMonitorDialog(this,
                                                                  "Building Chart to Display RFC Past Forecasts",
                                                                  buildPanelJob,
                                                                  false);
        buildPanelJob.addListener(new JobListener()
        {
            @Override
            public void processJobFailure(final Exception exc, final GenericJob theJob, final boolean displayMessage)
            {
                //exc.printStackTrace();
                fireDiagnostic("Unable to build chart displaying RFC past forecasts:", exc.getMessage());
                JOptionPane.showMessageDialog(SwingTools.getGlobalDialogParent(RFCForecastPEStepOptionsPanel.this),
                                              StringTools.wordWrap(exc.getMessage(), 100),
                                              "Unable to Build Diagnostic Panel",
                                              JOptionPane.ERROR_MESSAGE);
            }

            @Override
            public void processSuccessfulJobCompletion(final GenericJob theJob)
            {
            }
        });
        buildPanelJob.startJob();
        jobDialog.setMinimumSize(new Dimension(350, 10));
        jobDialog.setModal(true);
        jobDialog.setVisible(true);
    }
    /*
     * public static void main(String args[]) { LoggingTools.initializeStdoutLoggerForDebugging("ohd.hseb");
     * HistoricalDataHandler histHandler = new HistoricalDataHandler(); histHandler.setDataHandlerBaseDirectory(new
     * File("testdata/guiTestBed")); try { histHandler.initialize(); } catch(Exception e) {
     * System.out.println("Unable to initialize handler.  Exception was " + e.getMessage()); System.exit(1); }
     * List<LocationAndDataTypeIdentifier> identifiers = histHandler.getIdentifiersWithData(); RFCForecastDataHandler
     * testHandler; testHandler = new RFCForecastDataHandler(); testHandler.setDataHandlerBaseDirectory(new
     * File("testdata/guiTestBed")); RFCForecastPEStepOptionsPanel panel = new RFCForecastPEStepOptionsPanel();
     * panel.reactToEstimatedLocationsListChange(identifiers); JFrame frame = new JFrame("Options Panel");
     * frame.setSize(600, 400); frame.setContentPane(panel); frame.setVisible(true);
     * frame.setDefaultCloseOperation(JFrame.EXIT_ON_CLOSE); }
     */

}
