package ohd.hseb.hefs.mefp.sources.rfcfcst;

import java.util.List;

import ohd.hseb.hefs.mefp.pe.core.MEFPParameterEstimatorRunInfo;
import ohd.hseb.hefs.mefp.sources.rfcfcst.database.ConnectionEditorPanel;
import ohd.hseb.hefs.pe.core.GenericParameterEstimatorStepProcessor;
import ohd.hseb.hefs.pe.core.ParameterEstimatorStepOptionsPanel;
import ohd.hseb.hefs.pe.core.StepUnit;
import ohd.hseb.hefs.pe.tools.LocationAndDataTypeIdentifier;
import ohd.hseb.hefs.utils.jobs.JobMessenger;
import ohd.hseb.hefs.utils.status.StatusIndicator;

import com.google.common.collect.Lists;

public class RFCForecastPEStepProcessor extends GenericParameterEstimatorStepProcessor
{
    public static final String STATUS_TRUE = "Forecast and observed files were found in the appropriate directory.";
    public static final String STATUS_FALSE = "<html>The files expected for the row's location and data type were not found.<br>"
        + "See the instructions for more information.</html>";
    public static final String STATUS_NULL = "The import options have been changed since the import has last been performed.";

    private final MEFPParameterEstimatorRunInfo _runInfo;
    private final RFCForecastDataHandler _pairsDataHandler;

    /**
     * @param pairsDataHandler An initialized ForecastObservedPairsDataHandler.
     */
    public RFCForecastPEStepProcessor(final MEFPParameterEstimatorRunInfo runInfo)
    {
        _runInfo = runInfo;
        _pairsDataHandler = runInfo.getRFCForecastDataHandler();
    }

    public RFCForecastDataHandler getPairsDataHandler()
    {
        return _pairsDataHandler;
    }

    @Override
    public StatusIndicator getStatus(final StepUnit unit)
    {
        if(!(unit instanceof LocationAndDataTypeIdentifier))
        {
            throw new IllegalArgumentException("Must be passed an Identifier.");
        }
        final LocationAndDataTypeIdentifier identifier = (LocationAndDataTypeIdentifier)unit;

        return _pairsDataHandler.getStatus(identifier);
    }

    @Override
    public void performStep(final StepUnit unit) throws Exception
    {
        JobMessenger.newMonitorSubJob();
        JobMessenger.setMaximumNumberOfSteps(2);
        JobMessenger.updateNote("Retrieving Observation / Forecast Pairs from the Archive Database...");

        //Load the pairs sub job
        JobMessenger.newMonitorSubJob();
        JobMessenger.setIndeterminate(true);
        JobMessenger.updateNote("This could take a while.");
        try
        {
            _pairsDataHandler.clearLoadedTimeSeries();
            final List<LocationAndDataTypeIdentifier> identifiers = Lists.newArrayList((LocationAndDataTypeIdentifier)unit);
            _pairsDataHandler.loadOriginalTimeSeries(identifiers);
        }
        catch(final Exception e)
        {
            JobMessenger.clearMonitorSubJob(); //Indeterminate
            JobMessenger.clearMonitorSubJob(); //2-step progress
            throw e;
        }
        JobMessenger.clearMonitorSubJob(); //Indeterminate

        try
        {
            JobMessenger.updateNote("Writing Time Series...");
            _pairsDataHandler.prepareDataFiles();
            _runInfo.getRfcDataOptions().notifySourceWasPrepared((LocationAndDataTypeIdentifier)unit, this);
            JobMessenger.madeProgress();
        }
        finally
        {
            JobMessenger.clearMonitorSubJob(); //2-step progress
        }
    }

    @Override
    public ParameterEstimatorStepOptionsPanel constructOptionsPanel()
    {
        return new RFCForecastPEStepOptionsPanel(_runInfo, this);
    }

    @Override
    public String getToolTipTextDescribingStep()
    {
        return "Read forecast-observed pairs from archived database and build the required files for parameter estimation.";
    }

    @Override
    public String getShortNameForIdentifierTableColumn()
    {
        return "RFC";
    }

    @Override
    public String getToolTipTextForIdentifierTableColumnHeader()
    {
        return "<html>Displays if the files specifying RFC forecasts and observations for a location and "
            + "data type were found under<br>" + _pairsDataHandler.getPreparedDataFilesDirectory() + ".</html>";
    }

    @Override
    public String getTabNameForStep()
    {
        return "RFC Forecasts";
    }

    @Override
    public String getStepNameForRunButton()
    {
        return "Prepare RFC Archived Forecast Files";
    }

    @Override
    public String getPerformStepPrefix()
    {
        return "Preparing RFC forecast pairs";
    }

    @Override
    public void prepareStep(final boolean skipUserInput) throws Exception
    {
        if(!skipUserInput)
        {
            if(!_pairsDataHandler.getOptions().canConnect())
            {
                ConnectionEditorPanel.openConnectionDialogAndProcessResults(_runInfo, null);

                // If we still haven't connected, fail
                // We don't want to lock them in a loop until they can connect.
                if(!_pairsDataHandler.getOptions().canConnect())
                {
                    throw new Exception("Unable to connect to archive database.");
                }
            }
        }
    }
}
