package ohd.hseb.hefs.mefp.sources.rfcfcst;

import java.awt.Component;
import java.awt.Dimension;
import java.awt.event.ActionEvent;
import java.io.File;
import java.io.FileFilter;
import java.io.IOException;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.swing.AbstractAction;
import javax.swing.JFileChooser;
import javax.swing.JOptionPane;

import ohd.hseb.hefs.mefp.tools.QuestionableMessageMap;
import ohd.hseb.hefs.mefp.tools.QuestionableTools;
import ohd.hseb.hefs.pe.tools.LocationAndDataTypeIdentifier;
import ohd.hseb.hefs.utils.gui.tools.SwingTools;
import ohd.hseb.hefs.utils.jobs.GenericJob;
import ohd.hseb.hefs.utils.jobs.HJobMonitorDialog;
import ohd.hseb.hefs.utils.jobs.JobMessenger;
import ohd.hseb.hefs.utils.tools.FileTools;
import ohd.hseb.hefs.utils.tools.MapTools;
import ohd.hseb.hefs.utils.tools.ParameterId;

import org.apache.logging.log4j.Logger;
import org.apache.logging.log4j.LogManager;

import com.google.common.collect.ImmutableMap;
import com.google.common.collect.Lists;
import com.google.common.io.Files;

@SuppressWarnings("serial")
public class RFCImportAction extends AbstractAction
{
    private final RFCForecastDataHandler _handler;
    private File _lastImportDirectory = null;

    private static final Logger LOG = LogManager.getLogger(RFCImportAction.class);

    public static final Map<String, ParameterId> EXTENSION_PARM_ID;
    static
    {
        EXTENSION_PARM_ID = ImmutableMap.<String, ParameterId>builder()
                                        .put("pfcst06", ParameterId.FMAP)
                                        .put("rfctmxfcst", ParameterId.TFMX)
                                        .put("rfctmnfcst", ParameterId.TFMN)
                                        .put("pobs06", ParameterId.MAP)
                                        .put("rfctmxobs", ParameterId.TMAX)
                                        .put("rfctmnobs", ParameterId.TMIN)
                                        .build();
    }

    public RFCImportAction(final RFCForecastDataHandler handler)
    {
        super("", SwingTools.loadImageIcon("hefsIcons/import20x20.png"));
        putValue(SHORT_DESCRIPTION, "Import Prepared RFC Forecast Files");
        _handler = handler;
    }

    @Override
    public void actionPerformed(final ActionEvent e)
    {
        final JFileChooser chooser = new JFileChooser(_lastImportDirectory);
        chooser.setDialogTitle("Choose a directory from which to import RFC files");
        chooser.setFileSelectionMode(JFileChooser.DIRECTORIES_ONLY);
        final int result = chooser.showDialog(null, "Import Directory");

        if(result == JFileChooser.CANCEL_OPTION)
        {
            return;
        }

        final File directory = chooser.getSelectedFile();
        _lastImportDirectory = directory;
        final FileFilter filter = FileTools.makeExtensionFilter(RFCForecastDataHandler.EXTENSION_PARM_ID.keySet());
        final Set<File> files = FileTools.listFilesRecursively(directory, filter);
        final File targetDirectory = _handler.getPreparedDataFilesDirectory();

        final GenericJob importFilesJob = new GenericJob()
        {
            @Override
            public void processJob()
            {
                setMaximumNumberOfSteps(files.size() + 3);
                setIndeterminate(false);
                JobMessenger.updateNote("Importing (copying) " + files.size()
                    + " files...                               ");

                // Delay the start briefly to make sure the progress dialog is displayed so that any error windows
                // are displayed on top of it.

                try
                {
                    Thread.sleep(100);
                }
                catch(final InterruptedException e)
                {
                }

                //Maintains a list of the files already processed.
                final List<File> processedTargetFiles = Lists.newArrayList();

                for(final File file: files)
                {
                    JobMessenger.updateNote("Importing file " + file.getName());

                    if(isCanceled())
                    {
                        fireProcessJobFailure(new Exception("Import job canceled by user."), true);
                        return;
                    }

                    //Where the current file will be copied... its target location.
                    final File targetFile = new File(targetDirectory, FileTools.getRelativeFile(directory, file)
                                                                               .getPath());
                    processedTargetFiles.add(targetFile);

                    //==========================================================
                    //Identify if the file has a pair.  If not, delete the target file before copying.
                    //Otherwise, check to see if its pair has already been processed.  If not, then delete both
                    //files.  If so, then do not delete, as the other file was already imported and should have
                    //resulted in this being deleted.
                    final File pairedTargetFile = _handler.getPairedFile(targetFile);
                    try
                    {
                        if(pairedTargetFile == null)
                        {
                            FileTools.deleteFileIfItExists(targetFile);
                        }
                        else if(!processedTargetFiles.contains(pairedTargetFile))
                        {
                            FileTools.deleteFileIfItExists(targetFile);
                            FileTools.deleteFileIfItExists(pairedTargetFile);
                        }
                        else if(targetFile.exists())
                        {
                            throw new IllegalStateException("Somehow, the target file has not been removed even though the paired file was processed.  INTERNAL ERROR!");
                        }
                    }
                    catch(final Throwable t)
                    {
                        final int option = JOptionPane.showConfirmDialog(getParentComponent(),
                                                                         "Unable to delete the target file "
                                                                             + targetFile + " or its pair.\nReason: "
                                                                             + t.getMessage()
                                                                             + "\nContinue with import?",
                                                                         "Import File Failure!",
                                                                         JOptionPane.YES_NO_OPTION,
                                                                         JOptionPane.WARNING_MESSAGE);
                        if(option == JOptionPane.NO_OPTION)
                        {
                            fireProcessJobFailure(new Exception("Import job canceled by user due to error."), true);
                            return;
                        }
                        continue; // go to the next file.
                    }

                    //==========================================================
                    // Delete the questionable file associated with the file (if it exists). 
                    // If an exception occurs during deletion, don't continue with the file import.
                    try
                    {
                        final LocationAndDataTypeIdentifier identifier = _handler.getIdentifierForSource(targetFile);
                        QuestionableTools.deleteQuestionableFile(targetFile.getParent(), identifier);
                    }
                    catch(final Throwable t)
                    {
                        final int option = JOptionPane.showConfirmDialog(getParentComponent(),
                                                                         "Unable to delete the questionable file for "
                                                                             + targetFile + ".\nReason: "
                                                                             + t.getMessage()
                                                                             + "\nContinue with import?",
                                                                         "Import File Failure!",
                                                                         JOptionPane.YES_NO_OPTION,
                                                                         JOptionPane.WARNING_MESSAGE);
                        if(option == JOptionPane.NO_OPTION)
                        {
                            fireProcessJobFailure(new Exception("Import job canceled by user due to error."), true);
                            return;
                        }
                        continue; // go to the next file.
                    }

                    //==========================================================
                    try
                    {
                        // Create the target directory if needed and copy the file into place now.
                        Files.createParentDirs(targetFile);
                        Files.touch(targetFile);
                        Files.copy(file, targetFile);
                        _handler.notifySourceWasImported(targetFile, this);

                        // Create a questionable file for the imported file (if necessary)
                        // Note that this will not create anything for the first of two paired files; only when the second is processed.
                        QuestionableTools.createRFCQuestionableFile(targetFile, LOG);
                    }
                    catch(final IOException ex) // the copy fails
                    {
                        final int option = JOptionPane.showConfirmDialog(getParentComponent(),
                                                                         "Unable to copy file "
                                                                             + FileTools.determineFileNameRelativeToBaseDirectory(directory,
                                                                                                                                  file)
                                                                             + "\ninto " + targetFile.getAbsolutePath()
                                                                             + ".\nReason: " + ex.getMessage()
                                                                             + "\nContinue with import?",
                                                                         "Import File Failure!",
                                                                         JOptionPane.YES_NO_OPTION,
                                                                         JOptionPane.WARNING_MESSAGE);
                        if(option == JOptionPane.NO_OPTION)
                        {
                            fireProcessJobFailure(new Exception("Import job canceled by user due to error."), true);
                            return;
                        }
                    }
                    catch(final Throwable t) // the questionable file can't be created
                    {
                        final int option = JOptionPane.showConfirmDialog(getParentComponent(),
                                                                         "Unable to create a questionable file for "
                                                                             + targetFile + ".\nReason: "
                                                                             + t.getMessage()
                                                                             + "\nContinue with import?",
                                                                         "Import File Failure!",
                                                                         JOptionPane.YES_NO_OPTION,
                                                                         JOptionPane.WARNING_MESSAGE);
                        if(option == JOptionPane.NO_OPTION)
                        {
                            fireProcessJobFailure(new Exception("Import job canceled by user due to error."), true);
                            return;
                        }
                    }
                    JobMessenger.madeProgress();
                }

                JobMessenger.madeProgress();
                endTask();
            }
        };

        Component parent = null;
        if(e.getSource() instanceof Component)
        {
            parent = (Component)e.getSource();
        }

        final HJobMonitorDialog jobDialog = new HJobMonitorDialog(parent,
                                                                  "Import " + files.size() + " RFC Files",
                                                                  importFilesJob,
                                                                  true);
        importFilesJob.setParentComponent(jobDialog);
        importFilesJob.startJob();
        jobDialog.setMinimumSize(new Dimension(350, 10));
        jobDialog.setModal(true);
        jobDialog.setVisible(true);
    }

    /**
     * @param identifier - identifier used to construct the questionableFile
     * @return The map created by {@link QuestionableTools#toHash(java.io.File)}.
     * @throws Exception
     */
    public HashMap<String, HashMap<Long, HashMap<Long, String>>> loadQuestionableHash(final LocationAndDataTypeIdentifier identifier) throws Exception
    {
        File questionableFile;

        // Get the fcstHash
        HashMap<String, HashMap<Long, HashMap<Long, String>>> fcstHash = null;
        questionableFile = QuestionableTools.getFile(buildDirectoryName(identifier, false), identifier);
        if(questionableFile.exists())
        {
            fcstHash = new QuestionableMessageMap(questionableFile);
        }

        // Get the obsHash
        HashMap<String, HashMap<Long, HashMap<Long, String>>> obsHash = null;
        questionableFile = QuestionableTools.getFile(buildDirectoryName(identifier, true), identifier);
        if(questionableFile.exists())
        {
            obsHash = new QuestionableMessageMap(questionableFile);
        }

        if(fcstHash != null)
        {
            if(obsHash != null) // put obsHash into fcstHash
            {
                MapTools.concatenateMaps(fcstHash, obsHash);
            }

            return (fcstHash);
        }
        else if(obsHash != null)
        {
            return (obsHash);
        }

        // If you got here, both fcstHash and obsHash are null

        return null;
    }

    /**
     * @param identifier Identifier for which to build the directory name for Imported RFC files.
     * @param observed True to get the directory for observed data, false for forecast data.
     * @return The complete path name for the directory that contains data files for the identifier and type of time
     *         series: observed or forecast.
     */
    public String buildDirectoryName(final LocationAndDataTypeIdentifier identifier, final boolean observed)
    {
        if(identifier.isPrecipitationDataType())
        {
            if(observed)
            {
                return _handler.getPreparedDataFilesDirectory() + "/rfc_pobs06";
            }
            else
            {
                return _handler.getPreparedDataFilesDirectory() + "/rfc_pfcst06";
            }
        }
        else
        // temp
        {
            if(observed)
            {
                return _handler.getPreparedDataFilesDirectory() + "/rfc_tobs";
            }
            else
            {
                return _handler.getPreparedDataFilesDirectory() + "/rfc_tfcst";
            }
        }
    }
}
