package ohd.hseb.hefs.mefp.sources.rfcfcst;

import java.util.Collection;
import java.util.Collections;

import javax.swing.JOptionPane;
import javax.swing.table.TableCellEditor;

import ohd.hseb.hefs.mefp.pe.core.MEFPParameterEstimatorRunInfo;
import ohd.hseb.hefs.mefp.sources.rfcfcst.database.DatabaseConnectionSpecification;
import ohd.hseb.hefs.pe.core.ParameterEstimatorStepTable;
import ohd.hseb.hefs.pe.core.StepTableModel;
import ohd.hseb.hefs.pe.tools.DefaultStepTableStatusProvider;
import ohd.hseb.hefs.pe.tools.LocationAndDataTypeIdentifier;
import ohd.hseb.hefs.pe.tools.LocationAndDataTypeIdentifierList;
import ohd.hseb.hefs.utils.gui.jtable.TableTools;
import ohd.hseb.hefs.utils.gui.jtable.models.DefaultEditorsAndRenderersOverrideTableModel;
import ohd.hseb.hefs.utils.gui.tools.SwingTools;
import ohd.hseb.hefs.utils.notify.ObjectModifiedNotice;
import ohd.hseb.hefs.utils.tools.ParameterId;

import com.google.common.collect.Lists;
import com.google.common.eventbus.Subscribe;

/**
 * Directly modifies the options of the passed in handler.
 * 
 * @author alexander.garbarino
 */
@SuppressWarnings("serial")
public class RFCLocationTableModel extends StepTableModel implements ObjectModifiedNotice.Subscriber,
ConnectionChangeListener, DefaultEditorsAndRenderersOverrideTableModel
{
    private static final String OPEN_EDITOR = "Open Editor";
    private static final String CELL_COPY = "Copy to All";
    public static final int[] COLUMN_WIDTHS = {60, 60, 60};

    private final RFCForecastDataHandler _handler;
    private RFCLocationTable _table = null;

    public RFCLocationTableModel(final MEFPParameterEstimatorRunInfo runInfo,
                                 final RFCForecastPEStepProcessor stepProcessor)
    {
        super(runInfo, stepProcessor.getClass(), new DefaultStepTableStatusProvider(stepProcessor));
        _handler = runInfo.getRFCForecastDataHandler();
        _handler.getOptions().register(this);
    }

    public void setTable(final RFCLocationTable table)
    {
        _table = table;
    }

    private boolean isPrecipitation()
    {
        return getRunInfo().getSelectedDataType() == ParameterId.Type.PRECIPITATION;
    }

    private int getPedtsepIndexForColumn(final int col)
    {
        return col - super.getColumnCount() - 1;
    }

    @Override
    public Integer getPreferredWidth(int index)
    {
        if(index < super.getColumnCount())
        {
            return super.getPreferredWidth(index);
        }
        index -= super.getColumnCount();
        if(0 <= index && index < COLUMN_WIDTHS.length)
        {
            return COLUMN_WIDTHS[index];
        }
        return null;
    }

    @Override
    public void setIdentifiers(final Collection<LocationAndDataTypeIdentifier> identifiers)
    {
        //This can NOT call the super class version, because it calls fireTableDataChanged.  However,
        //for this version, it needs to fire a table structure changed.  In JXTable, doing fireTableDataChanged
        //causes it to crash because the structure wasn't changed yet.
        _identifiers = new LocationAndDataTypeIdentifierList(identifiers);
        Collections.sort(_identifiers);
        this.fireTableStructureChanged();
    }

    @Override
    public int getColumnCount()
    {
        final int base = super.getColumnCount() + 2;
        final ParameterId.Type type = getRunInfo().getSelectedDataType();
        switch(type)
        {
            case PRECIPITATION:
                return base;
            case TEMPERATURE:
                return base + 1;
            default:
                throw new IllegalStateException("Unknown data type [" + type + "].");
        }
    }

    @Override
    public String getColumnName(final int column)
    {
        if(column == super.getColumnCount() + 0)
        {
            return "ADB Location ID";
        }
        else if(column == super.getColumnCount() + 1)
        {
            if(isPrecipitation())
            {
                return "SHEF Code";
            }
            else
            {
                return "TMAX SHEF Code";
            }
        }
        else if(column == super.getColumnCount() + 2)
        {
            return "TMIN SHEF Code";
        }
        return super.getColumnName(column);
    }

    @Override
    public Object getValueAt(final int row, final int column)
    {
        if(column == super.getColumnCount() + 0)
        {
            return _handler.getOptions().getSourceFor(getIdentifier(row)).getFilter().getLocationId();
        }
        else if(column == super.getColumnCount() + 1)
        {
            return _handler.getOptions().getSourceFor(getIdentifier(row)).getFilter().getCodes().get(0).asForecast();
        }
        else if(column == super.getColumnCount() + 2)
        {
            return _handler.getOptions().getSourceFor(getIdentifier(row)).getFilter().getCodes().get(1).asForecast();
        }
        return super.getValueAt(row, column);
    }

    @Override
    public boolean isCellEditable(final int row, final int column)
    {
        return column >= super.getColumnCount();
    }

    @Override
    public void setValueAt(final Object value, final int row, final int col)
    {
        if(col == super.getColumnCount()) //location id
        {
            _handler.getOptions().setSourceLocation(getIdentifier(row), (String)value);
        }
        else
        // pedtsep code
        {
            final int index = getPedtsepIndexForColumn(col);

            //Open editor option
            if(value.equals(OPEN_EDITOR))
            {
                final RFCPedtsepSubEditor editor = new RFCPedtsepSubEditor((MEFPParameterEstimatorRunInfo)getRunInfo(),
                                                                           getIdentifier(row),
                                                                           index);
                editor.setVisible(true);
                JOptionPane.showMessageDialog(SwingTools.getGlobalDialogParent(_table),
                                              editor,
                                              "Edit Forecast PEDTSEP Code",
                                              JOptionPane.QUESTION_MESSAGE);
            }
            //All option
            else if(value.equals(CELL_COPY))
            {
                final PedtsepCode code = (PedtsepCode)getValueAt(row, col);
                for(int i = 0; i < getRowCount(); i++)
                {
                    _handler.getOptions().setSourceCode(getIdentifier(i), index, code);
                }
                this.fireTableDataChanged();
            }
            //A code is given, which may or may not be bad.
            else
            {
                PedtsepCode code = null;
                try
                {
                    if(value instanceof String)
                    {
                        code = PedtsepCode.make((String)value);
                    }
                    else
                    {
                        code = (PedtsepCode)value;
                    }
                }
                catch(final IllegalArgumentException e)
                {
                    JOptionPane.showMessageDialog(SwingTools.getGlobalDialogParent(_table), "The PEDTSEP code given, '"
                        + (String)value + "', is not a valid code.", "Bad PEDTSEP Code", JOptionPane.ERROR_MESSAGE);
                    return;
                }

                if(code.isObserved())
                {
                    JOptionPane.showMessageDialog(SwingTools.getGlobalDialogParent(_table),
                                                  "Type source specified in the PEDTSEP code, '" + code.getTS()
                                                      + "', is for observed data.\nA forecast type source is required.",
                                                  "Bad Type Source",
                                                  JOptionPane.ERROR_MESSAGE);
                    return;
                }

                _handler.getOptions().setSourceCode(getIdentifier(row), index, code);
            }
        }
        fireTableDataChanged();
    }

    @Override
    public Class<?> getColumnClass(final int columnIndex)
    {
        if(columnIndex == super.getColumnCount() + 1 || columnIndex == super.getColumnCount() + 2)
        {
            return PedtsepCode.class;
        }
        else if(columnIndex >= super.getColumnCount())
        {
            return String.class;
        }
        return super.getColumnClass(columnIndex);
    }

    @Override
    public String getColumnHeaderToolTip(final int visibleColIndex)
    {
        if(visibleColIndex == super.getColumnCount())
        {
            return "Location ID to grab from Archival Database.";
        }
        else if(visibleColIndex == super.getColumnCount() + 1)
        {
            if(isPrecipitation())
            {
                return "SHEF code to grab from Archival Database.";
            }
            else
            {
                return "TMAX SHEF code to grab from Archival Database.";
            }

        }
        else if(visibleColIndex == super.getColumnCount() + 2)
        {
            return "TMIN SHEF code to grab from Archival Database.";
        }
        return super.getColumnHeaderToolTip(visibleColIndex);
    }

    @Override
    public String getCellToolTip(final int rowIndex, final int visibleColIndex)
    {
        return super.getCellToolTip(rowIndex, visibleColIndex);
    }

    @Override
    public void setupOverrideEditorsAndRenderers()
    {
        getTable().setDefaultEditor(PedtsepCode.class,
                                    TableTools.buildComboBoxEditor(Lists.newArrayList(OPEN_EDITOR, CELL_COPY)));
    }

    @Override
    public void setConnection(final DatabaseConnectionSpecification connection)
    {
        if(connection == null || connection.equals(_handler.getOptions().getDefaultConnection()))
        {
            return;
        }

        _handler.getOptions().setDefaultConnection(connection);
        for(int i = 0; i < this.getRowCount(); i++)
        {
            _handler.getOptions().getSourceFor(getIdentifier(i)).setConnection(connection);
        }

        this.fireTableDataChanged();
    }

    @Override
    protected RFCLocationTable instantiateDefaultTable()
    {
        return new RFCLocationTable();
    }

    @Override
    @Subscribe
    public void reactToObjectModified(final ObjectModifiedNotice notice)
    {
        if(notice.getTarget().equals(_handler.getOptions()))
        {
            fireTableDataChanged();
        }
    }

    private class RFCLocationTable extends ParameterEstimatorStepTable
    {
        private RFCLocationTable()
        {
            super(RFCForecastPEStepProcessor.class, RFCLocationTableModel.this);
            RFCLocationTableModel.this.setTable(this);
            putClientProperty("terminateEditOnFocusLost", true);
        }

        @Override
        public void setDefaultEditor(final Class<?> columnClass, final TableCellEditor editor)
        {
            super.setDefaultEditor(columnClass, editor);
        }

        @Override
        public MEFPParameterEstimatorRunInfo getRunInfo()
        {
            return (MEFPParameterEstimatorRunInfo)super.getRunInfo();
        }
    }

}
