package ohd.hseb.hefs.mefp.sources.rfcfcst;

/**
 * An observed/forecast value pairing, along with its SHEF code.
 * 
 * @author alexander.garbarino
 */
public class VfyPair implements Comparable<VfyPair>, Cloneable
{
    public static final int DEFAULT_QUALITY_CODE = 1879048191;

    private final String _locationId;
    private final VfyPairPedtsepCode _code;
    private long _validTime = Long.MIN_VALUE;
    private long _basisTime = Long.MIN_VALUE;
    private long _observedTime = Long.MIN_VALUE;
    private double _forecastValue = Double.NaN;
    private double _observedValue = Double.NaN;
    private int _qualityCode = DEFAULT_QUALITY_CODE;

    public VfyPair(final String location,
                   final VfyPairPedtsepCode code,
                   final long vTime,
                   final long bTime,
                   final long oTime,
                   final double fVal,
                   final double oVal)
    {
        this(location, code, vTime, bTime, oTime, fVal, oVal, DEFAULT_QUALITY_CODE);
    }

    public VfyPair(final String location,
                   final VfyPairPedtsepCode code,
                   final long vTime,
                   final long bTime,
                   final long oTime,
                   final double fVal,
                   final double oVal,
                   final int qualCode)
    {
        _locationId = location;
        _code = code;
        _validTime = vTime;
        _basisTime = bTime;
        _observedTime = oTime;
        _forecastValue = fVal;
        _observedValue = oVal;
        _qualityCode = qualCode;
    }

    @Override
    public String toString()
    {
        return String.format("%s || %s[%s/%s] || %ts - %ts - %ts || %f - %f || %d",
                             _locationId,
                             _code,
                             _code.getObservedTS(),
                             _code.getForecastTS(),
                             _validTime,
                             _basisTime,
                             _observedTime,
                             _forecastValue,
                             _observedValue,
                             _qualityCode);
    }

    @Override
    public boolean equals(final Object other)
    {
        if(!(other instanceof VfyPair))
        {
            return false;
        }
        final VfyPair o = (VfyPair)other;
        return _locationId.equals(o._locationId) && _code.equals(o._code) && _validTime == o._validTime
            && _basisTime == o._basisTime && _observedTime == o._observedTime && _forecastValue == o._forecastValue
            && _observedValue == o._observedValue && _qualityCode == o._qualityCode;
    }

    @Override
    public int hashCode()
    {
        return toString().hashCode();
    }

    public String getLocationId()
    {
        return _locationId;
    }

    /**
     * Gets a copy of the pedtsep code for this pair.
     * 
     * @return
     */
    public VfyPairPedtsepCode getPedtsepCode()
    {
        return _code;
    }

    public long getValidTime()
    {
        return _validTime;
    }

    public long getBasisTime()
    {
        return _basisTime;
    }

    public long getObservedTime()
    {
        return _observedTime;
    }

    public double getForecastValue()
    {
        return _forecastValue;
    }

    public double getObservedValue()
    {
        return _observedValue;
    }

    public int getQualityCode()
    {
        return _qualityCode;
    }

    /**
     * Sorts by location, code, basis, valid, then observed time.
     */
    @Override
    public int compareTo(final VfyPair other)
    {
        int comp;
        comp = _locationId.compareTo(other._locationId);
        if(comp == 0)
        {
            //Observed TS is ignored when sorting pairs, because it is not in the primary key.
            comp = _code.asForecast().toString().compareTo(other._code.asForecast().toString());
            if(comp == 0)
            {
                comp = new Long(_basisTime).compareTo(other._basisTime);
                if(comp == 0)
                {
                    comp = new Long(_validTime).compareTo(other._validTime);
                    if(comp == 0)
                    {
                        comp = new Long(_observedTime).compareTo(other._observedTime);
                    }
                }
            }
        }
        return comp;
    }

    @Override
    public VfyPair clone()
    {
        try
        {
            return (VfyPair)super.clone();
        }
        catch(final CloneNotSupportedException e)
        {
            throw new RuntimeException(e);
        }
    }

}
