package ohd.hseb.hefs.mefp.sources.rfcfcst;

import java.util.concurrent.TimeUnit;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import net.jcip.annotations.Immutable;
import nl.wldelft.util.timeseries.TimeSeriesArray;
import nl.wldelft.util.timeseries.TimeSeriesHeader;
import ohd.hseb.hefs.pe.tools.LocationAndDataTypeIdentifier;
import ohd.hseb.hefs.utils.Duration;
import ohd.hseb.hefs.utils.tools.ComparableTools;
import ohd.hseb.hefs.utils.tools.ParameterId;
import ohd.hseb.hefs.utils.xml.XMLReader;
import ohd.hseb.hefs.utils.xml.XMLReaderException;
import ohd.hseb.hefs.utils.xml.XMLReaderFactory;

import org.xml.sax.Attributes;

import com.google.common.base.Supplier;

/**
 * SHEF pedtsep info holder, extended for VfyPair's duplicate TS value. Uses '_' to specify unset values.<br>
 * <br>
 * NOTE: though this will track the observe TS, {@link #_ots}, it is important to remember that the observed TS is not
 * part of the primary key of a vfypair, so it is generally not important. For example, it should not be used in sorting
 * pairs.
 * 
 * @author alexander.garbarino
 */
@Immutable
public class VfyPairPedtsepCode extends PedtsepCodeBase
{
    public static final String XML_TAG = "vfyPairPedtsepCode";

    private String _ots = null;
    private String _fts = null;

    public VfyPairPedtsepCode()
    {
        super(null, null, null, null);
        _ots = null;
        _fts = null;
    }

    public VfyPairPedtsepCode(final String pe,
                              final String d,
                              final String ots,
                              final String fts,
                              final String e,
                              final String p)
    {
        super(processPE(pe), processD(d), processE(e), processP(p));
        _ots = processTS(ots);
        _fts = processTS(fts);
    }

    /**
     * Recreate the code from the given toString() result.
     * 
     * @param code the code in a string format
     */
    public static VfyPairPedtsepCode make(final String code)
    {
        final Matcher matcher = Pattern.compile("^(..)(.)\\((..)\\|(..)\\)(.)(.)$").matcher(code);
        if(!matcher.matches())
        {
            throw new IllegalArgumentException("Provided code is not valid.");
        }

        return new VfyPairPedtsepCode(matcher.group(1),
                                      matcher.group(2),
                                      matcher.group(3),
                                      matcher.group(4),
                                      matcher.group(5),
                                      matcher.group(6));
    }

    /**
     * Creates a pair code which will match the same values as the given single code.
     * 
     * @param code the code to match
     */
    public static VfyPairPedtsepCode make(final PedtsepCode code)
    {
        VfyPairPedtsepCode newCode = new VfyPairPedtsepCode();
        newCode = newCode.withPE(code.getPE());
        newCode = newCode.withD(code.getD());
        newCode = newCode.withE(code.getE());
        newCode = newCode.withP(code.getP());
        if(!code.isObserved())
        {
            newCode = newCode.withForecastTS(code.getTS());
        }
        if(!code.isForecast())
        {
            newCode = newCode.withObservedTS(code.getTS());
        }
        return newCode;
    }

    /**
     * Makes a pedtsep code for the two given series.
     * 
     * @param observed the observed series header
     * @param forecast the forecast series header
     */
    public static VfyPairPedtsepCode make(final TimeSeriesHeader observed, final TimeSeriesHeader forecast)
    {
        VfyPairPedtsepCode code = new VfyPairPedtsepCode();
        final ParameterId observedId = ParameterId.of(observed);

        if(observedId.isPrecipitation())
        {
            code = code.withPE("PP");
        }
        else if(observedId.isTemperature())
        {
            code = code.withPE("TA");
        }
        else
        {
            throw new IllegalArgumentException("Unknown Parameter Type.");
        }

        // Grab shortest duration of the two.
        try
        {
            final Duration duration = ComparableTools.min(new Duration(observed.getTimeStep().getStepMillis(),
                                                                       TimeUnit.MILLISECONDS),
                                                          new Duration(forecast.getTimeStep().getStepMillis(),
                                                                       TimeUnit.MILLISECONDS));
            final Pedtsep.D d = Pedtsep.D.forDuration(duration);
            if(d != null)
            {
                code = code.withD(d.toString());
            }
            else
            {
                throw new IllegalArgumentException("Illegal duration.");
            }
        }
        catch(final UnsupportedOperationException e) // Irregular Timestep.
        {
        }

        code = code.withObservedTS("RA");
        code = code.withForecastTS("FA");

        if(observedId.isMax())
        {
            code = code.withE("X");
        }
        else if(observedId.isMin())
        {
            code = code.withE("N");
        }
        else
        {
            code = code.withE("Z");
        }

        code = code.withP("Z");

        return code;
    }

    public static VfyPairPedtsepCode make(final TimeSeriesArray observed, final TimeSeriesArray forecast)
    {
        return make(observed.getHeader(), forecast.getHeader());
    }

    public static VfyPairPedtsepCode make(final LocationAndDataTypeIdentifier identifier)
    {
        return make(identifier.makeHeader(), identifier.makeHeader());
    }

    public VfyPairPedtsepCode withPE(final String pe)
    {
        return new VfyPairPedtsepCode(processPE(pe), getD(), _ots, _fts, getE(), getP());
    }

    public VfyPairPedtsepCode withD(final String d)
    {
        return new VfyPairPedtsepCode(getPE(), processD(d), _ots, _fts, getE(), getP());
    }

    public VfyPairPedtsepCode withObservedTS(final String ots)
    {
        return new VfyPairPedtsepCode(getPE(), getD(), processTS(ots), _fts, getE(), getP());
    }

    public VfyPairPedtsepCode withForecastTS(final String fts)
    {
        return new VfyPairPedtsepCode(getPE(), getD(), _ots, processTS(fts), getE(), getP());
    }

    public VfyPairPedtsepCode withE(final String e)
    {
        return new VfyPairPedtsepCode(getPE(), getD(), _ots, _fts, processE(e), getP());
    }

    public VfyPairPedtsepCode withP(final String p)
    {
        return new VfyPairPedtsepCode(getPE(), getD(), _ots, _fts, getE(), processP(p));
    }

    @Override
    public String toString()
    {
        String rep = "";
        rep += getPE() != null ? getPE() : "__";
        rep += getD() != null ? getD() : "_";
        rep += "(";
        rep += _ots != null ? _ots : "__";
        rep += "|";
        rep += _fts != null ? _fts : "__";
        rep += ")";
        rep += getE() != null ? getE() : "_";
        rep += getP() != null ? getP() : "_";
        return rep;
    }

    public String getObservedTS()
    {
        return _ots;
    }

    public String getForecastTS()
    {
        return _fts;
    }

    /**
     * @return the observed pedtsep code
     */
    public PedtsepCode asObserved()
    {
        return new PedtsepCode(getPE(), getD(), _ots, getE(), getP());
    }

    /**
     * @return the forecast pedtsep code
     */
    public PedtsepCode asForecast()
    {
        return new PedtsepCode(getPE(), getD(), _fts, getE(), getP());
    }

    @Override
    public String getXMLTagName()
    {
        return XML_TAG;
    }

    public static class Reader implements XMLReader, Supplier<VfyPairPedtsepCode>
    {
        public static final XMLReaderFactory<Reader> FACTORY = new XMLReaderFactory<Reader>()
        {
            @Override
            public Reader get()
            {
                return new Reader();
            }
        };

        private VfyPairPedtsepCode _code;

        public Reader()
        {
            _code = new VfyPairPedtsepCode();
        }

        @Override
        public VfyPairPedtsepCode get()
        {
            return _code;
        }

        @Override
        public XMLReader getReader()
        {
            return this;
        }

        @Override
        public String getXMLTagName()
        {
            return XML_TAG;
        }

        @Override
        public void setValueOfElement(final String elementName, final String value) throws XMLReaderException
        {
            if(!elementName.equals(getXMLTagName()))
            {
                throw new XMLReaderException("Unknown tag " + elementName);
            }
            _code = VfyPairPedtsepCode.make(value);
        }

        @Override
        public XMLReader readInPropertyFromXMLElement(final String elementName, final Attributes attr) throws XMLReaderException
        {
            return null;
        }

        @Override
        public void finalizeReading() throws XMLReaderException
        {
        }

        @Override
        public void validate() throws XMLReaderException
        {
        }
    }
}
