package ohd.hseb.hefs.mefp.sources.rfcfcst;

import static com.google.common.collect.Maps.newHashMap;

import java.util.Collection;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.TreeSet;

import nl.wldelft.util.timeseries.DefaultTimeSeriesHeader;
import nl.wldelft.util.timeseries.ParameterType;
import nl.wldelft.util.timeseries.SimpleEquidistantTimeStep;
import nl.wldelft.util.timeseries.TimeSeriesArray;
import ohd.hseb.hefs.pe.tools.TimeSeriesSorter;
import ohd.hseb.hefs.utils.datetime.DateTools;
import ohd.hseb.hefs.utils.tools.IterableTools;
import ohd.hseb.hefs.utils.tsarrays.TimeSeriesArrayTools;
import ohd.hseb.util.misc.HCalendar;

import org.apache.logging.log4j.Logger;
import org.apache.logging.log4j.LogManager;

/**
 * A set of vfy pairs, kept sorted in natural order. All pairs must have the same SHEF code. Also deals with converting
 * from pairs to series and vice versa. English units are assumed: IN and DEGF.
 * 
 * @author alexander.garbarino
 * @author hank.herr
 */
public class VfyPairSet extends TreeSet<VfyPair>
{
    private static final long serialVersionUID = 1L;
    private static final Logger LOG = LogManager.getLogger(VfyPairSet.class);
    private final static String OBSERVED_FILLER_TS = "XX";

    /**
     * Empty constructor.
     */
    public VfyPairSet()
    {
    }

    /**
     * Creates a pair set containing all the given pairs.
     * 
     * @param pairColl the pairs to incorporate in the new set
     * @throws IllegalArgumentException if the collection has pairs with different locations or SHEF codes
     */
    public VfyPairSet(final Collection<VfyPair> pairColl)
    {
        this.addAll(pairColl);
    }

    /**
     * Creates a pairset that will only accept pairs with the given location and code.
     * 
     * @param location the location to accept
     * @param code the code to accept
     */
    public VfyPairSet(final String location, final VfyPairPedtsepCode code)
    {
        if(location == null)
        {
            throw new IllegalArgumentException("Cannot restrict location to null.");
        }
        if(code == null)
        {
            throw new IllegalArgumentException("Cannot restrict code to null.");
        }
    }

    /**
     * Generate a set of all pairs between the given observed and forecast time series.
     * 
     * @param observed the observed time series
     * @param forecast the forecast time series
     */
    public VfyPairSet(final TimeSeriesArray observed, final TimeSeriesArray forecast)
    {
        super();

        // Sanity Check.
        if(!observed.getHeader().getLocationId().equals(forecast.getHeader().getLocationId()))
        {
            throw new IllegalArgumentException("Series have different location ids.");
        }

        final VfyPairPedtsepCode code = VfyPairPedtsepCode.make(observed, forecast);
        final long forecastTime = forecast.getHeader().getForecastTime();

        for(int i = 0; i < forecast.size(); i++)
        {
            final long validTime = forecast.getTime(i);
            final int observedIndex = observed.indexOfTime(validTime);
            if(observedIndex == -1) // No matching observed time, so skip this forecast.
            {
                continue;
            }

            final VfyPair pair = new VfyPair(observed.getHeader().getLocationId(),
                                             code,
                                             validTime,
                                             forecastTime,
                                             observed.getTime(observedIndex),
                                             forecast.getValue(i),
                                             observed.getValue(observedIndex));
            this.add(pair);
        }
    }

    /**
     * Generate a list of all pairs between the given observed series and all given forecast series.
     * 
     * @param observed the observed time series
     * @param forecastColl all forecast time series
     */
    public VfyPairSet(final TimeSeriesArray observed, final Collection<TimeSeriesArray> forecastColl)
    {
        super();

        for(final TimeSeriesArray forecast: forecastColl)
        {
            final VfyPairSet set = new VfyPairSet(observed, forecast);
            this.addAll(set);
        }
    }

    /**
     * Generate as many possible vfy pairs from the series within the sorter.
     * 
     * @param sorter the sorter to draw pairs from
     */
    public VfyPairSet(final TimeSeriesSorter sorter)
    {
        super();

        // Cycle through each observed / forecast parameter pair.
        final List<TimeSeriesSorter> sorters = sorter.splitByOFPairsAndLocation();
        for(final TimeSeriesSorter pairSorter: sorters)
        {
            // Split into observed and forecast.
            final TimeSeriesArray observed = pairSorter.restrictViewToObserved().iterator().next();
            final TimeSeriesSorter forecast = pairSorter.restrictViewToForecast();
            if(observed != null && !forecast.isEmpty())
            {
                this.addAll(new VfyPairSet(observed, forecast));
            }
        }
    }

    /**
     * @return A time that is on the 12Z clock and for which basisTime is within and window that starts one hour before
     *         the 12Z time and ends one hour before the next time step (based on timeStep). {@link Long#MIN_VALUE} is
     *         returned if the loop breaks down for whatever reason.
     */
    private long determineAssumed12ZClockTimeStep(final long basisTime, final long timeStep)
    {
        //This is the starting point for the search.
        final long first12ZTimeAfter = DateTools.findFirstExact12ZTimeAfterOrOn(basisTime);

        //This should work for any time step of 1 day or less.
        for(long workingTime = first12ZTimeAfter; workingTime >= basisTime - timeStep; workingTime -= timeStep)
        {
            if((basisTime >= workingTime - HCalendar.MILLIS_IN_HR)
                && (basisTime < workingTime + timeStep - HCalendar.MILLIS_IN_HR))
            {
                return workingTime;
            }
        }

        return Long.MIN_VALUE;
    }

    /**
     * @return If any forecast in the provided map has a validtime that is not after the provided T0.
     */
    private boolean isForcastValidtimePresentThatIsNotAfterT0(final long t0, final Map<Long, Float> forecasts)
    {
        for(final Map.Entry<Long, Float> forecast: forecasts.entrySet())
        {
            if(forecast.getKey().longValue() <= t0)
            {
                return true;
            }
        }
        return false;
    }

    /**
     * Creates a forecast array based on the given data.
     * 
     * @param basisTime the basis time of the series, according to the raw data.
     * @param location the location id of the series
     * @param code the pedtsep code for the series
     * @param forecasts map from times to forecast values
     * @return A time series of the provided data with gaps between the start and end filled with NaNs. Null is returned
     *         if the provided basisTime implies a non-12Z forecast. If the basis time is between 12Z and 14Z of its
     *         given day, it will be assumed to be a 12Z basis time forecast and an appropriate time series will be
     *         returned with its first value one step after 12Z.
     */
    private TimeSeriesArray createForecastArray(final long basisTime,
                                                final String location,
                                                final PedtsepCode code,
                                                final Map<Long, Float> forecasts)
    {
        //Determine the 12Z-clock time of the time series that will be constructed based on forecasts.  Work that time
        //backward one step at a time if forecasts include a validtime that is on or before the 12Z time.  This means
        //that the assumed 12Z time is wrong: apparently pairs were posted that had a late basistime relative to what
        //we normally expect.  
        long assumed12ZClockTime = determineAssumed12ZClockTimeStep(basisTime, code.getDurationInMillis());
        while(isForcastValidtimePresentThatIsNotAfterT0(assumed12ZClockTime, forecasts))
        {
            assumed12ZClockTime -= code.getDurationInMillis();
        }

        //If the time is not 12Z, throw it out.
        if(!DateTools.is12ZTime(assumed12ZClockTime))
        {
            return null;
        }

//Debug line for breaking at a particular T0.
//        if(HCalendar.convertStringToCalendar("2010-03-06 12:00:00", HCalendar.DEFAULT_DATE_FORMAT).getTimeInMillis() == forecastTimeCal.getTimeInMillis())
//        {
//            System.out.println("HERE I AM !!!");
//        }

        final DefaultTimeSeriesHeader header = new DefaultTimeSeriesHeader();
        header.setLocationDescription(location);
        header.setLocationId(location);
        header.setLocationName(location);
        final String param = code.asParameterId().toString();
        header.setParameterId(param);
        header.setParameterName(param);
        header.setParameterType(ParameterType.ACCUMULATIVE);
        header.setForecastTime(basisTime);
        header.setTimeStep(SimpleEquidistantTimeStep.getInstance(code.getDurationInMillis()));
        if(code.asParameterId().isPrecipitation())
        {
            header.setUnit("IN");
        }
        else if(code.asParameterId().isTemperature())
        {
            header.setUnit("DEGF");
        }

        //Set the forecast time to the computed calendar.
        header.setForecastTime(assumed12ZClockTime);

        //Create the time series array.  Put a NaN as the start time.  It will be overridden if the start time has a value
        //within the list of forecast values.  If not, it will be used to guide the fillNans that is done later.
        final TimeSeriesArray array = new TimeSeriesArray(header);
        for(final Map.Entry<Long, Float> forecast: forecasts.entrySet())
        {
            array.put(forecast.getKey(), forecast.getValue());
        }

        //Put in the first value if not found and fill in NaNs.  Combined with the determination
        //of the forecast time above, and this should result in a complete time series starting one step after T0.
        if(!array.containsTime(header.getForecastTime() + code.getDurationInMillis()))
        {
//            System.out.println("####>> HERE -- " + (header.getForecastTime() + code.getDurationInMillis()));
            array.put(header.getForecastTime() + code.getDurationInMillis(), Float.NaN);
        }
        TimeSeriesArrayTools.fillNaNs(array);

        return array;
    }

    private static TimeSeriesArray createObservedArray(final String location,
                                                       final PedtsepCode code,
                                                       final Map<Long, Float> observations)
    {
        final DefaultTimeSeriesHeader header = new DefaultTimeSeriesHeader();
        header.setLocationDescription(location);
        header.setLocationId(location);
        header.setLocationName(location);
        final String param = code.asParameterId().toString();
        header.setParameterId(param);
        header.setParameterName(param);
        header.setParameterType(ParameterType.ACCUMULATIVE);
        header.setForecastTime(Long.MIN_VALUE);
        header.setTimeStep(SimpleEquidistantTimeStep.getInstance(code.getDurationInMillis()));
        if(code.asParameterId().isPrecipitation())
        {
            header.setUnit("IN");
        }
        else if(code.asParameterId().isTemperature())
        {
            header.setUnit("DEGF");
        }
        else
        {
            throw new IllegalArgumentException("Time series provided that is neither precipitation nor temperature.");
        }

        final TimeSeriesArray array = new TimeSeriesArray(header);
        for(final Map.Entry<Long, Float> observation: observations.entrySet())
        {
            array.put(observation.getKey(), observation.getValue());
        }

        //Fill any gaps of missing data between the start and end times of the time series.  
        TimeSeriesArrayTools.fillNaNs(array);

        return array;
    }

    /**
     * Adds the tim series to the sorter, checking for duplicate forecast time if needed.
     * 
     * @param ts Time series to add.
     * @param sorter Sorter to which to add it.
     * @param allowDuplicateForecastTimes If true, multiple time series with the same time will be allowed. If false,
     *            time series found after the first with that time will be thrown out and a warning message will be
     *            printed.
     * @return 0 if the time series was added, 1 if it was a duplicate. Return value is used to count number of
     *         duplicates.
     */
    private int addTimeSeriesToSorter(final TimeSeriesArray ts,
                                      final TimeSeriesSorter sorter,
                                      final boolean allowDuplicateForecastTimes)
    {
        if((!allowDuplicateForecastTimes) && (sorter.containsTimeSeriesWithSameForecastTime(ts)))
        {
            LOG.warn("Multiple time series were found for forecast time "
                + HCalendar.buildDateTimeTZStr(ts.getHeader().getForecastTime()) + " for the time series "
                + TimeSeriesArrayTools.createHeaderString(ts)
                + " (location, parameter, ensemble id, member index); all after the first will be ignored.");
            return 1;
        }
        else
        {
            sorter.add(ts);
            return 0;
        }
    }

    /**
     * Takes all the pairs in this set and constructs the time series in can with them. Only 12Z forecasts are included
     * in the sorter, however.
     * 
     * @param allowDuplicateForecastTimes If true, multiple time series with the same time will be allowed. If false,
     *            time series found after the first with that time will be thrown out and a warning message will be
     *            printed.
     * @return A sorter that contains both observed and forecast time series. Use the
     *         {@link TimeSeriesSorter#restrictViewToObserved()} and {@link TimeSeriesSorter#restrictViewToForecast()}
     *         methods to acquire the time series.
     */
    public TimeSeriesSorter constructTimeSeries(final boolean allowDuplicateForecastTimes)
    {
        final TimeSeriesSorter sorter = new TimeSeriesSorter();

        LOG.info("Constructing forecast time series from " + this.size() + " verification forecast-observed pairs.");

        final Iterator<VfyPair> iterator = this.iterator(); // We're sorted by basis time.
        Long basisTime = null;
        String location = null;
        VfyPairPedtsepCode code = null;
        final Map<Long, Float> forecasts = newHashMap(); // Gather all forecasts for a given basis time in here.
        final Map<Long, Float> observations = newHashMap();
        int numberOfTSThrownOut = 0;
        int numberOfDuplicates = 0;

        while(iterator.hasNext())
        {
            final VfyPair pair = iterator.next();

            // Different series, so need to update both forecast and observed.
            //For observations, the observed ts can be different between pairs.  So, we ignore the obs ts by filling
            //it with XX.
            if(location == null || code == null || !location.equals(pair.getLocationId())
                || !code.equals(pair.getPedtsepCode().withObservedTS(OBSERVED_FILLER_TS)))
            {
                if(!observations.isEmpty())
                {
                    sorter.add(createObservedArray(location, code.asObserved().withTS(OBSERVED_FILLER_TS), observations));
                    observations.clear();
                }
                if(!forecasts.isEmpty())
                {
                    final TimeSeriesArray ts = createForecastArray(basisTime, location, code.asForecast(), forecasts);
                    if(ts != null)
                    {
                        numberOfDuplicates += addTimeSeriesToSorter(ts, sorter, allowDuplicateForecastTimes);
                    }
                    forecasts.clear();
                }

                basisTime = pair.getBasisTime();
                location = pair.getLocationId();
                code = pair.getPedtsepCode().withObservedTS(OBSERVED_FILLER_TS);
            }
            // Only need to start over forecast series.
            else if(!(new Long(pair.getBasisTime()).equals(basisTime)))
            {
                if(!forecasts.isEmpty()) // Make the time series
                {
                    final TimeSeriesArray ts = createForecastArray(basisTime, location, code.asForecast(), forecasts);
                    if(ts != null)
                    {
                        numberOfDuplicates += addTimeSeriesToSorter(ts, sorter, allowDuplicateForecastTimes);
                    }
                    else
                    {
                        numberOfTSThrownOut++;
                    }
                    forecasts.clear();
                }

                basisTime = pair.getBasisTime();
            }

            //Observations must be stored by their corresponding forecast valid time, not observed time.
            forecasts.put(pair.getValidTime(), (float)pair.getForecastValue());
            observations.put(pair.getValidTime(), (float)pair.getObservedValue());
        }

        // Add last series.
        if(!forecasts.isEmpty())
        {
            final TimeSeriesArray ts = createForecastArray(basisTime, location, code.asForecast(), forecasts);
            if((ts != null) && (DateTools.is12ZTime(ts.getHeader().getForecastTime())))
            {
                numberOfDuplicates += addTimeSeriesToSorter(ts, sorter, allowDuplicateForecastTimes);
            }
            else
            {
                numberOfTSThrownOut++;
            }
        }
        LOG.info("The pairs yielded " + sorter.size() + " acceptable 12Z forecast time series, " + numberOfTSThrownOut
            + " time series thrown out because of not being 12Z forecasts, and " + numberOfDuplicates
            + " time seires thrown out because of a already used forecast time.");
        LOG.info("The first time series has a forecast time of "
            + HCalendar.buildDateTimeTZStr(IterableTools.first(sorter).getHeader().getForecastTime())
            + " and the last time series has a forecast time of "
            + HCalendar.buildDateTimeTZStr(IterableTools.last(sorter).getHeader().getForecastTime()));

        if(!observations.isEmpty())
        {
            LOG.info("Constructing observed time series from " + observations.size()
                + " unique (based on time) observations found.");
            final TimeSeriesArray ts = createObservedArray(location, code.asObserved(), observations);
            sorter.add(ts);
            LOG.info("Constructed observed time series starts at " + HCalendar.buildDateTimeTZStr(ts.getStartTime())
                + " and ends at " + HCalendar.buildDateTimeTZStr(ts.getEndTime()));
        }

        return sorter;
    }
}
