package ohd.hseb.hefs.mefp.sources.rfcfcst.database;

import java.awt.Component;
import java.awt.Dimension;
import java.awt.FlowLayout;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.Insets;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.WindowEvent;

import javax.swing.AbstractAction;
import javax.swing.Action;
import javax.swing.JButton;
import javax.swing.JLabel;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.JTable;
import javax.swing.JTextField;
import javax.swing.ListSelectionModel;
import javax.swing.table.AbstractTableModel;

import ohd.hseb.hefs.mefp.pe.core.MEFPParameterEstimatorRunInfo;
import ohd.hseb.hefs.mefp.sources.rfcfcst.RFCDataOptions;
import ohd.hseb.hefs.pe.tools.ConnectionAction;
import ohd.hseb.hefs.utils.gui.jtable.GenericTable;
import ohd.hseb.hefs.utils.gui.jtable.models.ToolTipTableModel;
import ohd.hseb.hefs.utils.gui.tools.GenericDialogPanel;
import ohd.hseb.hefs.utils.status.StatusIndicator;
import ohd.hseb.hefs.utils.status.StatusLabel;

import com.google.common.base.Strings;
import com.google.common.eventbus.Subscribe;

/**
 * A panel that allows the user to directly modify the default connection in the given handler.
 * 
 * @author alexander.garbarino
 */
@SuppressWarnings("serial")
public class ConnectionEditorPanel extends GenericDialogPanel implements ConnectionTestedNotice.Subscriber
{
    private final RFCDataOptions _options;
    private final DatabaseConnectionTester _tester;
    private final JTextField _connectionStringField;

    private DatabaseConnectionSpecification _connection;

    private final ConnectionTableModel _model;
    private JTable _table;

    private void processOK()
    {
        _options.setAllConnections(_connection);
    }

    private final Action _retestAction = new AbstractAction("Retest Connection")
    {
        {
            putValue(SHORT_DESCRIPTION, "Forget the results of previous tests and retest all connections.");
        }

        @Override
        public void actionPerformed(ActionEvent e)
        {
            _tester.clearTests();
            _model.fireTableCellUpdated(0, 4);
        }
    };

    public ConnectionEditorPanel(MEFPParameterEstimatorRunInfo runInfo)
    {
        runInfo.register(this);

        _options = runInfo.getRfcDataOptions();
        _tester = runInfo.getRfcDataOptions().getTester();
        _model = new ConnectionTableModel();
        _connectionStringField = makeConnectionStringField();

        this.setPreferredSize(new Dimension(440, 160));

        clearChanges();
    }

    @Override
    protected void initializeGui()
    {
        GridBagConstraints cons;

        _table = new GenericTable(_model);
        _table.setRowSelectionAllowed(false);
        _table.setColumnSelectionAllowed(false);
        _table.setSelectionMode(ListSelectionModel.SINGLE_SELECTION);
        _table.setRowHeight(25);

        JScrollPane tablePanel = new JScrollPane(_table);

        JPanel textEditorPanel = new JPanel();
        textEditorPanel.setLayout(new GridBagLayout());
        cons = new GridBagConstraints();
        cons.gridy = 0;
        textEditorPanel.add(new JLabel("jdbc:postgresql:"), cons);
        cons = new GridBagConstraints();
        cons.gridy = 0;
        cons.weightx = 1;
        cons.fill = GridBagConstraints.HORIZONTAL;
        cons.anchor = GridBagConstraints.WEST;
        textEditorPanel.add(_connectionStringField, cons);

        JPanel retestPanel = new JPanel(new FlowLayout(FlowLayout.CENTER));
        retestPanel.add(new JButton(_retestAction));

        this.setLayout(new GridBagLayout());
        cons = new GridBagConstraints();
        cons.gridx = 0;
        cons.weightx = 1;
        cons.fill = GridBagConstraints.HORIZONTAL;
        cons.insets = new Insets(5, 5, 5, 5);
        cons.anchor = GridBagConstraints.WEST;
        cons.weighty = 0;
        this.add(new JLabel("Specify connection parameters:"), cons);
        cons.anchor = GridBagConstraints.CENTER;
        cons.ipady = 9;
        cons.weighty = 1;
        cons.fill = GridBagConstraints.BOTH;
        this.add(tablePanel, cons);
        cons.fill = GridBagConstraints.HORIZONTAL;
        cons.weighty = 0;
        cons.ipady = 0;
        this.add(textEditorPanel, cons);
        cons.fill = GridBagConstraints.NONE;
        this.add(retestPanel, cons);
//        inputPanel.setBorder(HSwingFactory.createTitledBorder(BorderFactory.createEtchedBorder(),
//                                                              "Specify Working Connection",
//                                                              null));
//
//        JPanel southPanel = new JPanel(new FlowLayout(FlowLayout.CENTER));
//        southPanel.add(new JButton(_okAction));
//        southPanel.add(new JButton(_cancelAction));
//        this.setLayout(new BorderLayout());
//        this.add(inputPanel, BorderLayout.CENTER);
//        this.add(southPanel, BorderLayout.SOUTH);
//        this.setBorder(HSwingFactory.createTitledBorder(BorderFactory.createEtchedBorder(1),
//                                                        "Edit Database Connection",
//                                                        null));
        updateDisplay();
    }

    @Override
    public String getTitle()
    {
        return "Edit Database Connection";
    }

    private JTextField makeConnectionStringField()
    {
        JTextField field = new JTextField();
        field.addActionListener(new ActionListener()
        {
            @Override
            public void actionPerformed(ActionEvent e)
            {
                _connection = DatabaseConnectionSpecification.parse("jdbc:postgresql:"
                    + _connectionStringField.getText());
                _model.fireTableDataChanged();
                _table.setVisible(false);
                _table.setVisible(true);
            }
        });
        return field;
    }

    @Override
    @Subscribe
    public void reactToConnectionTested(ConnectionTestedNotice evt)
    {
        _model.fireTableCellUpdated(0, 4);
    }

    @Override
    public void windowClosed(WindowEvent evt)
    {
        clearChanges();
    }

    private void updateString()
    {
        _connectionStringField.setText(_connection.toString().substring(16));
    }

    private void updateDisplay()
    {
        updateString();
        _model.fireTableDataChanged();
        this.repaint();
    }

    public boolean isConnected()
    {
        try
        {
            return _tester.canConnect(_options.getDefaultConnection());
        }
        catch(InterruptedException e)
        {
            return false;
        }
    }

    /**
     * Resets to the active connection.
     */
    public void clearChanges()
    {
        _connection = _options.getDefaultConnection();
        updateDisplay();
    }

    public static void openConnectionDialogAndProcessResults(MEFPParameterEstimatorRunInfo runInfo,
                                                             Component parentComponent) throws Exception
    {
        ConnectionEditorPanel panel = new ConnectionEditorPanel(runInfo);
        int option = JOptionPane.showConfirmDialog(parentComponent,
                                                   panel,
                                                   "Edit Database Connection",
                                                   JOptionPane.OK_CANCEL_OPTION,
                                                   JOptionPane.QUESTION_MESSAGE);
        if(option == JOptionPane.OK_OPTION)
        {
            panel.processOK();
        }
        else
        {
            throw new Exception("RFC Forecast file preparation was canceled.");
        }
    }

    public static Action createModalAction(final MEFPParameterEstimatorRunInfo runInfo, final Component parentComponent)
    {
        Action action = new ConnectionAction()
        {
            @Override
            public void actionPerformed(ActionEvent e)
            {
                try
                {
                    openConnectionDialogAndProcessResults(runInfo, parentComponent);
                }
                catch(Exception exc)
                {
                    //For a modal action, the button must have been clicked.  If a cancellation occurs, which is
                    //the only way an exception should happen, ignore it.
                }
            }
        };
        runInfo.register(action);
        runInfo.getRfcDataOptions().resendDefaultConnectionTest();
        return action;
    }

    private class ConnectionTableModel extends AbstractTableModel implements ToolTipTableModel
    {
        @Override
        public int getColumnCount()
        {
            return 5;
        }

        @Override
        public int getRowCount()
        {
            if(_connection != null)
            {
                return 1;
            }
            else
            {
                return 0;
            }
        }

        @Override
        public Class<?> getColumnClass(int col)
        {
            if(col == 4)
            {
                return StatusLabel.class;
            }
            return String.class;
        }

        @Override
        public String getColumnName(int col)
        {
            if(col == 0)
            {
                return "Host";
            }
            else if(col == 1)
            {
                return "Port";
            }
            else if(col == 2)
            {
                return "Database";
            }
            else if(col == 3)
            {
                return "User";
            }
            else if(col == 4)
            {
                return "Found?";
            }
            return null;
        }

        @Override
        public Object getValueAt(int row, int col)
        {
            if(col == 0)
            {
                return Strings.nullToEmpty(_connection.getHost());
            }
            else if(col == 1)
            {
                return Strings.nullToEmpty(_connection.getPort());
            }
            else if(col == 2)
            {
                return Strings.nullToEmpty(_connection.getDatabase());
            }
            else if(col == 3)
            {
                return Strings.nullToEmpty(_connection.getUser());
            }
            else if(col == 4)
            {
                return _tester.isConnected(_connection);
            }
            return null;
        }

        @Override
        public void setValueAt(Object value, int row, int col)
        {
            if(!(value instanceof String))
            {
                return;
            }
            String val = (String)value;

            if(col == 0)
            {
                _connection = _connection.withHost(val);
            }
            else if(col == 1)
            {
                _connection = _connection.withPort(val);
            }
            else if(col == 2)
            {
                _connection = _connection.withDatabase(val);
            }
            else if(col == 3)
            {
                _connection = _connection.withUser(val);
            }

            fireTableCellUpdated(row, col);
            updateString();
        }

        @Override
        public boolean isCellEditable(int row, int col)
        {
            return col < 4;
        }

        @Override
        public String getColumnHeaderToolTip(int col)
        {
            if(col == 0)
            {
                return "The host you are connecting to. If blank connects to this computer.";
            }
            else if(col == 1)
            {
                return "The port you are connecting to. Leave blank to use default value.";
            }
            else if(col == 2)
            {
                return "The name of the database you are connecting to.";
            }
            else if(col == 3)
            {
                return "The user name to connect to the database with. May be blank.";
            }
            else if(col == 4)
            {
                return "If the specified database was found.";
            }
            return null;
        }

        @Override
        public String getCellToolTip(int rowIndex, int visibleColIndex)
        {
            if(visibleColIndex == 4)
            {
                return ((StatusIndicator)getValueAt(rowIndex, visibleColIndex)).getDescription();
            }
            return null;
        }
    }

}
