package ohd.hseb.hefs.mefp.sources.rfcfcst.database;

import java.sql.Connection;
import java.sql.DriverManager;
import java.sql.SQLException;
import java.util.Collections;
import java.util.Map;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import net.jcip.annotations.Immutable;
import ohd.hseb.hefs.utils.xml.CompositeXMLWriter;
import ohd.hseb.hefs.utils.xml.Properties;
import ohd.hseb.hefs.utils.xml.XMLWritable;
import ohd.hseb.hefs.utils.xml.XMLWriter;
import ohd.hseb.hefs.utils.xml.vars.XMLString;

import com.google.common.base.Objects;
import com.google.common.base.Splitter;
import com.google.common.base.Strings;

@Immutable
public class DatabaseConnectionSpecification implements Cloneable, XMLWritable
{
    public static final String XML_TAG = "databaseConnectionString";
    public static final String HOST_TAG = "host";
    public static final String PORT_TAG = "port";
    public static final String DB_TAG = "database";
    public static final String PROP_TAG = "properties";

    public static final String USER_PROPERTY = "user";

    private final String _host;
    private final String _port;
    private final String _database;
    private final Properties _properties;

    public DatabaseConnectionSpecification()
    {
        this(null, null, null);
    }

    public DatabaseConnectionSpecification(String database)
    {
        this(null, null, database);
    }

    public DatabaseConnectionSpecification(String host, String database)
    {
        this(host, null, database);
    }

    public DatabaseConnectionSpecification(String host, String port, String database)
    {
        _host = host;
        _port = port;
        _database = database;
        _properties = new Properties(PROP_TAG);
        _properties.setProperty(USER_PROPERTY, "pguser");
    }

    public DatabaseConnectionSpecification(String host, String port, String database, String user)
    {
        this(host, port, database);
        _properties.setProperty(USER_PROPERTY, user);
    }

    private DatabaseConnectionSpecification(String host, String port, String database, Properties properties)
    {
        _host = host;
        _port = port;
        _database = database;
        _properties = properties;
    }

    /**
     * Creates a connection specification from the string format.
     * 
     * @param connection
     * @return
     */
    public static DatabaseConnectionSpecification parse(String connection)
    {
        DatabaseConnectionSpecification spec;

        String regex = "jdbc:postgresql:(?://([^/]*?)(?::([^/]*))?/)?([^?]*)(?:\\?(.*))?";
        Matcher matcher = Pattern.compile(regex).matcher(connection);
        if(matcher.matches())
        {
            spec = new DatabaseConnectionSpecification(matcher.group(1), matcher.group(2), matcher.group(3));
            Map<String, String> props;
            if(matcher.group(4) != null)
            {
                props = Splitter.on("&").withKeyValueSeparator("=").split(matcher.group(4));
            }
            else
            {
                props = Collections.emptyMap();
            }
            spec = spec.withProperties(new Properties(PROP_TAG, props));
        }
        else
        {
            throw new IllegalArgumentException("Cannot parse connection string \"" + connection + "\".");
        }

        return spec;
    }

    /**
     * Gets the connection string, and ensures that it is valid.
     * 
     * @return this connection as a string
     * @throws IllegalStateException if a valid connection string cannot be formed
     */
    public String getConnectionString() throws IllegalStateException
    {
        assertValid();
        return this.toString();
    }

    /**
     * Ensures this connection string is valid
     * 
     * @throws IllegalStateException if this connection is invalid
     */
    private void assertValid() throws IllegalStateException
    {
        if(!isValid())
        {
            throw new IllegalStateException("Cannot form a valid connection string.");
        }
    }

    /**
     * Tests if a legal connection string can be formed from this object's current state.
     */
    private boolean isValid()
    {
        if(_database == null || _database.isEmpty())
        {
            return false;
        }

        if((_host == null || _host.isEmpty()) && (_port != null && !_port.isEmpty()))
        {
            return false;
        }

        return true;
    }

    /**
     * Returns this as a connection string.
     */
    @Override
    public String toString()
    {
        String str = "jdbc:postgresql:";

        if(_host != null && !_host.equals(""))
        {
            str += "//" + _host;
            if(_port != null && !_port.equals(""))
            {
                str += ":" + _port;
            }
            str += "/";
        }

        str += _database;

        if(!_properties.isEmpty())
        {
            String propStr = "";
            for(Map.Entry<Object, Object> entry: _properties.entrySet())
            {
                if(propStr != "")
                {
                    propStr += "&";
                }
                propStr += entry.getKey() + "=" + entry.getValue();
            }

            str += "?" + propStr;
        }

        return str;
    }

    /**
     * Picks the driver class for this connection.
     * 
     * @return the name of the driver class for this connection
     * @throws IllegalStateException if this connection is invalid
     */
    public String pickDriver() throws IllegalStateException
    {
        assertValid();
        return "org.postgresql.Driver";
    }

    /**
     * Creates the connections specified by this object
     * 
     * @return the new connection
     * @throws SQLException if the connection failed
     * @throws IllegalStateException if this connection specification is invalid
     */
    public Connection create() throws SQLException, IllegalStateException
    {
        assertValid();

        // Load the driver
        try
        {
            Class.forName(pickDriver());
        }
        catch(ClassNotFoundException e)
        {
            throw new SQLException("Could not load driver.", e);
        }

        return DriverManager.getConnection(this.toString());
    }

    @Override
    public boolean equals(Object other)
    {
        if(this == other)
        {
            return true;
        }

        if(!(other instanceof DatabaseConnectionSpecification))
        {
            return false;
        }
        DatabaseConnectionSpecification that = (DatabaseConnectionSpecification)other;
        return Objects.equal(this._host, that._host) && Objects.equal(this._port, that._port)
            && Objects.equal(this._database, that._database) && this._properties.equals(that._properties);
    }

    @Override
    public int hashCode()
    {
        return toString().hashCode();
    }

    @Override
    public XMLWriter getWriter()
    {
        return new CompositeXMLWriter(getXMLTagName(),
                                      new XMLString(HOST_TAG, _host),
                                      new XMLString(PORT_TAG, _port),
                                      new XMLString(DB_TAG, _database),
                                      _properties);
    }

    public String getXMLTagName()
    {
        return XML_TAG;
    }

    public String getHost()
    {
        return _host;
    }

    public String getPort()
    {
        return _port;
    }

    public String getDatabase()
    {
        return _database;
    }

    public Properties getProperties()
    {
        return _properties.clone();
    }

    public String getUser()
    {
        return _properties.getProperty(USER_PROPERTY);
    }

    public DatabaseConnectionSpecification withHost(String host)
    {
        return new DatabaseConnectionSpecification(host, _port, _database, _properties.clone());
    }

    public DatabaseConnectionSpecification withPort(String port)
    {
        return new DatabaseConnectionSpecification(_host, port, _database, _properties.clone());
    }

    public DatabaseConnectionSpecification withDatabase(String database)
    {
        return new DatabaseConnectionSpecification(_host, _port, database, _properties.clone());
    }

    public DatabaseConnectionSpecification withProperty(String key, String value)
    {
        Properties properties = _properties.clone();
        if(Strings.isNullOrEmpty(value))
        {
            properties.remove(key);
        }
        else
        {
            properties.setProperty(key, value);
        }
        return new DatabaseConnectionSpecification(_host, _port, _database, properties);

    }

    public DatabaseConnectionSpecification withProperties(Properties properties)
    {
        DatabaseConnectionSpecification connection = this;
        for(Map.Entry<Object, Object> entry: properties.entrySet())
        {
            connection = connection.withProperty((String)entry.getKey(), (String)entry.getValue());
        }
        return connection;
    }

    public DatabaseConnectionSpecification withUser(String user)
    {
        return this.withProperty(USER_PROPERTY, user);
    }
}
