package ohd.hseb.hefs.mefp.sources.rfcfcst.database;

import java.util.Map;
import java.util.concurrent.Callable;
import java.util.concurrent.ExecutionException;

import ohd.hseb.hefs.utils.notify.NotifierBase;
import ohd.hseb.hefs.utils.status.StatusIndicator;
import ohd.hseb.hefs.utils.status.StatusLabel;

import org.apache.logging.log4j.Logger;
import org.apache.logging.log4j.LogManager;

import com.google.common.base.Throwables;
import com.google.common.collect.Maps;
import com.google.common.util.concurrent.FutureCallback;
import com.google.common.util.concurrent.Futures;
import com.google.common.util.concurrent.ListenableFuture;
import com.google.common.util.concurrent.ListeningExecutorService;

/**
 * Tests database connections on the given executor. Produces {@link ConnectionTestedNotice}s whenever a connection
 * finishes testing.
 * 
 * @author alexander.garbarino
 */
public class DatabaseConnectionTester extends NotifierBase
{

    private static final Logger LOG = LogManager.getLogger(DatabaseConnectionTester.class);
    private final ListeningExecutorService _executor;
    private final Map<DatabaseConnectionSpecification, ListenableFuture<Boolean>> _connections;

    public DatabaseConnectionTester(final ListeningExecutorService executor)
    {
        _executor = executor;
        _connections = Maps.newHashMap();
    }

    /**
     * Makes a future to test the given database's ability to connect.
     * 
     * @param connection
     * @return
     */
    public ListenableFuture<Boolean> test(final DatabaseConnectionSpecification connection)
    {
        if(!_connections.containsKey(connection))
        {
            // Testing procedure - try to create connection.
            final Callable<Boolean> tester = new Callable<Boolean>()
            {
                @Override
                public Boolean call() throws Exception
                {
                    try
                    {
                        connection.create();
                        return true;
                    }
                    catch(final Exception e)
                    {
                        LOG.warn("Problem while testing connection with URL " + connection.toString() + ": "
                            + e.getMessage());
                        return false;
                    }
                }
            };

            // Listener for test complete - post a tested event.
            final FutureCallback<Boolean> callback = new FutureCallback<Boolean>()
            {
                @Override
                public void onSuccess(final Boolean result)
                {
                    post(new ConnectionTestedNotice(DatabaseConnectionTester.this, connection, result));
                }

                @Override
                public void onFailure(final Throwable t)
                {
                    onSuccess(false);
                }
            };

            final ListenableFuture<Boolean> result = _executor.submit(tester);
            Futures.addCallback(result, callback, _executor);

            _connections.put(connection, result);
            return result;
        }
        else
        {
            return _connections.get(connection);
        }
    }

    /**
     * Forces all connections to retest.
     */
    public void clearTests()
    {
        _connections.clear();
    }

    /**
     * Test the given connection. Blocks until completion.
     * 
     * @param connection the connection to test
     * @return if the connection can be made
     */
    public boolean canConnect(final DatabaseConnectionSpecification connection) throws InterruptedException
    {
        try
        {
            return test(connection).get();
        }
        catch(final ExecutionException e)
        {
            throw Throwables.propagate(e);
        }
    }

    public boolean isFinishedTesting(final DatabaseConnectionSpecification connection)
    {
        if(_connections.containsKey(connection))
        {
            return _connections.get(connection).isDone();
        }
        else
        {
            return false;
        }
    }

    public void resendConnectionEvent(final DatabaseConnectionSpecification connection)
    {
        if(isFinishedTesting(connection))
        {
            boolean canConnect = false;
            try
            {
                canConnect = canConnect(connection);
            }
            catch(final InterruptedException e)
            {
                // Should never happen - tested with isFinishedTesting.
                e.printStackTrace();
            }
            post(new ConnectionTestedNotice(DatabaseConnectionTester.this, connection, canConnect));
        }
    }

    /**
     * Gets the current status of testing the connection
     * 
     * @param connection
     * @return a status indicator saying we can connect, we can't, or that the test is still in progress.
     */
    public StatusIndicator isConnected(final DatabaseConnectionSpecification connection)
    {
        final ListenableFuture<Boolean> future = test(connection);
        if(future.isDone())
        {
            try
            {
                if(future.get())
                {
                    return StatusLabel.make(true, "Successfully connected.");
                }
                else
                {
                    return StatusLabel.make(false, "Could not connect.");
                }
            }
            catch(final Exception e)
            {
                e.printStackTrace();
                return StatusLabel.make(false);
            }
        }
        else
        {
            return StatusLabel.make((Boolean)null, "Testing still in progress.");
        }
    }
}
