package ohd.hseb.hefs.mefp.sources.rfcfcst.database;

import static com.google.common.collect.Lists.newArrayList;

import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Timestamp;
import java.util.Calendar;
import java.util.Collection;
import java.util.List;
import java.util.TimeZone;

import ohd.hseb.hefs.mefp.sources.rfcfcst.VfyPair;
import ohd.hseb.hefs.mefp.sources.rfcfcst.VfyPairPedtsepCode;
import ohd.hseb.hefs.mefp.sources.rfcfcst.VfyPairSet;
import ohd.hseb.hefs.pe.tools.TimeSeriesSorter;

import org.apache.logging.log4j.Logger;
import org.apache.logging.log4j.LogManager;

public class VfyPairsTable extends DbTable
{
    private static final Logger LOG = LogManager.getLogger(VfyPairsTable.class);

    public VfyPairsTable(final Database database)
    {
        super(database, "vfypairs");
    }

    /**
     * Generate pairs from the sorter and insert them into the table.
     * 
     * @param sorter the sorter to add to the table
     * @throws SQLException
     */
    public void insert(final TimeSeriesSorter sorter) throws SQLException
    {
        for(final VfyPair pair: new VfyPairSet(sorter))
        {
            insert(pair);
        }
    }

    /**
     * Adds all pairs to the table
     * 
     * @param pairColl all pairs to add
     * @throws SQLException
     */
    public void insert(final Collection<VfyPair> pairColl) throws SQLException
    {
        for(final VfyPair pair: pairColl)
        {
            insert(pair);
        }
    }

    /**
     * Adds the given pair to the table.
     * 
     * @param pair the pair to add
     * @return the number of records updated
     */
    public int insert(final VfyPair pair) throws SQLException
    {
        final VfyPairPedtsepCode code = pair.getPedtsepCode();

        final String s = " INSERT INTO " + getTableName() + " VALUES (?,?,?,?,?,?,?,?,?,?,?,?,?,?,"
            + "ROUND(CAST (? AS numeric),2),ROUND(CAST (? AS numeric),2),?)";
        final PreparedStatement ps = getConnection().prepareStatement(s);

        final Calendar cal = Calendar.getInstance(TimeZone.getTimeZone("GMT"));

        ps.setString(1, pair.getLocationId());
        ps.setString(2, code.getPE().substring(0, 1));
        ps.setString(3, code.getPE().substring(1, 2));
        ps.setString(4, code.getD());
        ps.setShort(5, (short)code.getDurationInHours());
        ps.setString(6, code.getForecastTS().substring(0, 1));
        ps.setString(7, code.getForecastTS().substring(1, 2));
        ps.setString(8, code.getE());
        ps.setString(9, code.getP());
        ps.setTimestamp(10, new Timestamp(pair.getValidTime()), cal);
        ps.setTimestamp(11, new Timestamp(pair.getBasisTime()), cal);
        ps.setString(12, code.getObservedTS().substring(0, 1));
        ps.setString(13, code.getObservedTS().substring(1, 2));
        ps.setTimestamp(14, new Timestamp(pair.getObservedTime()), cal);
        ps.setFloat(15, (float)pair.getForecastValue());
        ps.setFloat(16, (float)pair.getObservedValue());
        ps.setInt(17, pair.getQualityCode());

        return ps.executeUpdate();
    }

    /**
     * Selects the pairs with the given where string.
     * 
     * @param where the string starting with "WHERE" specifying what pairs to grab
     * @return a collection of all the specified pairs
     * @throws SQLException
     */
    public Collection<VfyPair> selectPairs(final String where) throws SQLException
    {
        LOG.debug("Selecting from vfypairs table with where clause " + where + "...");
        final List<VfyPair> list = newArrayList();
        final Calendar cal = Calendar.getInstance(TimeZone.getTimeZone("GMT"));
        final ResultSet rs = super.select(where);
        while(rs.next())
        {
            final VfyPairPedtsepCode code = new VfyPairPedtsepCode(rs.getString(2) + rs.getString(3),
                                                                   rs.getString(4),
                                                                   rs.getString(12) + rs.getString(13),
                                                                   rs.getString(6) + rs.getString(7),
                                                                   rs.getString(8),
                                                                   rs.getString(9));

            final VfyPair pair = new VfyPair(rs.getString(1), code, rs.getTimestamp(10, cal).getTime(), // valid time
                                             rs.getTimestamp(11, cal).getTime(), // basis time
                                             rs.getTimestamp(14, cal).getTime(), // observed time
                                             rs.getDouble(15), // forecast value
                                             rs.getDouble(16), // observed value
                                             rs.getInt(17)); // quality code
            list.add(pair);
        }
        LOG.debug("Found " + list.size() + " forecast-observed pairs.");
        return list;
    }

    /**
     * Selects the pairs from the given location and SHEF code. Any empty values in the code are ignored.
     * 
     * @param location location to restrict pairs to
     * @param code code to restrict pairs to
     * @return all pairs in the given location and code
     */
    public Collection<VfyPair> selectPairs(final String location, final VfyPairPedtsepCode code) throws SQLException
    {
        return selectPairs(generateWhereClause(location, code));
    }

    /**
     * Selects the pairs matching the given filter.
     * 
     * @param filter the filter specifying which pairs to retrieve
     * @return all pairs specified by the filter
     * @throws SQLException if an underlying sql error occurs
     */
    public Collection<VfyPair> selectPairs(final VfyPairFilter filter) throws SQLException
    {
        final List<VfyPair> pairs = newArrayList();
        for(final VfyPairPedtsepCode code: filter.getCodes())
        {
            pairs.addAll(selectPairs(filter.getLocationId(), code));
        }
        return pairs;
    }

    /**
     * Create the where clause which will affect the given codes. Any argument or values in the pedtsep code can be
     * null, in which case they will not be restricted respective to that value.
     * 
     * @param location the location to restrict to
     * @param code the pedtsep code to restrict to
     * @return a where clause which will limit sql instructions to the given location and code
     */
    public static String generateWhereClause(final String location, final VfyPairPedtsepCode code)
    {
        final List<String> components = newArrayList();

        if(location != null)
        {
            components.add("lid = '" + location + "'");
        }

        if(code != null)
        {
            String subcode;

            subcode = code.getPE();
            if(subcode != null)
            {
                components.add("pe1 = '" + subcode.charAt(0) + "'");
                components.add("pe2 = '" + subcode.charAt(1) + "'");
            }

            subcode = code.getD();
            if(subcode != null)
            {
                components.add("dur = '" + subcode + "'");
            }

            subcode = code.getForecastTS();
            if(subcode != null)
            {
                components.add("fcst_t = '" + subcode.charAt(0) + "'");
                components.add("fcst_s = '" + subcode.charAt(1) + "'");
            }

            subcode = code.getE();
            if(subcode != null)
            {
                components.add("e = '" + subcode + "'");
            }

            subcode = code.getP();
            if(subcode != null)
            {
                components.add("p = '" + subcode + "'");
            }

            subcode = code.getObservedTS();
            if(subcode != null)
            {
                components.add("obs_t = '" + subcode.charAt(0) + "'");
                components.add("obs_s = '" + subcode.charAt(1) + "'");
            }

        }

        if(components.isEmpty())
        {
            return "";
        }
        else
        {
            String clause = "WHERE " + components.get(0);
            components.remove(0);
            for(final String component: components)
            {
                clause += " AND " + component;
            }

            return clause;
        }
    }

}