package ohd.hseb.hefs.mefp.tools;

import static com.google.common.base.Preconditions.checkArgument;

import java.text.DecimalFormat;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.EnumSet;
import java.util.List;
import java.util.TimeZone;

import ohd.hseb.hefs.mefp.models.MEFPEnsembleGeneratorModel;
import ohd.hseb.hefs.mefp.models.MEFPParameterEstimationModel;
import ohd.hseb.hefs.mefp.models.parameters.MEFPFullModelParameters;
import ohd.hseb.hefs.mefp.models.parameters.MEFPSourceModelParameters;
import ohd.hseb.hefs.mefp.models.precipitation.PrecipitationEnsembleGenerationModel;
import ohd.hseb.hefs.mefp.models.precipitation.PrecipitationParameterEstimationModel;
import ohd.hseb.hefs.mefp.models.temperature.TemperatureEnsembleGenerationModel;
import ohd.hseb.hefs.mefp.models.temperature.TemperatureParameterEstimationModel;
import ohd.hseb.hefs.mefp.pe.estimation.MEFPEstimationControlOptions;
import ohd.hseb.hefs.mefp.pe.estimation.MEFPPrecipitationEstimationControlOptions;
import ohd.hseb.hefs.mefp.pe.estimation.MEFPTemperatureEstimationControlOptions;
import ohd.hseb.hefs.mefp.sources.MEFPForecastSource;
import ohd.hseb.hefs.mefp.sources.MEFPSourceDataHandler;
import ohd.hseb.hefs.mefp.sources.cfsv2.CFSv2ForecastSource;
import ohd.hseb.hefs.mefp.sources.gefs.GEFSForecastSource;
import ohd.hseb.hefs.mefp.sources.historical.HistoricalForecastSource;
import ohd.hseb.hefs.mefp.sources.rfcfcst.RFCForecastSource;
import ohd.hseb.hefs.mefp.tools.canonical.CanonicalEvent;
import ohd.hseb.hefs.pe.model.ModelParameterType;
import ohd.hseb.hefs.pe.model.ParameterEstimationModel;
import ohd.hseb.hefs.pe.tools.HEFSTools;
import ohd.hseb.hefs.pe.tools.LocationAndDataTypeIdentifier;
import ohd.hseb.hefs.utils.tools.ParameterId;
import ohd.hseb.hefs.utils.tools.ParameterId.Type;
import ohd.hseb.util.misc.HCalendar;

import org.apache.commons.lang.ArrayUtils;

import com.google.common.collect.Lists;

public abstract class MEFPTools
{
    public static final String noObserved = "no observed data";

    /**
     * List of old source parameter file name prefixes. See {@link MEFPSourceModelParameters} for information on how the
     * prefix is determined, but is generally the short prefix associated with the source plus 'SourceModelParameters'.
     */
    public static final String[] OLD_SOURCE_PARAMETER_FILE_PREFIX = {"GFSSourceModelParameters"};

    /**
     * @param type An instance of {@link Type}.
     * @return An instance of {@link MEFPParameterEstimationModel} after verifying the type is valid.
     */
    public static MEFPParameterEstimationModel determineParameterEstimationModel(final ParameterId.Type type)
    {
        checkArgument(getSupportedDataTypes().contains(type), "%s is not supported.", type);
        switch(type)
        {
            case TEMPERATURE:
                return new TemperatureParameterEstimationModel();
            case PRECIPITATION:
                return new PrecipitationParameterEstimationModel();
            default:
                throw new RuntimeException("INTERNAL ERROR: Should never happen -- got here with an unrecognized data type!");
        }
    }

    /**
     * @return {@link EnumSet} of supported {@link Type}s.
     */
    public static EnumSet<Type> getSupportedDataTypes()
    {
        return EnumSet.of(Type.PRECIPITATION, Type.TEMPERATURE);
    }

    /**
     * @param type An instance of {@link Type}.
     * @param sources {@link Iterable} collection of {@link MEFPSourceDataHandler} instances.
     * @return
     */
    public static MEFPEstimationControlOptions constructEstimationControlOptions(final ParameterId.Type type,
                                                                                 final Iterable<? extends MEFPForecastSource> sources)
    {
        final ParameterEstimationModel model = determineParameterEstimationModel(type);
        switch(type)
        {
            case TEMPERATURE:
                return new MEFPTemperatureEstimationControlOptions((TemperatureParameterEstimationModel)model, sources);
            case PRECIPITATION:
                return new MEFPPrecipitationEstimationControlOptions((PrecipitationParameterEstimationModel)model,
                                                                     sources);
            default:
                throw new RuntimeException("INTERNAL ERROR: Should never happen -- got here with an unrecognized data type!");
        }
    }

    /**
     * @param identifier
     * @return An instance of the {@link MEFPEnsembleGeneratorModel} model to use.
     */
    public static MEFPEnsembleGeneratorModel determineEnsembleGenerationModel(final LocationAndDataTypeIdentifier identifier)
    {
        if(identifier.isTemperatureDataType())
        {
            return determineEnsembleGenerationModel(false);
        }
        else if(identifier.isPrecipitationDataType())
        {
            return determineEnsembleGenerationModel(true);
        }
        throw new IllegalArgumentException("Data type of identifier, " + identifier.buildStringToDisplayInTree()
            + ", does not have a corresponding MEFP ensemble generation model.");
    }

    /**
     * @param precipitation True for precip, false for temp data.
     * @return An instance of the {@link MEFPEnsembleGeneratorModel} model to use.
     */
    public static MEFPEnsembleGeneratorModel determineEnsembleGenerationModel(final boolean precipitation)
    {
        if(!precipitation)
        {
            return new TemperatureEnsembleGenerationModel();
        }
        return new PrecipitationEnsembleGenerationModel();
    }

    /**
     * Do not call this method to instantiate an individual source. Rather, the sources must be maintained in a list by
     * the relevant software and acquired from that list as needed.
     * 
     * @param prefix Can be RFC, GFS, CFSv2, Climatology or Historical. All map directory to a forecast source by the
     *            name [prefix]ForecastSource, exception Climatology which maps to Historical. Note that whenever a
     *            forecast source is added, this method must be changed to allow for its return.
     * @return An instance of {@link MEFPForecastSource}.
     */
    @Deprecated
    public static MEFPForecastSource instantiateSourceByClassSimpleNamePrefix(final String prefix)
    {
        if(prefix.equalsIgnoreCase("RFC"))
        {
            return new RFCForecastSource();
        }
        if(prefix.equalsIgnoreCase("GEFS"))
        {
            return new GEFSForecastSource();
        }
        if(prefix.equalsIgnoreCase("CFSv2"))
        {
            return new CFSv2ForecastSource();
        }
        if(prefix.equalsIgnoreCase("Climatology") || prefix.equals("Historical"))
        {
            return new HistoricalForecastSource();
        }
        throw new IllegalArgumentException("Source name " + prefix + " is not valid.");
    }

    /**
     * This method must be kept up-to-date with the list of forecast sources!!!
     * 
     * @return {@link List} of {@link MEFPForecastSource}s that MEFP handles, in the order that it should be for
     *         parameters estimation and ensemble generation.
     */
    public static List<MEFPForecastSource> instantiateDefaultMEFPForecastSources()
    {
        final List<MEFPForecastSource> sources = new ArrayList<MEFPForecastSource>();
        sources.add(new RFCForecastSource());
        sources.add(new GEFSForecastSource());
        sources.add(new CFSv2ForecastSource());
        sources.add(new HistoricalForecastSource());
        return sources;
    }

    /**
     * @return A {@link MEFPFullModelParameters} ready for reading.
     */
    public static MEFPFullModelParameters instantiateMEFPFullModelParametersForReading()
    {
        return new MEFPFullModelParameters(instantiateDefaultMEFPForecastSources());
    }

    /**
     * @return True if the provided file name prefix is contained within {@link #OLD_SOURCE_PARAMETER_FILE_PREFIX}.
     */
    public static boolean isOldSourceParameterFilePrefix(final String prefix)
    {
        return ArrayUtils.contains(MEFPTools.OLD_SOURCE_PARAMETER_FILE_PREFIX, prefix);
    }

    /**
     * @param locationId
     * @param precipitation True for precip, false for temp.
     * @return The name of the parameter file.
     */
    public static String determineParameterFileName(final String locationId, final boolean precipitation)
    {
        return locationId + "." + HEFSTools.determineDataTypeString(precipitation) + ".mefp.parameters.tgz";
    }

    /**
     * Calls {@link #determineParameterFileName(String, boolean)} based on the provided identifier.
     * 
     * @param identifier Location and data type for which to determine file name.
     * @return The name of the parameter file.
     */
    public static String determineParameterFileName(final LocationAndDataTypeIdentifier identifier)
    {
        return determineParameterFileName(identifier.getLocationId(), identifier.isPrecipitationDataType());
    }

    /**
     * @param identifier Location and data type for which to determine file name.
     * @return The name of the processed historical data binary file.
     */
    public static String determineProcessedHistoricalDataFileName(final LocationAndDataTypeIdentifier identifier)
    {
        return identifier.getLocationId() + "."
            + HEFSTools.determineDataTypeString(identifier.isPrecipitationDataType()) + ".historical.bin.gz";
    }

    /**
     * @return A list of the {@link MEFPForecastSource} with estimate parameters contained with provided full model
     *         parameters.
     */
    public static List<MEFPForecastSource> generateListOfSourcesWithParameters(final MEFPFullModelParameters fullParameters)
    {
        final List<MEFPForecastSource> sources = Lists.newArrayList();
        for(final MEFPForecastSource source: fullParameters.getOrderedForecastSources())
        {
            if(fullParameters.getSourceModelParameters(source).wereParametersEstimatedForSource())
            {
                sources.add(source);
            }
        }
        return sources;
    }

    /**
     * @return A {@link List} of the {@link MEFPForecastSource} instances for which parameters were computed AND the
     *         requested type was computed. This calls
     *         {@link #generateListOfSourcesWithParameters(MEFPFullModelParameters)} to get the base list and then
     *         searches the list removing sources for which the provided type was not calculated.
     */
    public static List<MEFPForecastSource> generateListOfSourcesWithParameterOfType(final MEFPFullModelParameters fullParameters,
                                                                                    final ModelParameterType type)
    {
        final List<MEFPForecastSource> sources = generateListOfSourcesWithParameters(fullParameters);

        for(final MEFPForecastSource source: MEFPTools.generateListOfSourcesWithParameters(fullParameters))
        {
            if(!fullParameters.getSourceModelParameters(source).getModelParameterTypesWithStoredValues().contains(type))
            {
                sources.remove(source);
            }
        }

        return sources;
    }

    /**
     * Static method returns a {@link String} that corresponds to the day-of-year. This will building a {@link Calendar}
     * each time it is called, so only call this as needed.
     * 
     * @param dayOfYear 1 through 366.
     */
    public static String getDayOfYearStr(final int dayOfYear)
    {
        if(dayOfYear < 0)
        {
            return "All Days of the Year";
        }
        final Calendar cal = Calendar.getInstance(TimeZone.getTimeZone("GMT"));
        cal.set(Calendar.DAY_OF_YEAR, dayOfYear);
        return HCalendar.buildDateStr(cal, "MMM dd") + " (day " + dayOfYear + ")";
    }

    /**
     * @param evt The event for which to create the string.
     * @param isPrecipitationDataType True for precipitation, false for temperature.
     * @param endPeriodFlag True for the end period of the event, false for the start.
     * @param includeUnits True to include units (hrs or dys) in the string, false to only include the number.
     * @param formatter {@link DecimalFormat} to use to format the numbers.
     * @return A {@link String} displaying the period as dictated by the arguments.
     */
    public static String getCanonicalEventPeriodStr(final CanonicalEvent evt,
                                                    final boolean isPrecipitationDataType,
                                                    final boolean endPeriodFlag,
                                                    final boolean includeUnits,
                                                    final DecimalFormat formatter)
    {
        final int eventTimeStep = CanonicalEvent.determineCanonicalEventPeriodUnitInHours(isPrecipitationDataType);
        final int pdHours;
        if(endPeriodFlag)
        {
            pdHours = eventTimeStep * evt.getEndLeadPeriod();
        }
        else
        {
            pdHours = eventTimeStep * (evt.getStartLeadPeriod() - 1);
        }
        final float pdDays = (float)(pdHours / 24d);

        String results = Integer.toString(pdHours);
        if(includeUnits)
        {
            results += " hrs";
        }
        if(evt.getDuration() * eventTimeStep >= 24)
        {
            results = formatter.format(pdDays);
            if(includeUnits)
            {
                results += " dys";
            }
        }
        return results;
    }
}
