package ohd.hseb.hefs.mefp.tools;

import java.io.File;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import nl.wldelft.util.Period;
import nl.wldelft.util.timeseries.TimeSeriesArray;
import ohd.hseb.hefs.pe.tools.HEFSTools;
import ohd.hseb.hefs.utils.xml.CollectionXMLReader;
import ohd.hseb.hefs.utils.xml.XMLReaderFactory;
import ohd.hseb.hefs.utils.xml.XMLTools;
import ohd.hseb.hefs.utils.xml.vars.XMLLong;
import ohd.hseb.hefs.utils.xml.vars.XMLString;
import ohd.hseb.util.misc.HCalendar;

import com.google.common.collect.Lists;
import com.google.common.collect.Maps;

/**
 * A map with added functionality for managing/navigating questioanble messages. The map is a parameter id (String) to
 * forecast time (Long) to time series time (Long) to a message explaining why its questionable (String).
 * 
 * @author hankherr
 */

@SuppressWarnings("serial")
public class QuestionableMessageMap extends HashMap<String, HashMap<Long, HashMap<Long, String>>>
{

    public QuestionableMessageMap(final File questionableFile) throws Exception
    {
        loadQuestionableFile(questionableFile);
    }

    public QuestionableMessageMap()
    {
    }

    /**
     * Puts the message into this, making sure to create any necessary maps in the middle. Call this instead of calling
     * the various {@link Map} puts to avoid needing to think about the middle-man maps.
     * 
     * @param parameterId
     * @param forecastTime
     * @param time
     * @param message
     */
    public void addQuestionableMessage(final String parameterId,
                                       Long forecastTime,
                                       final Long time,
                                       final String message)
    {
        if(HEFSTools.isObservedDataType(parameterId))
        {
            forecastTime = QuestionableTools.NO_TIME;
        }

        HashMap<Long, HashMap<Long, String>> newForecastTimeMap = get(parameterId);
        if(newForecastTimeMap == null)
        {
            newForecastTimeMap = Maps.newHashMap();
            put(parameterId, newForecastTimeMap);
        }

        HashMap<Long, String> newTimeMap = newForecastTimeMap.get(forecastTime);
        if(newTimeMap == null)
        {
            newTimeMap = Maps.newHashMap();
            newForecastTimeMap.put(forecastTime, newTimeMap);
        }
        newTimeMap.put(time, message);
    }

    /**
     * @param ts
     * @return True if any value is questionable in the time series, calling
     *         {@link #isAnyValueQuestionable(String, long)} for the time series parameter id and forecast time.
     */
    public Boolean isAnyValueQuestionable(final TimeSeriesArray ts)
    {
        try
        {
            return isAnyValueQuestionable(ts.getHeader().getParameterId(), ts.getHeader().getForecastTime());
        }
        catch(final NullPointerException e)
        {
            return null;
        }
    }

    /**
     * @param forecastTime
     * @param timeSeriesTime
     * @return True if there are any questionable entries for the parameter and forecast time.
     */
    public boolean isAnyValueQuestionable(final long forecastTime, final long timeSeriesTime)
    {
        for(final String parameterId: keySet())
        {
            if(isQuestionable(parameterId, forecastTime, timeSeriesTime))
            {
                return true;
            }
        }
        return false;
    }

    /**
     * @param parameterId
     * @param forecastTime
     * @return True if there are any questionable entries for the parameter and forecast time.
     */
    public boolean isAnyValueQuestionable(final String parameterId, long forecastTime)
    {
        if(HEFSTools.isObservedDataType(parameterId))
        {
            forecastTime = QuestionableTools.NO_TIME;
        }
        return (get(parameterId) != null) && (get(parameterId).get(forecastTime) != null);
    }

    /**
     * @param timeSeries
     * @param timeSeriesTime
     * @return True if there are any questionable messages recorded for any of the provided time series at the specified
     *         time series time.
     */
    public boolean isAnyValueQuestionable(final Collection<TimeSeriesArray> timeSeries, final long timeSeriesTime)
    {
        for(final TimeSeriesArray ts: timeSeries)
        {
            long forecastTime = ts.getHeader().getForecastTime();

            if(HEFSTools.isObservedDataType(ts.getHeader().getParameterId()))
            {
                forecastTime = QuestionableTools.NO_TIME;
            }

            if(isQuestionable(ts.getHeader().getParameterId(), forecastTime, timeSeriesTime))
            {
                return true;
            }
        }
        return false;
    }

    /**
     * isAnyValueQuestionable() returns true if there are any questionable values in the HashMap for that time.
     * 
     * @param timeSeriesTime The time for which to check for questionable messages.
     * @return true if there are any questionable values in the map for that time.
     */
    public boolean isAnyValueQuestionable(final long timeSeriesTime)
    {
        for(final String parmId: keySet())
        {
            for(final Long forecastTime: get(parmId).keySet())
            {
                if(isQuestionable(parmId, forecastTime, timeSeriesTime))
                {
                    return true;
                }
            }
        }
        return false;
    }

    /**
     * @param forecastTimesToCheck
     * @return True if for any parameter that is a forecast parameter, calling
     *         {@link HEFSTools#isForecastDataType(String)}, a time series with any of the given forecast times includes
     *         questionable data. This makes use of the method {@link #isAnyValueQuestionable(String, long)} for check
     *         all applicable forecast times.
     */
    public boolean isAnyForecastValueQuestionable(final Collection<Long> forecastTimesToCheck)
    {
        for(final String parameterId: keySet())
        {
            if(HEFSTools.isForecastDataType(parameterId))
            {
                for(final long time: forecastTimesToCheck)
                {
                    if(isAnyValueQuestionable(parameterId, time))
                    {
                        return true;
                    }
                }
            }
        }
        return false;
    }

    /**
     * @param timePeriod
     * @return Looking only at the observed parameters in the map, calling {@link HEFSTools#isObservedDataType(String)},
     *         this check to see if any date in the provided {@link Period} is questioanble.
     */
    public boolean isAnyObservedValueQuestionable(final Period timePeriod)
    {
        for(final String parameterId: keySet())
        {
            if(HEFSTools.isObservedDataType(parameterId))
            {
                final HashMap<Long, String> tsTimeToMessageMap = get(parameterId).get(QuestionableTools.NO_TIME);
                if(tsTimeToMessageMap != null)
                {
                    for(final Long tsTime: tsTimeToMessageMap.keySet())
                    {
                        if(timePeriod.contains(tsTime))
                        {
                            return true;
                        }
                    }
                }
            }
        }
        return false;
    }

    /**
     * @param parameterId
     * @param forecastTime
     * @param tsTime
     * @return The questionable message for the given parameter, forecast time, and time series time. Return null if
     *         none.
     */
    public String getQuestionableMessage(final String parameterId, final long forecastTime, final long tsTime)
    {
        try
        {
            if(get(parameterId) == null)
            {
                return null;
            }
            if(get(parameterId).get(forecastTime) == null)
            {
                return null;
            }
            return get(parameterId).get(forecastTime).get(tsTime);
        }
        catch(final NullPointerException e)
        {
            return null;
        }
    }

    /**
     * @param parameterId
     * @param forecastTime
     * @param time
     * @return Checks the return of {@link #getQuestionableMessage(String, long, long)} for null.
     */
    public boolean isQuestionable(final String parameterId, final long forecastTime, final long time)
    {
        return getQuestionableMessage(parameterId, forecastTime, time) != null;
    }

    /**
     * @param forecastTime
     * @param time
     * @return Across all parameters, gets the messages called {@link #getQuestionableMessage(String, long, long)} and
     *         places them in an array. The returned {@link String}s includes the parameter id as a prefix to the
     *         message. Example: "TMAX: data is missing".
     */
    public List<String> getMessagesForAllParameters(final long forecastTime, final long time)
    {
        final List<String> returnStringList = new ArrayList<String>();

        for(final String key: keySet())
        {
            final String comment = getQuestionableMessage(key, forecastTime, time);
            if(comment != null)
            {
                returnStringList.add(key + ": " + comment);
            }
        }
        return returnStringList;
    }

    /**
     * @param timeSeriesTime
     * @return All messages across all parameters and time series for the given time series time.
     */
    public List<String> getMessagesForAllTimeSeries(final long timeSeriesTime)
    {
        final List<String> returnStringList = new ArrayList<String>();

        for(final String key: keySet())
        {
            for(final Long forecastTime: get(key).keySet())
            {
                final String comment = get(key).get(forecastTime).get(timeSeriesTime);
                if(comment != null)
                {
                    if(forecastTime == QuestionableTools.NO_TIME)
                    {
                        returnStringList.add(key + " (observed): " + comment);
                    }
                    else
                    {
                        returnStringList.add(key + " (forecast T0 "
                            + HCalendar.buildDateStr(forecastTime, HCalendar.DEFAULT_DATE_FORMAT) + "): " + comment);
                    }
                }
            }
        }
        return returnStringList;
    }

    /**
     * loadQuestionableFile() reads a questionableFile and loads a triple HashMap of its entries. The first HashMap key
     * is the parameterId. The first HashMap value is a second HashMap. The second HashMap key is the forecast time in
     * milliseconds. The second HashMap value is a third HashMap. The third HashMap key is the time series time in
     * milliseconds. The third HashMap value is questionable string.
     * 
     * @param questionableFile - questionableFile name
     * @return - nothing
     * @throws Exception
     */
    public void loadQuestionableFile(final File questionableFile) throws Exception
    {
        final List<XMLString> questionableList = Lists.newArrayList();

        try
        {
            // Pass in a factory to tell CollectionXMLReader how to construct an element within the collection. 

            final CollectionXMLReader<XMLString> reader = new CollectionXMLReader<XMLString>(QuestionableTools.QUESTIONABLE_DATA_LOG,
                                                                                             questionableList,
                                                                                             new XMLReaderFactory<XMLString>()
                                                                                             {
                                                                                                 @Override
                                                                                                 public XMLString get()
                                                                                                 {
                                                                                                     final XMLString questionable = new XMLString(QuestionableTools.QUESTIONABLE_DATA);
                                                                                                     questionable.addAttribute(QuestionableTools.PARAMETER_ID,
                                                                                                                               XMLString.class,
                                                                                                                               true);
                                                                                                     questionable.addAttribute(QuestionableTools.FORECAST_TIME,
                                                                                                                               new XMLLong(QuestionableTools.NO_TIME), // default
                                                                                                                               true);
                                                                                                     questionable.addAttribute(QuestionableTools.TS_TIME,
                                                                                                                               XMLLong.class,
                                                                                                                               true);
                                                                                                     return questionable;
                                                                                                 }
                                                                                             });
            reader.addAttribute(QuestionableTools.LOCATION_ID, XMLString.class, true);

            XMLTools.readXMLFromFile(questionableFile, reader);
        }
        catch(final Exception e)
        {
            throw new Exception("Unable to read " + questionableFile.getAbsolutePath() + ": " + e.getMessage());
        }

        // Construct the triple HashMap

        String parameterId;
        Long forecastTime;
        Long tsTime;
        String message;

        for(final XMLString questionable: questionableList)
        {
            parameterId = (String)questionable.getAttributeValue(QuestionableTools.PARAMETER_ID);
            forecastTime = (Long)questionable.getAttributeValue(QuestionableTools.FORECAST_TIME);
            tsTime = (Long)questionable.getAttributeValue(QuestionableTools.TS_TIME);
            message = questionable.get();

            addQuestionableMessage(parameterId, forecastTime, tsTime, message);
        }
    }
}
