package ohd.hseb.hefs.mefp.tools;

import java.io.File;
import java.text.DecimalFormat;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;

import nl.wldelft.util.timeseries.TimeSeriesArray;
import ohd.hseb.hefs.mefp.sources.rfcfcst.PreparedRFCFileTools;
import ohd.hseb.hefs.mefp.sources.rfcfcst.RFCForecastDataHandler;
import ohd.hseb.hefs.pe.tools.HEFSTools;
import ohd.hseb.hefs.pe.tools.LocationAndDataTypeIdentifier;
import ohd.hseb.hefs.utils.tools.FileTools;
import ohd.hseb.hefs.utils.tools.ParameterId;
import ohd.hseb.hefs.utils.tsarrays.TimeSeriesArrayTools;
import ohd.hseb.hefs.utils.xml.CollectionXMLWriter;
import ohd.hseb.hefs.utils.xml.XMLReaderException;
import ohd.hseb.hefs.utils.xml.XMLTools;
import ohd.hseb.hefs.utils.xml.vars.XMLLong;
import ohd.hseb.hefs.utils.xml.vars.XMLString;
import ohd.hseb.util.misc.HCalendar;

import org.apache.logging.log4j.Logger;
import org.apache.logging.log4j.LogManager;

import com.google.common.collect.Lists;
import com.google.common.collect.Maps;

public abstract class QuestionableTools
{
    // Extremes are put here in case we want to read them from a site file.
    //
    // Don't know good rate of change (ROC) values for precip and temp.
    // On our test data, setting MAX_PRECIP_CHANGE = 90.0 MM and
    // MAX_TEMP_CHANGE = 60.0 degC causes no questionables to be generated.
    // Paul Tilles says there is a ROC_checker, but doesn't have values.

    public static final float MAX_PRECIP = (float)100.0; /* MM */
    public static final float MIN_PRECIP = (float)0.0; /* MM */
    public static final float MAX_PRECIP_CHANGE = (float)-1.0; /* MM */

    public static final float MAX_TEMP = (float)100.0; /* degC */
    public static final float MIN_TEMP = (float)-100.0; /* degC */
    public static final float MAX_TEMP_CHANGE = (float)-1.0; /* degC */

    public static final float MAX_TEMP_RANGE = (float)50.0; /* degC */
    public static final float MIN_TEMP_RANGE = (float)0.0; /* degC */

    public static final String QUESTIONABLE_DATA_LOG = "questionableDataLog";
    public static final String QUESTIONABLE_DATA = "questionableData";

    public static final Long NO_TIME = Long.MIN_VALUE; /* -9223372036854775808 */
    public static final String FORECAST_TIME = "forecastTime"; /* milliseconds */
    public static final String TS_TIME = "tsTime"; /* milliseconds */

    public static final String NO_DATA = "no data";
    public static final String IS_MISSING = "value is missing";
    public static final String RANGE_CHECK_FAILED = "range check failed";

    // LOCATION_ID is an attribute of the QUESTIONABLE_DATA_LOG, but PARAMETER_ID is an attribute 
    // of the QUESTIONABLE_DATA because a TMAX value could be questionable (missing), but the TMIN
    // value could be OK.

    public static final String LOCATION_ID = "locationId";
    public static final String PARAMETER_ID = "parameterId";

    public static final String PRCP_SUFFIX = ".PRCP";
    public static final String TEMP_SUFFIX = ".TEMP";
    public static final String QUESTIONABLE_SUFFIX = ".questionable.xml";

    /**
     * newQuestionable() makes a standard questionable XMLString
     * 
     * @param parameterId
     * @param forecastTime
     * @param time
     * @return
     */
    public static XMLString newQuestionable(final String parameterId, final Long forecastTime, final long time)
    {
        final XMLString questionable = new XMLString(QUESTIONABLE_DATA);

        questionable.addAttribute(TS_TIME, new XMLLong(), true);
        questionable.setAttributeValue(TS_TIME, time);

        questionable.addAttribute(PARAMETER_ID, new XMLString(), true);
        questionable.setAttributeValue(PARAMETER_ID, parameterId);

        questionable.addAttribute(FORECAST_TIME, new XMLLong(), true);
        if(HEFSTools.isForecastDataType(parameterId))
        {
            questionable.setAttributeValue(FORECAST_TIME, forecastTime);
        }
        else
        // questionable is observed, no FORECAST_TIME
        {
            questionable.setAttributeValue(FORECAST_TIME, NO_TIME);
        }

        return (questionable);
    }

    /**
     * newQuestionable() makes a standard questionable XMLString
     * 
     * @param tsa - the time series array being checked for questionable
     * @param time - the UNIX time being checked
     * @return
     */
    public static XMLString newQuestionable(final TimeSeriesArray tsa, final long time)
    {
        return newQuestionable(tsa.getHeader().getParameterId(), tsa.getHeader().getForecastTime(), time);
    }

    /**
     * Builds a {@link List} of all the questionable messages for a {@link Collection} of {@link TimeSeriesArray}s.
     * 
     * @param tsas - collection of time series arrays to check. For efficiency, it assumes the collection has been
     *            trimmed before calling.
     * @return - an XMLString list of the questionable values
     * @throws XMLReaderException
     */
    public static List<XMLString> buildQuestionables(final Collection<TimeSeriesArray> tsas) throws XMLReaderException
    {
        float value;
        float previousValue;
        float aboveValue;
        float belowValue;
        float maxChange;
        String strValue;
        Long forecastTime;
        Long time;
        String parameterId;
        long timeStep;
        String unit;
        XMLString questionable;
        float tempRange;
        final HashMap<Long, HashMap<Long, Float>> minFcstTimeToTSTimeToValueMap = Maps.newHashMap();
        final HashMap<Long, HashMap<Long, Float>> maxFcstTimeToTSTimeToValueMap = Maps.newHashMap();
        final HashMap<Long, TimeSeriesArray> minFcstTimeToTS = Maps.newHashMap();
        final List<XMLString> questionables = Lists.newArrayList();
        final DecimalFormat df = new DecimalFormat("0.######");

        // Loop over each tsa in the collection

        for(final TimeSeriesArray tsa: tsas)
        {
            timeStep = tsa.getHeader().getTimeStep().getStepMillis();
            unit = tsa.getHeader().getUnit();
            parameterId = tsa.getHeader().getParameterId();

            if(ParameterId.valueOf(parameterId).isPrecipitation())
            {
                aboveValue = MAX_PRECIP;
                belowValue = MIN_PRECIP;
                maxChange = MAX_PRECIP_CHANGE;
            }
            else
            // temp
            {
                aboveValue = MAX_TEMP;
                belowValue = MIN_TEMP;
                maxChange = MAX_TEMP_CHANGE;
            }

            if(HEFSTools.isForecastDataType(parameterId))
            {
                forecastTime = tsa.getHeader().getForecastTime();
            }
            else
            {
                forecastTime = NO_TIME;
            }

            // For time series, the start time should always correspond to index 0. 
            // Since they are regular time series, the next index is one step after the current index.

            int indexToCheck = -1;
            for(time = tsa.getStartTime(); time <= tsa.getEndTime(); time += timeStep)
            {
                indexToCheck++;

                // Missing check

                if(TimeSeriesArrayTools.isMissing(tsa, indexToCheck))
                {
                    questionable = newQuestionable(tsa, time);
                    questionable.setValueOfElement(QUESTIONABLE_DATA, IS_MISSING);
                    questionables.add(questionable);
                    continue;
                }

                // Gross range check

                value = tsa.getValue(indexToCheck);

                if(value < belowValue)
                {
                    questionable = newQuestionable(tsa, time);
                    strValue = RANGE_CHECK_FAILED + " " + df.format(value) + " < " + df.format(belowValue) + " " + unit;
                    questionable.setValueOfElement(QUESTIONABLE_DATA, strValue);
                    questionables.add(questionable);
                    continue;
                }
                else if(value > aboveValue)
                {
                    questionable = newQuestionable(tsa, time);
                    strValue = RANGE_CHECK_FAILED + " " + df.format(value) + " > " + df.format(aboveValue) + " " + unit;
                    questionable.setValueOfElement(QUESTIONABLE_DATA, strValue);
                    questionables.add(questionable);
                    continue;
                }

                // Rate of change check, currently ignored

                if(maxChange > 0 && indexToCheck > 0)
                {
                    if(!TimeSeriesArrayTools.isMissing(tsa, indexToCheck - 1))
                    {
                        previousValue = tsa.getValue(indexToCheck - 1);
                        if(Math.abs(value - previousValue) > maxChange)
                        {
                            questionable = newQuestionable(tsa, time);
                            strValue = "change from " + df.format(previousValue) + " to " + df.format(value) + " > "
                                + df.format(maxChange) + " " + unit;
                            questionable.setValueOfElement(QUESTIONABLE_DATA, strValue);
                            questionables.add(questionable);
                            continue;
                        }
                    }
                }

                // If you got here, the value is acceptable. If it's a temp, add it the 
                // HashMaps for tmin/tmax checking after all the time series have been acceptable checked.

                if(ParameterId.valueOf(parameterId).isTemperature())
                {
                    if(ParameterId.valueOf(parameterId).isMin())
                    {
                        HashMap<Long, Float> minHash = minFcstTimeToTSTimeToValueMap.get(forecastTime);

                        if(minHash == null) // haven't created a Hash for this forecastTime
                        {
                            minHash = new HashMap<Long, Float>();
                            minFcstTimeToTSTimeToValueMap.put(forecastTime, minHash);
                            minFcstTimeToTS.put(forecastTime, tsa);
                        }
                        minFcstTimeToTSTimeToValueMap.get(forecastTime).put(time, value);
                    }
                    else if(ParameterId.valueOf(parameterId).isMax())
                    {
                        HashMap<Long, Float> maxHash = maxFcstTimeToTSTimeToValueMap.get(forecastTime);

                        if(maxHash == null) // haven't created a Hash for this forecastTime
                        {
                            maxHash = new HashMap<Long, Float>();
                            maxFcstTimeToTSTimeToValueMap.put(forecastTime, maxHash);
                        }
                        maxFcstTimeToTSTimeToValueMap.get(forecastTime).put(time, value);
                    }
                }
            }
        }

        // tmin/tmax checking. Loop over tmin and report questionables as tmin.

        for(final Long fcstTime: minFcstTimeToTSTimeToValueMap.keySet())
        {
            final HashMap<Long, Float> minCheck = minFcstTimeToTSTimeToValueMap.get(fcstTime);
            final HashMap<Long, Float> maxCheck = maxFcstTimeToTSTimeToValueMap.get(fcstTime);

            if(maxCheck == null) // no valid tmax
            {
                continue;
            }

            for(final Long tsTime: minCheck.keySet()) // iterate over tmin keys
            {
                final Float tmin = minCheck.get(tsTime);
                final Float tmax = maxCheck.get(tsTime);

                if(tmax != null) // there is a matching tmax for this tmin
                {
                    tempRange = tmax - tmin;

                    if(tempRange < MIN_TEMP_RANGE)
                    {
                        questionable = newQuestionable(minFcstTimeToTS.get(fcstTime).getHeader().getParameterId(),
                                                       fcstTime,
                                                       tsTime);
                        strValue = "tmin " + df.format(tmin) + " > tmax " + df.format(tmax) + " "
                            + minFcstTimeToTS.get(fcstTime).getHeader().getUnit();
                        questionable.setValueOfElement(QUESTIONABLE_DATA, strValue);
                        questionables.add(questionable);
                    }
                    else if(tempRange > MAX_TEMP_RANGE)
                    {
                        questionable = newQuestionable(minFcstTimeToTS.get(fcstTime).getHeader().getParameterId(),
                                                       fcstTime,
                                                       tsTime);
                        strValue = "tmax " + df.format(tmax) + " - tmin " + df.format(tmin) + " = "
                            + df.format(tempRange) + " > " + MAX_TEMP_RANGE + " "
                            + minFcstTimeToTS.get(fcstTime).getHeader().getUnit();
                        questionable.setValueOfElement(QUESTIONABLE_DATA, strValue);
                        questionables.add(questionable);
                    }
                }
            }
        }

        return questionables;
    }

    /**
     * toLog - writes a questionable string to a log
     * 
     * @param LOG - the log
     * @param the questionableString
     */
    public static void toLog(final Logger LOG, final XMLString questionable)
    {
        if(LOG != null)
        {
            final String logStr = "** QUESTIONABLE DATA ** " + questionable.getAttributeValue(PARAMETER_ID) + " "
                + HCalendar.buildDateTimeTZStr((Long)questionable.getAttributeValue(TS_TIME)) + " "
                + questionable.get();

            LOG.info(logStr);
        }
    }

    /**
     * listQuestionable() - checks a collection of time series to see if any values are questionable. It lists the
     * questionable values and returns the list in XML format. The calling routine can write the list to a file. It
     * accepts a collection so tmax/tmin can be compared in parallel.
     * 
     * @param tsas - collection of time series
     * @param LOG - if not null, the questionable values will also be logged.
     * @return - list of questionable data
     */
    public static List<XMLString> listQuestionable(final Collection<TimeSeriesArray> tsas, final Logger LOG)
    {
        final List<XMLString> questionableList = Lists.newArrayList();

        XMLString questionable = null;
        List<XMLString> questionables = Lists.newArrayList();
        final Collection<TimeSeriesArray> trimmed = new ArrayList<TimeSeriesArray>();
        TimeSeriesArray trim;

        // Trim each time series

        for(final TimeSeriesArray tsa: tsas)
        {
            trim = TimeSeriesArrayTools.trimMissingValuesFromBeginningAndEndOfTimeSeries(tsa);

            if(trim.isEmpty())
            {
                try
                {
                    questionable = newQuestionable(tsa, tsa.getStartTime());
                    questionable.setValueOfElement(QUESTIONABLE_DATA, NO_DATA);
                    questionables.add(questionable);
                }
                catch(final XMLReaderException e)
                {
                    e.printStackTrace();
                }
            }
            else
            {
                trimmed.add(trim);
            }
        }

        questionableList.addAll(questionables);

        // Find the questionables and add them to the list.

        if(!trimmed.isEmpty())
        {
            try
            {
                questionables = buildQuestionables(trimmed);
            }
            catch(final Exception e)
            {
                e.printStackTrace();
            }

            questionableList.addAll(questionables);

        }

        // Log the questionables. We may want to comment this as it floods the Logs panel and adds time

        for(final XMLString q: questionableList)
        {
            toLog(LOG, q);
        }

        return questionableList;
    }

    /**
     * idToFile() - make a questionableFile from a directory and an identifier
     * 
     * @param directoryName - directory where the questionableFile is to be written
     * @param identifier - used to build the questionableFile name
     * @return - the questionableFile
     */
    public static File getFile(final String directoryName, final LocationAndDataTypeIdentifier identifier)
    {
        String fileStr = directoryName + File.separator + identifier.getLocationId();

        if(identifier.isPrecipitationDataType())
        {
            fileStr += PRCP_SUFFIX;
        }
        else
        {
            fileStr += TEMP_SUFFIX;
        }

        fileStr += QUESTIONABLE_SUFFIX;

        return new File(fileStr);
    }

    /**
     * deletes a questionableFile. tmin/tmax share a questionableFile suffixed by TEMP_SUFFIX
     * 
     * @param directoryName - directory where the questionableFile are written
     * @param identifier - used to build the questionableFile name
     * @throws Throwable
     */
    public static void deleteQuestionableFile(final String directoryName, final LocationAndDataTypeIdentifier identifier) throws Throwable
    {
        final File questionableFile = getFile(directoryName, identifier);
        FileTools.deleteFileIfItExists(questionableFile);
    }

    /**
     * deletes a list of questionableFiles.
     * 
     * @param directoryName - directory where the questionableFiles are written
     * @param identifiers - used to build the questionableFile name
     * @throws Throwable
     */
    public static void deleteQuestionableFiles(final String directoryName,
                                               final Collection<LocationAndDataTypeIdentifier> identifiers) throws Throwable
    {
        for(final LocationAndDataTypeIdentifier identifier: identifiers)
        {
            deleteQuestionableFile(directoryName, identifier);
        }
    }

    /**
     * create a questionableFile.
     * 
     * @param directoryName - directory where the questionableFile is to be written
     * @param identifier - used to build the questionableFile name
     * @param TSA - time series array on which to build the questionableFile
     * @param LOG - if not null, the questionable values will also be logged.
     * @throws Throwable
     */
    public static void createQuestionableFile(final String directoryName,
                                              final LocationAndDataTypeIdentifier identifier,
                                              final List<TimeSeriesArray> TSA,
                                              final Logger LOG) throws Throwable
    {
        final List<XMLString> questionableData = listQuestionable(TSA, LOG);

        if(!questionableData.isEmpty())
        {
            final File questionableFile = getFile(directoryName, identifier);

            appendFile(questionableFile, questionableData, identifier);
        }
    }

    /**
     * creates a questionableFile paired with a file.
     * 
     * @param file - file that may have a paired questionableFile
     * @param LOG - if not null, the questionable values will also be logged.
     * @throws Throwable
     */
    public static void createRFCQuestionableFile(final File file, final Logger LOG) throws Throwable
    {
        final String location = FileTools.getBaseName(file);
        final String extension = FileTools.getExtension(file);
        final String parameter = RFCForecastDataHandler.EXTENSION_PARM_ID.get(extension).toString();
        final LocationAndDataTypeIdentifier identifier = LocationAndDataTypeIdentifier.get(location, parameter);

        List<TimeSeriesArray> TSA = new ArrayList<TimeSeriesArray>();

        if(ParameterId.valueOf(parameter).isPrecipitation())
        {
            if(HEFSTools.isForecastDataType(parameter))
            {
                TSA = PreparedRFCFileTools.readPrecipitationForecastFile(file, identifier, parameter);
            }
            else
            {
                TSA.add(PreparedRFCFileTools.readPrecipitationObservationFile(file, identifier, parameter));
            }
        }
        else
        // temp
        {
            String maxExtension;
            String minExtension;
            String maxParameter;
            String minParameter;

            if(HEFSTools.isForecastDataType(parameter))
            {
                maxExtension = "rfctmxfcst";
                minExtension = "rfctmnfcst";
            }
            else
            {
                maxExtension = "rfctmxobs";
                minExtension = "rfctmnobs";
            }
            maxParameter = RFCForecastDataHandler.EXTENSION_PARM_ID.get(maxExtension).toString();
            minParameter = RFCForecastDataHandler.EXTENSION_PARM_ID.get(minExtension).toString();

            final File maxFile = FileTools.newFile(file.getParent() + File.separator + location + "." + maxExtension);
            final File minFile = FileTools.newFile(file.getParent() + File.separator + location + "." + minExtension);

            if((minFile.exists() && (maxFile.exists())))
            {
                //Max file.
                if(HEFSTools.isForecastDataType(parameter))
                {
                    TSA = PreparedRFCFileTools.readTemperatureForecastFile(maxFile, identifier, maxParameter);
                }
                else
                {
                    TSA.add(PreparedRFCFileTools.readTemperatureObservationFile(maxFile, identifier, maxParameter));
                }

                //Min file.
                if(HEFSTools.isForecastDataType(parameter))
                {
                    TSA.addAll(PreparedRFCFileTools.readTemperatureForecastFile(minFile, identifier, minParameter));
                }
                else
                {
                    TSA.add(PreparedRFCFileTools.readTemperatureObservationFile(minFile, identifier, minParameter));
                }

            }
            else
            {
                // Do not have the paired file for temperature, so we cannot create the questionable file.
                // A minFile or maxFile by itself will not be checked for acceptable values.
                return;
            }
        }

//            if(extension.equals(maxExtension)) // is a maxFile, check it now
//            {
//                parameter = EXTENSION_PARM_ID.get(maxExtension).toString();
//                identifier = LocationAndDataTypeIdentifier.get(location, parameter);
//
//                if(HEFSTools.isForecastDataType(parameter))
//                {
//                    TSA = PreparedRFCFileTools.readTemperatureForecastFile(maxFile, identifier, parameter);
//                }
//                else
//                {
//                    TSA.add(PreparedRFCFileTools.readTemperatureObservationFile(maxFile, identifier, parameter));
//                }
//
//                if(minFile.exists()) // also have a minFile, check it now, together with max
//                {
//                    if(HEFSTools.isForecastDataType(parameter))
//                    {
//                        TSA.addAll(PreparedRFCFileTools.readTemperatureForecastFile(minFile, identifier, parameter));
//                    }
//                    else
//                    {
//                        TSA.add(PreparedRFCFileTools.readTemperatureObservationFile(minFile, identifier, parameter));
//                    }
//                }
//            }
//            else
//            // have a minFile
//            {
//                if(maxFile.exists()) // do nothing, minFile will be checked with maxFile
//                {
//                    return;
//                }
//                else
//                // minFile without a maxFile
//                {
//                    parameter = EXTENSION_PARM_ID.get(minExtension).toString();
//                    identifier = LocationAndDataTypeIdentifier.get(location, parameter);
//
//                    if(HEFSTools.isForecastDataType(parameter))
//                    {
//                        TSA = PreparedRFCFileTools.readTemperatureForecastFile(minFile, identifier, parameter);
//                    }
//                    else
//                    {
//                        TSA.add(PreparedRFCFileTools.readTemperatureObservationFile(minFile, identifier, parameter));
//                    }
//                }
//            }
//        }

        createQuestionableFile(file.getParent(), identifier, TSA, LOG);
    }

    /**
     * appendFile() appends a list of questionable values in XML format to a questionableFile. The list of questionable
     * values was generated by listQuestionable().
     * 
     * @param questionableFile - questionableFile name
     * @param questionableList - list of questionable data in XML format
     * @param identifier - location and parameter file, for attributes
     * @throws Exception - if cannot append to the file
     */
    public static void appendFile(final File questionableFile,
                                  final List<XMLString> questionableList,
                                  final LocationAndDataTypeIdentifier identifier) throws Exception
    {
        // Append to the file

        if(questionableList.isEmpty()) // no questionable data, do nothing
        {
            return;
        }

        try
        {
            final CollectionXMLWriter writer = new CollectionXMLWriter(QUESTIONABLE_DATA_LOG, questionableList);

            writer.addAttribute(new XMLString(LOCATION_ID, identifier.getLocationId()), true);

            XMLTools.writeXMLFileFromXMLWriter(questionableFile, writer, true); // true adds newline
        }
        catch(final Exception e)
        {
            throw new Exception("Unable to write " + questionableFile.getAbsolutePath() + ": " + e.getMessage());
        }
    }

    /**
     * isQuestionable() checks to see if the data is questionable. The data is questionable if a questionableFile
     * exists.
     * 
     * @param directoryName - directory where the questionableFile is
     * @param identifier - used to build the questionableFile name
     * @return - true if the questionableFile was found, false if not
     */
    public static boolean isQuestionable(final String directoryName, final LocationAndDataTypeIdentifier identifier)
    {
        final File questionableFile = getFile(directoryName, identifier);

        return (questionableFile.exists());
    }

    /**
     * isQuestionable() checks to see if a collection of identifiers is questionable. It returns true if one identifier
     * in the collection is questionable, otherwise false.
     * 
     * @param directoryName - directory where the questionableFiles are
     * @param identifiers - used to build the questionableFile name
     * @return - true if one of the identifiers has a questionableFile, false if not
     */
    public static boolean isQuestionable(final String directoryName,
                                         final Collection<LocationAndDataTypeIdentifier> identifiers)
    {
        for(final LocationAndDataTypeIdentifier identifier: identifiers)
        {
            if(isQuestionable(directoryName, identifier))
            {
                return true;
            }
        }

        return (false);
    }
}
