package ohd.hseb.hefs.mefp.tools.canonical;

import java.io.File;
import java.util.ArrayList;
import java.util.List;

import ohd.hseb.hefs.pe.tools.LocationAndDataTypeIdentifier;

import org.apache.logging.log4j.Logger;
import org.apache.logging.log4j.LogManager;

/**
 * For this handler, it is assumed that all ASCII (non-XML) canonical events files are located under a directory called
 * control within the base directory.
 * 
 * @author hank.herr
 */
public class CanonicalEventFileHandler
{
    private static final Logger LOG = LogManager.getLogger(CanonicalEventFileHandler.class);
    private File _baseDirectory;
    private File _parameterFileControlDirectory;

    public CanonicalEventFileHandler()
    {

    }

    public CanonicalEventFileHandler(final File baseDirectory)
    {
        setBaseDirectory(baseDirectory);
    }

    public void setBaseDirectory(final File directory)
    {
        _baseDirectory = directory;
        _parameterFileControlDirectory = new File(_baseDirectory.getAbsolutePath() + "/control");
    }

    public File getBaseDirectory()
    {
        return _baseDirectory;
    }

    public File getParameterFileControlDirectory()
    {
        return _parameterFileControlDirectory;
    }

    protected File generateBaseCanonicalEventFile(final LocationAndDataTypeIdentifier identifier)
    {
        if(identifier.isPrecipitationDataType())
        {
            return generateBaseCanonicalEventFile(true);
        }
        else if(identifier.isTemperatureDataType())
        {
            return generateBaseCanonicalEventFile(false);
        }
        return null;
    }

    protected File generateModulationCanonicalEventFile(final LocationAndDataTypeIdentifier identifier)
    {
        if(identifier.isPrecipitationDataType())
        {
            return generateModulationCanonicalEventFile(true);
        }
        else if(identifier.isTemperatureDataType())
        {
            return generateModulationCanonicalEventFile(false);
        }
        return null;
    }

    private List<File> generateAllASCIICanonicalEventsFiles()
    {
        final List<File> files = new ArrayList<File>();
        files.add(this.generateBaseCanonicalEventFile(true));
        files.add(this.generateModulationCanonicalEventFile(true));
        files.add(this.generateBaseCanonicalEventFile(false));
        files.add(this.generateModulationCanonicalEventFile(false));
        return files;
    }

    public boolean doAnyASCIICanonicalEventsFilesExist()
    {
        final List<File> files = generateAllASCIICanonicalEventsFiles();
        for(final File file: files)
        {
            if(file.exists())
            {
                return true;
            }
        }
        return false;
    }

    private File generateBaseCanonicalEventFile(final boolean precipitation)
    {
        if(precipitation)
        {
            return new File(_parameterFileControlDirectory.getAbsolutePath() + "/base_events_precip_v2.txt");
        }
        else
        {
            return new File(_parameterFileControlDirectory.getAbsolutePath() + "/base_events_temp_cfsv2.txt");
        }
    }

    private File generateModulationCanonicalEventFile(final boolean precipitation)
    {
        if(precipitation)
        {
            return new File(_parameterFileControlDirectory.getAbsolutePath() + "/modulation_events_precip_v2.txt");
        }
        else
        {
            return new File(_parameterFileControlDirectory.getAbsolutePath() + "/modulation_events_temp_cfsv2.txt");
        }
    }

    public boolean baseEventFileExists(final LocationAndDataTypeIdentifier identifier)
    {
        final File file = this.generateBaseCanonicalEventFile(identifier);
        return file.exists();
    }

    public CanonicalEventList loadBaseEvents(final LocationAndDataTypeIdentifier identifier) throws Exception
    {
        final File file = this.generateBaseCanonicalEventFile(identifier);
        if(!file.exists())
        {
            throw new Exception("Base events file, " + file.getAbsolutePath() + ", was not found.");
        }
        return new CanonicalEventList(identifier.isPrecipitationDataType(), file);
    }

    public CanonicalEventList loadModulationEvents(final LocationAndDataTypeIdentifier identifier) throws Exception
    {
        final File file = this.generateModulationCanonicalEventFile(identifier);
        if(!file.exists())
        {
            throw new Exception("Modulation events file, " + file.getAbsolutePath() + ", was not found.");
        }
        return new CanonicalEventList(identifier.isPrecipitationDataType(), file);
    }

    /**
     * @param precipitation If true, load the events for precip. If false, temperature.
     * @return List of canonical events.
     * @throws Exception See {@link CanonicalEventList#CanonicalEventList(boolean, File)}.
     */
    public CanonicalEventList loadBaseEvents(final boolean precipitation) throws Exception
    {
        final File file = this.generateBaseCanonicalEventFile(precipitation);
        if(!file.exists())
        {
            LOG.warn("Base events file, " + file.getAbsolutePath()
                + ", was not found; corresponding events will not be changed.");
            return null;
        }
        return new CanonicalEventList(precipitation, file);
    }

    /**
     * @param precipitation If true, load the events for precip. If false, temperature.
     * @return List of canonical events. If no modulation events exist, list will be empty.
     * @exception See {@link CanonicalEventList#CanonicalEventList(boolean, File)}.
     */
    public CanonicalEventList loadModulationEvents(final boolean precipitation) throws Exception
    {
        final File file = this.generateModulationCanonicalEventFile(precipitation);
        if(!file.exists())
        {
            LOG.warn("Modulation events file, " + file.getAbsolutePath()
                + ", was not found; corresponding events will not be changed.");
            return null;
        }
        return new CanonicalEventList(precipitation, file);
    }

    /**
     * Removes all canonical events files that should exist.
     */
    public void removeAllASCIICanonicalEventsFiles()
    {
        final List<File> importFiles = this.generateAllASCIICanonicalEventsFiles();
        for(final File file: importFiles)
        {
            if(file.exists())
            {
                file.delete();
            }
        }
    }

    /**
     * @param identifier Identifier for which events are to be written. Determines if it is precip or temp data.
     * @param base True for base events, false for modulation events.
     * @param events The events to write.
     * @throws Exception
     */
    public void writeASCIIEventsFile(final LocationAndDataTypeIdentifier identifier,
                                     final boolean base,
                                     final CanonicalEventList events) throws Exception
    {
        if(identifier.isPrecipitationDataType())
        {
            writeASCIIEventsFile(true, base, events);
        }
        else if(identifier.isTemperatureDataType())
        {
            writeASCIIEventsFile(false, base, events);
        }
        else
        {
            throw new Exception("Invalid data type specified in identifier: " + identifier.toString());
        }
    }

    /**
     * @param precipitation True for precipitation, false for temperature.
     * @param base True for base events, false for modulation events.
     * @param events The events to write.
     * @throws Exception
     */
    public void writeASCIIEventsFile(final boolean precipitation, final boolean base, final CanonicalEventList events) throws Exception
    {
        File outFile;
        if(base)
        {
            outFile = this.generateBaseCanonicalEventFile(precipitation);
        }
        else
        {
            outFile = this.generateModulationCanonicalEventFile(precipitation);
        }
        events.writeToFile(outFile);
    }

}
